package com.kms.katalon.core.cucumber;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.stream.Stream;

import org.eclipse.e4.core.contexts.ContextInjectionFactory;
import org.eclipse.e4.core.contexts.EclipseContextFactory;

import com.kms.katalon.core.setting.BundleSettingStore;

import org.apache.commons.lang3.StringUtils;
import org.osgi.framework.FrameworkUtil;

public class CucumberPreferences extends BundleSettingStore implements ICucumberPreferences {

    private static final String BUNDLE_ID = "com.katalon.cucumber";

    private static final String GLUES_KEY = "gluePackages";

    private static final String VALIDATE_PROJECT_KEY = "validateProject";

    private static final String GLUES_SEPARATOR = ",";

    public static CucumberPreferences forProject(String projectDir) {
        Path path = Paths.get(projectDir);

        // Check if the file has a .prj extension
        if (path.toString().endsWith(".prj")) {
            path = path.getParent();
        }

        var preferences = new CucumberPreferences(path.toString());
        var bundleContext = FrameworkUtil.getBundle(CucumberPreferences.class).getBundleContext();
        var context = EclipseContextFactory.getServiceContext(bundleContext);
        ContextInjectionFactory.inject(preferences, context);

        return preferences;
    }

    public CucumberPreferences(String projectDir) {
        super(projectDir, BUNDLE_ID, false);
    }

    @Override
    public CucumberSettings loadSettings() throws IOException {
        var gluesString = getString(GLUES_KEY, StringUtils.EMPTY);
        var glues = Stream.of(StringUtils.split(gluesString, GLUES_SEPARATOR))
                .filter(StringUtils::isNotBlank)
                .toList();

        boolean validateProject = getBoolean(VALIDATE_PROJECT_KEY, false);

        var settings = new CucumberSettings();
        settings.setGluePackages(glues);
        settings.setValidateProject(validateProject);

        return settings;
    }

    @Override
    public void saveSettings(CucumberSettings settings) throws IOException {
        setProperty(GLUES_KEY, StringUtils.join(settings.getGluePackages(), GLUES_SEPARATOR));
        setProperty(VALIDATE_PROJECT_KEY, settings.isValidateProject());
    }
}
