package com.kms.katalon.core.keyword.internal

import com.kms.katalon.core.annotation.internal.Action
import com.kms.katalon.core.configuration.RunConfiguration
import com.kms.katalon.core.constants.CoreConstants
import com.kms.katalon.core.constants.StringConstants
import com.kms.katalon.core.exception.StepFailedException
import com.kms.katalon.core.keyword.IKeywordManager
import com.kms.katalon.core.util.internal.KeywordLoader
import groovy.transform.CompileStatic

import java.text.MessageFormat

@CompileStatic
public class KeywordExecutor {

    public static final String PLATFORM_WEB = CoreConstants.PLATFORM_WEB;
    public static final String PLATFORM_MOBILE = CoreConstants.PLATFORM_MOBILE;
    public static final String PLATFORM_WEB_SERVICE = CoreConstants.PLATFORM_WEB_SERVICE;
    public static final String PLATFORM_WINDOWS = CoreConstants.PLATFORM_WINDOWS;
    public static final String PLATFORM_BUILT_IN = CoreConstants.PLATFORM_BUILT_IN;

    public static final String WEB_BUILT_IN_KEYWORD_PACKAGE = "com.kms.katalon.core.webui.keyword.builtin";
    public static final String MOBILE_BUILT_IN_KEYWORD_PACKAGE = "com.kms.katalon.core.mobile.keyword.builtin";
    public static final String WEB_SERVICE_BUILT_IN_KEYWORD_PACKAGE = "com.kms.katalon.core.webservice.keyword.builtin";
    public static final String WINDOWS_BUILT_IN_KEYWORD_PACKAGE = "com.kms.katalon.core.windows.keyword.builtin";
    public static final String CORE_BUILT_IN_KEYWORD_PACKAGE = "com.kms.katalon.core.keyword.builtin";

    // ---------- instance fields ----------
    private final IKeywordLoader keywordLoader;
    private final IKeywordManager keywordManager;
    private final Map<String, Map<String, List<IKeyword>>> cachePlatforms = new HashMap<>();

    // ---------- multi-instance support ----------
    private static final Map<String, KeywordExecutor> EXECUTORS = new HashMap<>();

    private KeywordExecutor(String keywordPlatform) {
        this.keywordLoader = new KeywordLoader(keywordPlatform);
        this.keywordManager = new KeywordManager();
    }

    @CompileStatic
    public static KeywordExecutor getInstance(String keywordPlatform) {
        synchronized (EXECUTORS) {
            String key = keywordPlatform;
            KeywordExecutor executor = EXECUTORS.get(key);
            if (executor == null) {
                executor = new KeywordExecutor(keywordPlatform);
                EXECUTORS.put(key, executor);
            }

            return executor;
        }
    }

    // ---------- public API ----------
    public IKeywordManager getKeywordManager() {
        return keywordManager;
    }

    public void setCurrentKeywordTimeout(long timeoutInMillis) {
        keywordManager.setTimeout(timeoutInMillis);
    }

    public long getRemainingKeywordTimeoutInMillis() {
        return keywordManager.getRemainingTimeoutInMillis();
    }

    public Object executeKeywordForPlatform(String platform, String keyword, Object... params) {
        keywordManager.setCurrentKeywordContext(platform, keyword);

        IKeyword[] actions = getActions(platform, keyword, getSuitablePackage(platform));
        if (actions.length < 1) {
            throw new StepFailedException(
                    MessageFormat.format(
                            StringConstants.KEYWORD_X_DOES_NOT_EXIST_ON_PLATFORM_Y,
                            keyword, platform
                    )
            );
        }

        KeywordExecutionContext.saveRunningKeywordAndPlatform(platform, keyword);
        KeywordExecutionContext.markKeywordsUsage(platform);
        KeywordExecutionContext.addExecutedKeyword(getPlatformAlias(platform) + "." + keyword);

        int actionDelayInMillis = RunConfiguration.getActionDelayInMillisForKeyword(platform, keyword);
        if (actionDelayInMillis > 0) {
            try {
                Thread.sleep(actionDelayInMillis);
            } catch (Exception e) {
                // Ignore this
            }
        }

        return actions[0].execute(params);
    }

    // ---------- helpers ----------
    private String[] getSuitablePackage(String platform) {
        switch (platform) {
            case PLATFORM_WEB:
                return [WEB_BUILT_IN_KEYWORD_PACKAGE] as String[]
            case PLATFORM_MOBILE:
                return [
                    MOBILE_BUILT_IN_KEYWORD_PACKAGE
                ] as String[]
            case PLATFORM_WEB_SERVICE:
                return [
                    WEB_SERVICE_BUILT_IN_KEYWORD_PACKAGE
                ] as String[]
            case PLATFORM_WINDOWS:
                return [
                    WINDOWS_BUILT_IN_KEYWORD_PACKAGE
                ] as String[]
            case PLATFORM_BUILT_IN:
                return [CORE_BUILT_IN_KEYWORD_PACKAGE] as String[]
            default:
                return [] as String[];
        }
    }

    private IKeyword[] getActions(String platform, String keyword, String[] searchPackages) {
        if (cachePlatforms.containsKey(platform)) {
            Map cacheActions = (Map) cachePlatforms.get(platform);
            if (cacheActions.containsKey(keyword)) {
                return cacheActions.get(keyword) as IKeyword[]
            }
        }

        // not found from cache
        if (searchPackages == null || searchPackages.length == 0) {
            // avoid to return null
            return [] as IKeyword[]
        }

        List<IKeyword> actions = new ArrayList<>()
        List<Class<?>> classes = keywordLoader.loadAllClass(searchPackages)
        for (Class<?> cls : classes) {
            if (!IKeyword.class.isAssignableFrom(cls)) {
                continue
            }

            Action act = (Action) cls.getAnnotation(Action.class)
            if (act == null || !act.value().equalsIgnoreCase(keyword)) {
                continue
            }

            try {
                actions.add((IKeyword) cls.newInstance())
            } catch (Exception ex) {
                throw new StepFailedException(MessageFormat.format(StringConstants.KEYWORD_EXECUTOR_ERROR_MSG, ex.getMessage()))
            }
        }

        Map cacheActions = cachePlatforms.containsKey(platform) ? (Map) cachePlatforms.get(platform) : new HashMap()
        cacheActions.put(keyword, actions)
        cachePlatforms.put(platform, cacheActions)
        return actions as IKeyword[]
    }

    @CompileStatic
    private static String getPlatformAlias(String platform) {
        switch (platform) {
            case PLATFORM_WEB:
                return "WebUI"
            case PLATFORM_MOBILE:
                return "Mobile"
            case PLATFORM_WINDOWS:
                return "Windows"
            case PLATFORM_WEB_SERVICE:
                return "WS"
            case PLATFORM_BUILT_IN:
                return "BuiltIn"
            default:
                return platform // fallback to original platform name
        }
    }
}
