package com.kms.katalon.core.main;

import java.io.File;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.codehaus.groovy.control.CompilationFailedException;
import org.codehaus.groovy.runtime.InvokerHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.kms.katalon.constants.GlobalStringConstants;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.context.internal.ExecutionEventManager;
import com.kms.katalon.core.context.internal.ExecutionListenerEvent;
import com.kms.katalon.core.context.internal.InternalTestCaseContext;
import com.kms.katalon.core.context.internal.InternalTestSuiteContext;
import com.kms.katalon.core.exception.KatalonRuntimeException;
import com.kms.katalon.core.logging.ErrorCollector;
import com.kms.katalon.core.logging.logback.LogbackConfigurator;
import com.kms.katalon.core.model.FailureHandling;
import com.kms.katalon.core.testcase.TestCaseBinding;
import com.kms.katalon.logging.LogConfigurator;
import com.kms.katalon.util.CryptoUtil;

import groovy.lang.Binding;
import groovy.lang.GroovyClassLoader;
import groovy.util.Node;
import groovy.util.NodeList;
import groovy.xml.XmlParser;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;

public class TestCaseMain {

    private static Logger logger = LoggerFactory.getLogger(TestCaseMain.class);

    private static final int DELAY_TIME = 50;

    private static ScriptEngine engine;

    private static ExecutionEventManager eventManager;

    private static Map<String, Object> secureVariables;

    /**
     * Setup test case or test suite before executing.
     *
     * CustomKeywords now has many custom keyword static methods, each one is
     * named with format [packageName].[className].[keywordName] but Groovy compiler
     * itself cannot invoke that formatted name. Therefore, we must change the
     * meta class of CustomKeywords to another one.
     * 
     * @throws IOException
     */
    public static void beforeStart() throws IOException {
        var defaultLogConfigPath = RunConfiguration.getLogbackConfigFileLocation();
        var defaultLogConfigFile = StringUtils.isNotEmpty(defaultLogConfigPath)
                ? new File(defaultLogConfigPath)
                : LogConfigurator.getConfigFile(LogConfigurator.EXECUTION_CONFIG_FILE);
        LogConfigurator.initialize(defaultLogConfigFile, findLogPropertyFile());

        GroovyClassLoader classLoader = new GroovyClassLoader(TestCaseMain.class.getClassLoader());
        engine = ScriptEngine.getDefault(classLoader);

        loadInternalGlobalVariableClass(classLoader);
        loadCustomKeywordsClass(classLoader);
        modifyCustomKeywordsClassAtRunTime(classLoader);

        eventManager = ExecutionEventManager.getInstance();
    }

    private static void loadCustomKeywordsClass(GroovyClassLoader cl) {
        // Load CustomKeywords class
        Class<?> clazz = cl.parseClass("class CustomKeywords { }");
        InvokerHelper.metaRegistry.setMetaClass(clazz, new CustomKeywordDelegatingMetaClass(clazz, cl));
    }

    private static void loadInternalGlobalVariableClass(GroovyClassLoader cl) {
        try {
            cl.loadClass(StringConstants.INTERNAL_GLOBAL_VARIABLE_CLASS_NAME);
        } catch (ClassNotFoundException ex) {
            try {
                cl.parseClass(
                        new File(RunConfiguration.getProjectDir(), StringConstants.INTERNAL_GLOBAL_VARIABLE_FILE_NAME));
            } catch (CompilationFailedException | IOException ignored) {}
        }
    }

    private static void modifyCustomKeywordsClassAtRunTime(GroovyClassLoader cl) {
        try {
            boolean isModified = false;
            ClassPool pool = ClassPool.getDefault();
            CtClass newCustomKeywordsClass = pool.get(StringConstants.CUSTOM_KEYWORD_CLASS_NAME);
            if (newCustomKeywordsClass != null) {
                newCustomKeywordsClass.defrost();
                CtMethod[] methods = newCustomKeywordsClass.getDeclaredMethods();
                for (CtMethod method : methods) {
                    // Only remove methods with name like "com.Example.HelloWord()"...
                    if (method.getName().contains(".")) {
                        newCustomKeywordsClass.removeMethod(method);
                        isModified = true;
                    }
                }
                if (isModified) {
                    String path = RunConfiguration.getProjectDir() + File.separator
                            + GlobalStringConstants.SYSTEM_FOLDER_NAME_BIN + File.separator + "lib";
                    newCustomKeywordsClass.writeFile(path);
                }
                Class<?> clazz = Class.forName(StringConstants.CUSTOM_KEYWORD_CLASS_NAME);
                InvokerHelper.metaRegistry.setMetaClass(clazz, new CustomKeywordDelegatingMetaClass(clazz, cl));
            }
        } catch (Exception e) {
            // ignored
        }
    }

    public static TestResult runTestCase(String testCaseId, TestCaseBinding testCaseBinding,
            FailureHandling flowControl) throws InterruptedException {
        return runTestCase(testCaseId, testCaseBinding, flowControl, true, true);
    }

    public static TestResult runTestCase(String testCaseId, TestCaseBinding testCaseBinding,
            FailureHandling flowControl, boolean doCleanUp) throws InterruptedException {
        return runTestCase(testCaseId, testCaseBinding, flowControl, true, doCleanUp);
    }

    public static TestResult runTestCase(String testCaseId, TestCaseBinding testCaseBinding,
            FailureHandling flowControl, boolean isMain, boolean doCleanUp) throws InterruptedException {
        Thread.sleep(DELAY_TIME);
        var bindedValue = testCaseBinding.getBindedValues() == null ? ""
                : testCaseBinding.getBindedValues().getOrDefault(testCaseBinding.getIterationVariableName(), "");
        var getIterationVariableValue = Objects.toString(bindedValue);
        InternalTestCaseContext testCaseContext = new InternalTestCaseContext(testCaseId, getIterationVariableValue);
        testCaseContext.setMainTestCase(isMain);
        return new TestCaseExecutor(testCaseBinding, engine, eventManager, testCaseContext, doCleanUp)
                .execute(flowControl);
    }

    public static TestResult runWSVerificationScript(String verificationScript, FailureHandling flowControl,
            boolean doCleanUp) throws InterruptedException {

        if (StringUtils.isBlank(verificationScript)) {
            return TestResult.getDefault();
        }

        Thread.sleep(DELAY_TIME);
        return new WSVerificationExecutor(verificationScript, engine, eventManager, doCleanUp).execute(flowControl);
    }

    public static TestResult runWSVerificationScript(TestCaseBinding testCaseBinding, String verificationScript,
            FailureHandling flowControl, boolean doCleanUp) throws InterruptedException {

        if (StringUtils.isBlank(verificationScript)) {
            return TestResult.getDefault();
        }

        Thread.sleep(DELAY_TIME);
        return new WSVerificationExecutor(testCaseBinding, verificationScript, engine, eventManager, doCleanUp)
                .execute(flowControl);
    }

    public static TestResult runTestCaseRawScript(String testScript, String testCaseId, TestCaseBinding testCaseBinding,
            FailureHandling flowControl) throws InterruptedException {
        Thread.sleep(DELAY_TIME);
        return new RawTestScriptExecutor(testScript, testCaseBinding, engine, eventManager,
                new InternalTestCaseContext(testCaseId)).execute(flowControl);
    }

    public static TestResult runFeatureFile(String featureFile) throws InterruptedException {
        Thread.sleep(DELAY_TIME);
        String verificationScript = MessageFormat
                .format("import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW\n" +

                        "CucumberKW.runFeatureFile(''{0}'')", featureFile);
        return new WSVerificationExecutor(null, verificationScript, engine, eventManager, true, true)
                .execute(FailureHandling.STOP_ON_FAILURE);
    }

    public static TestResult runTestCaseRawScript(String testScript, String testCaseId, TestCaseBinding testCaseBinding,
            FailureHandling flowControl, boolean doCleanUp) throws InterruptedException {
        Thread.sleep(DELAY_TIME);
        return new RawTestScriptExecutor(testScript, testCaseBinding, engine, eventManager,
                new InternalTestCaseContext(testCaseId, testCaseBinding.getIterationVariableName()), doCleanUp)
                        .execute(flowControl);
    }

    public static void startTestSuite(String testSuiteId, Map<String, String> suiteProperties,
            File testCaseBindingFile) {
        TestSuiteExecutor testSuiteExecutor = new TestSuiteExecutor(testSuiteId, engine, eventManager);
        testSuiteExecutor.execute(suiteProperties, testCaseBindingFile);
        
        // Need to exit explicitly to avoid hanging thread (ex: video recorder thread) preventing JVM from shutting down   
        // Sample of hanging thread:
        //        ALERT: Dangling Thread [pool-2-thread-1] is keeping JVM alive.
        //            at java.desktop/sun.lwawt.macosx.LWCToolkit.flushNativeSelectors(Native Method)
        //            at java.desktop/sun.lwawt.macosx.LWCToolkit.sync(LWCToolkit.java:519)
        //            at java.desktop/java.awt.Robot.createCompatibleImage(Robot.java:564)
        //            at java.desktop/java.awt.Robot.createScreenCapture(Robot.java:477)
        //            at com.kms.katalon.core.helper.screenrecorder.ATUVideoRecorder.grabScreen(ATUVideoRecorder.java:277)
        //            at com.kms.katalon.core.helper.screenrecorder.ATUVideoRecorder$1.run(ATUVideoRecorder.java:102)
        //            at java.base/java.util.concurrent.Executors$RunnableAdapter.call(Executors.java:572)
        //            at java.base/java.util.concurrent.FutureTask.runAndReset(FutureTask.java:358)
        //            at java.base/java.util.concurrent.ScheduledThreadPoolExecutor$ScheduledFutureTask.run(ScheduledThreadPoolExecutor.java:305)
        //            at java.base/java.util.concurrent.ThreadPoolExecutor.runWorker(ThreadPoolExecutor.java:1144)
        //            at java.base/java.util.concurrent.ThreadPoolExecutor$Worker.run(ThreadPoolExecutor.java:642)
        //            at java.base/java.lang.Thread.run(Thread.java:1583)
        
        System.exit(0);
    }

    public static void startTestCaseBinding(String testCaseId, File testCaseBindingFile) {
        TestCaseBindingExecutor testCaseBindingExecutor = new TestCaseBindingExecutor(testCaseId, engine, eventManager);
        testCaseBindingExecutor.execute(testCaseBindingFile);
    }

    public static void runTestCaseBindingRawScript(String testScript, String testCaseId, File testCaseBindingFile,
            FailureHandling flowControl, boolean doCleanUp) throws InterruptedException {
        Thread.sleep(DELAY_TIME);
        new RawTestBindingScriptExecutor(testScript, testCaseId, engine, eventManager, doCleanUp, flowControl)
                .execute(testCaseBindingFile);
    }

    public static void invokeStartSuite(String testSuiteId) {
        InternalTestSuiteContext testSuiteContext = new InternalTestSuiteContext();
        testSuiteContext.setTestSuiteId(testSuiteId);
        eventManager.publicEvent(ExecutionListenerEvent.BEFORE_TEST_SUITE, new Object[] { testSuiteContext });
    }

    public static void invokeEndSuite(String testSuiteId) {
        InternalTestSuiteContext testSuiteContext = new InternalTestSuiteContext();
        testSuiteContext.setTestSuiteId(testSuiteId);
        eventManager.publicEvent(ExecutionListenerEvent.AFTER_TEST_SUITE, new Object[] { testSuiteContext });
    }

    public static Map<String, Object> getProtectedGlobalVariables() {
        if (secureVariables != null) {
            return secureVariables;
        }

        secureVariables = new HashMap<String, Object>();
        String encryptedGlobalVariablesString = System.getenv().get(StringConstants.PROTECTED_GLOBAL_VARIABLES);
        if (StringUtils.isNotBlank(encryptedGlobalVariablesString)) {
            try {
                Map<String, Object> protectedGlobalVariables = convertStringToMap(encryptedGlobalVariablesString, true);
                for (Map.Entry<String, Object> entry : protectedGlobalVariables.entrySet()) {
                    secureVariables.put(entry.getKey(), entry.getValue());
                }
            } catch (GeneralSecurityException | IOException e) {
                logGlobalVariableError(e);
            }
        }

        String encryptedOverridingGlobalVariables = System.getenv().get(StringConstants.OVERRIDING_GLOBAL_VARIABLES);
        if (StringUtils.isNotBlank(encryptedOverridingGlobalVariables)) {
            try {
                Map<String, Object> overridingGlobalVariables = convertStringToMap(encryptedOverridingGlobalVariables,
                        true);
                List<String> warningOverridingVariables = new ArrayList<String>();
                for (Map.Entry<String, Object> entry : overridingGlobalVariables.entrySet()) {
                    String variableName = entry.getKey();
                    if (secureVariables.containsKey(variableName)) {
                        secureVariables.put(variableName, entry.getValue());
                        warningOverridingVariables.add(variableName);
                    }
                }

                if (warningOverridingVariables.size() > 0) {
                    logger.warn(MessageFormat.format("Using g_ for protected variables {0}, continuing as protected",
                            String.join(", ", warningOverridingVariables)));
                }
            } catch (GeneralSecurityException | IOException e) {
                logGlobalVariableError(e);
            }
        }

        String encryptedOverridingProtectedGlobalVariables = System.getenv()
                .get(StringConstants.OVERRIDING_PROTECTED_GLOBAL_VARIABLES);
        if (StringUtils.isNotBlank(encryptedOverridingProtectedGlobalVariables)) {
            try {
                Map<String, Object> overridingProtectedGlobalVariables = convertStringToMap(
                        encryptedOverridingProtectedGlobalVariables, true);
                List<String> warningOverridingVariables = new ArrayList<String>();
                for (Map.Entry<String, Object> entry : overridingProtectedGlobalVariables.entrySet()) {
                    String variableName = entry.getKey();
                    if (!secureVariables.containsKey(variableName)) {
                        warningOverridingVariables.add(variableName);
                    }

                    secureVariables.put(variableName, entry.getValue());
                }

                if (warningOverridingVariables.size() > 0) {
                    logger.warn(MessageFormat.format("Using p_ for unprotected variables {0}, continuing as protected",
                            String.join(", ", warningOverridingVariables)));
                }
            } catch (GeneralSecurityException | IOException e) {
                logGlobalVariableError(e);
            }
        }

        return secureVariables;
    }

    public static Map<String, Object> getGlobalVariables(String profileName) {
        Map<String, Object> protectedGlobalVariables = getProtectedGlobalVariables();
        Map<String, Object> overridingGlobalVariables = new HashMap<String, Object>();
        String encryptedOverridingGlobalVariables = System.getenv().get(StringConstants.OVERRIDING_GLOBAL_VARIABLES);
        if (StringUtils.isNotBlank(encryptedOverridingGlobalVariables)) {
            try {
                overridingGlobalVariables = convertStringToMap(encryptedOverridingGlobalVariables, true);
            } catch (Exception e) {
                logGlobalVariableError(e);
            }
        }

        try {
            Map<String, Object> selectedVariables = new HashMap<>();
            Node rootNode = new XmlParser()
                    .parse(new File(RunConfiguration.getProjectDir(), "Profiles/" + profileName + ".glbl"));
            NodeList variableNodes = (NodeList) rootNode.get("GlobalVariableEntity");
            for (int index = 0; index < variableNodes.size(); index++) {
                Node globalVariableNode = (Node) variableNodes.get(index);
                String variableName = ((Node) ((NodeList) globalVariableNode.get("name")).get(0)).text();
                Object defaultValue = ((Node) ((NodeList) globalVariableNode.get("initValue")).get(0)).text();
                try {
                    defaultValue = engine.runScriptWithoutLogging((String) defaultValue, new Binding());
                    selectedVariables.put(variableName, defaultValue);
                } catch (Exception e) {
                    logGlobalVariableError(e);
                }

                Object value = null;
                if (protectedGlobalVariables.containsKey(variableName)) {
                    value = protectedGlobalVariables.get(variableName);
                } else if (overridingGlobalVariables.containsKey(variableName)) {
                    value = overridingGlobalVariables.get(variableName);
                }

                if (value != null) {
                    String trimmed = String.valueOf(value).trim();
                    if (defaultValue instanceof Number) {
                        trimmed = trimmed.replaceAll("^'|'$", "");
                    } else if (defaultValue instanceof Boolean) {
                        trimmed = trimmed.equalsIgnoreCase("true") || trimmed.equalsIgnoreCase("false")
                                ? trimmed.toLowerCase()
                                : "'" + trimmed + "'";
                    } else {
                        trimmed = trimmed.matches("^'.*'$") ? trimmed : "'" + trimmed + "'";
                    }

                    try {
                        value = engine.runScriptWithoutLogging(trimmed, new Binding());
                        selectedVariables.put(variableName, value);
                    } catch (Exception e) {
                        logGlobalVariableError(e);
                    }
                }
            }

            return selectedVariables;
        } catch (Exception ex) {
            logGlobalVariableError(ex);
            return Collections.emptyMap();
        }
    }

    public static void logGlobalVariableError(Exception e) {
        KatalonRuntimeException runtimeException = new KatalonRuntimeException(
                String.format("There was something wrong in GlobalVariable. Details: %s", e.getMessage()));
        ErrorCollector.getCollector().addError(runtimeException);
    }

    public static ScriptEngine getScriptEngine() {
        return engine;
    }

    @SuppressWarnings("unchecked")
    private static Map<String, Object> convertStringToMap(String string, boolean encrypted)
            throws GeneralSecurityException, IOException {
        String mapString = string;
        if (encrypted) {
            CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(mapString);
            mapString = CryptoUtil.decode(cryptoInfo);
        }

        var result = new ObjectMapper().readValue(mapString, Map.class);
        if (result == null) {
            return Collections.emptyMap();
        }

        return result;
    }

    private static File findLogPropertyFile() {
        var configFolderPath = RunConfiguration.getProjectDir() + File.separator + LogConfigurator.CONFIG_FOLDER;
        var configFolder = new File(configFolderPath);
        if (!configFolder.exists()) {
            return null;
        }

        return FileUtils.listFiles(configFolder, new String[] { "properties" }, true)
                .stream()
                .filter(f -> LogConfigurator.CONFIG_PROPERTY_FILE.equals(f.getName()))
                .findFirst()
                .orElse(null);
    }
}
