package com.kms.katalon.core.reporting;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Type;
import java.net.InetAddress;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.stream.XMLStreamException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.eclipse.persistence.jaxb.JAXBContextFactory;

import com.google.gson.reflect.TypeToken;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.logging.XMLLoggerParser;
import com.kms.katalon.core.logging.XMLParserException;
import com.kms.katalon.core.logging.XmlLogRecord;
import com.kms.katalon.core.logging.model.ILogRecord;
import com.kms.katalon.core.logging.model.MessageLogRecord;
import com.kms.katalon.core.logging.model.TestCaseLogRecord;
import com.kms.katalon.core.logging.model.TestStatus;
import com.kms.katalon.core.logging.model.TestStatus.TestStatusValue;
import com.kms.katalon.core.logging.model.TestSuiteCollectionLogRecord;
import com.kms.katalon.core.logging.model.TestSuiteLogRecord;
import com.kms.katalon.core.reporting.ReportWriterUtil.SuiteReportGenerationOptions;
import com.kms.katalon.core.testcase.VariableReport;
import com.kms.katalon.core.testdata.reader.CsvWriter;
import com.kms.katalon.core.util.internal.JsonUtil;

public class ReportUtil {
    public static final String JUNIT_REPORT_FILE_NAME = "JUnit_Report.xml";

    public static String getOs() {
        return System.getProperty("os.name") + " " + System.getProperty("sun.arch.data.model") + "bit";
    }

    public static String getHostName() {
        String hostName = "Unknown";
        try {
            InetAddress addr;
            addr = InetAddress.getLocalHost();
            hostName = addr.getCanonicalHostName();
        } catch (UnknownHostException ex) {}
        return hostName;
    }

    private static void collectInfoLines(ILogRecord logRecord, List<ILogRecord> rmvLogs) {
        if (logRecord instanceof MessageLogRecord) {
            if (logRecord.getStatus().getStatusValue() == TestStatusValue.INCOMPLETE
                    || logRecord.getStatus().getStatusValue() == TestStatusValue.INFO) {
                rmvLogs.add(logRecord);
            }
        }
        for (ILogRecord childLogRecord : logRecord.getChildRecords()) {
            collectInfoLines(childLogRecord, rmvLogs);
        }
    }

    public static void writeLogRecordToCSVFile(TestSuiteLogRecord suiteLogEntity, File destFile,
            List<ILogRecord> filteredTestCases) throws IOException {
        writeLogRecordToCSVFile(suiteLogEntity, destFile, filteredTestCases, true);
    }

    public static void writeLogRecordToCSVFile(TestSuiteLogRecord suiteLogEntity, File destFile,
            List<ILogRecord> filteredTestCases, boolean stepsIncluded) throws IOException {
        CsvWriter.writeCsvReport(suiteLogEntity, destFile, filteredTestCases, stepsIncluded);
    }

    public static void writeLogRecordToJUnitFile(String logFolder) throws Exception {
        TestSuiteLogRecord testSuiteLogRecord = ReportWriterUtil.parseCollectionSuiteLog(logFolder);
        if (testSuiteLogRecord != null) {
            writeJUnitReport(testSuiteLogRecord, new File(logFolder));
        }
    }

    public static JUnitTestSuite generateJUnitTestSuite(TestSuiteLogRecord suiteLogEntity) {
        JUnitReportObjectFactory factory = new JUnitReportObjectFactory();

        String testSuiteName = suiteLogEntity.getName();
        String totalTests = suiteLogEntity.getTotalTestCases() + "";
        String totalError = suiteLogEntity.getTotalErrorTestCases() + "";
        String totalFailure = suiteLogEntity.getTotalFailedTestCases() + "";
        String totalSkipped = suiteLogEntity.getTotalSkippedTestCases() + "";

        TestCaseLogRecord lastTestCaseLogRecord = suiteLogEntity.getLastTestCaseLogRecord();
        String duration = ((float) ((lastTestCaseLogRecord != null ? lastTestCaseLogRecord.getEndTime()
                : suiteLogEntity.getEndTime()) - suiteLogEntity.getStartTime()) / 1000) + "";

        JUnitProperties properties = factory.createProperties();
        List<JUnitProperty> propertyList = properties.getProperty();
        propertyList.add(new JUnitProperty("deviceName", suiteLogEntity.getDeviceName()));
        propertyList.add(new JUnitProperty("devicePlatform", suiteLogEntity.getDevicePlatform()));
        propertyList.add(new JUnitProperty("logFolder", StringEscapeUtils.escapeJava(suiteLogEntity.getLogFolder())));
        propertyList.add(new JUnitProperty("logFiles", factory.sanitizeReportLogs(suiteLogEntity)));
        propertyList.add(new JUnitProperty("attachments", factory.sanitizeReportAttachments(suiteLogEntity)));
        suiteLogEntity.getRunData().forEach((name, value) -> propertyList.add(new JUnitProperty(name, value)));

        JUnitTestSuite ts = factory.createTestSuite();
        ts.setProperties(properties);
        ts.setId(suiteLogEntity.getId());
        ts.setName(testSuiteName);
        ts.setHostname(suiteLogEntity.getHostName());
        ts.setTime(duration);

        ZonedDateTime zdt = Instant.ofEpochMilli(suiteLogEntity.getStartTime()).atZone(ZoneId.systemDefault());
        ts.setTimestamp(DateTimeFormatter.ISO_INSTANT.format(zdt));

        ts.setSystemOut(suiteLogEntity.getSystemOutMsg().trim());
        ts.setSystemErr(suiteLogEntity.getSystemErrorMsg().trim());

        // tests: The total number of tests in the suite, required
        ts.setTests(totalTests);
        // errors: The total number of tests in the suite that error
        ts.setErrors(totalError);
        // failures: The total number of tests in the suite that failed
        ts.setFailures(totalFailure);
        // skipped: The total number of tests in the suite that is skipped by users
        ts.setSkipped(totalSkipped);

        StringBuilder tsSystemOut = new StringBuilder();

        for (ILogRecord log : suiteLogEntity.getBeforeTestSuiteLogRecords()) {
            String msg = log.getSystemOutMsg();
            if (StringUtils.isNotEmpty(msg)) {
                tsSystemOut.append(msg.trim()).append("\n\n");
            }
        }

        tsSystemOut.append(suiteLogEntity.getSystemOutMsg().trim());

        for (ILogRecord log : suiteLogEntity.getAfterTestSuiteLogRecords()) {
            String msg = log.getSystemOutMsg();
            if (StringUtils.isNotEmpty(msg)) {
                tsSystemOut.append("\n\n").append(msg.trim());
            }
        }

        ts.setSystemOut(tsSystemOut.toString());

        Arrays.asList(suiteLogEntity.filterFinalTestCasesResult()).stream().forEach(item -> {
            JUnitTestCase tc = factory.createTestCase();
            String time = ((float) (item.getEndTime() - item.getStartTime()) / 1000) + "";

            if (item instanceof TestCaseLogRecord) {
                TestCaseLogRecord testCaseLogRecord = (TestCaseLogRecord) item;
                JUnitProperties testCaseProperties = factory.createProperties();
                List<JUnitProperty> testCasePropertyList = testCaseProperties.getProperty();
                String rawDataBinding = getTestCaseLogRecordProperty(testCaseLogRecord,
                        StringConstants.EXECUTION_BINDING_VARIABLES);
                Map<String, Object> dataBinding = parseDataBindingAsMap(rawDataBinding);
                if (dataBinding != null && !dataBinding.isEmpty()) {
                    testCasePropertyList
                            .add(new JUnitProperty("data_binding", null, JsonUtil.toJson(dataBinding, false)));
                    tc.setProperties(testCaseProperties);
                }
            }

            tc.setClassname(item.getId());
            tc.setName(item.getName());
            tc.setTime(time);

            TestStatus status = item.getStatus();
            TestStatusValue statusValue = status.getStatusValue();
            String statusName = statusValue.name();
            String message = StringUtils.removeStart(item.getMessage(),
                    item.getName() + " " + statusName + " because (of) ");
            tc.setStatus(statusName);
            if (TestStatusValue.ERROR == statusValue) {
                JUnitError error = factory.createError();
                error.setType(statusName);
                error.setMessage(message);
                tc.getError().add(error);
            }
            if (TestStatusValue.FAILED == statusValue) {
                JUnitFailure failure = factory.createFailure();
                failure.setType(statusName);
                failure.setMessage(message);
                tc.getFailure().add(failure);
            }
            if (TestStatusValue.SKIPPED == statusValue) {
                JUnitSkipped skipped = factory.createSkipped();
                message = StringUtils.isNotEmpty(message) ? message : "Test was skipped";
                skipped.setMessage(message);
                tc.getSkipped().add(skipped);
            }

            tc.getSystemOut().add(item.getSystemOutMsg().trim());
            tc.getSystemErr().add(item.getSystemErrorMsg().trim());
            ts.getTestcase().add(tc);
        });

        return ts;
    }

    public static void writeJUnitReport(TestSuiteCollectionLogRecord suiteCollectionLogRecord, File logFolder)
            throws JAXBException, IOException {
        JUnitReportObjectFactory factory = new JUnitReportObjectFactory();
        List<JUnitTestSuite> tsList = new ArrayList<>();

        for (TestSuiteLogRecord suiteLogEntity : suiteCollectionLogRecord.getTestSuiteRecords()) {
            JUnitTestSuite ts = generateJUnitTestSuite(suiteLogEntity);
            tsList.add(ts);
        }

        String testSuiteCollectionName = suiteCollectionLogRecord.getName();
        String testSuiteCollectionTotalTests = suiteCollectionLogRecord.getTotalTestCases();
        String testSuiteCollectiontotalError = suiteCollectionLogRecord.getTotalErrorTestCases();
        String testSuiteCollectionTotalFailure = suiteCollectionLogRecord.getTotalFailedTestCases();
        String testSuiteCollectionDuration = ((float) (suiteCollectionLogRecord.getEndTime()
                - suiteCollectionLogRecord.getStartTime()) / 1000) + "";

        JUnitTestSuites tss = factory.createTestSuites();
        // errors: total number of tests with error result from all test suite
        tss.setErrors(testSuiteCollectiontotalError);
        // failures: total number of failed tests from all test suite
        tss.setFailures(testSuiteCollectionTotalFailure);
        // tests: total number of tests from all test suite
        tss.setTests(testSuiteCollectionTotalTests);
        // time: in seconds to execute all test suites
        tss.setTime(testSuiteCollectionDuration);
        // name
        tss.setName(testSuiteCollectionName);

        tss.getTestsuite().addAll(tsList);

        JAXBContext context = createJAXBContext();
        Marshaller marshaller = context.createMarshaller();
        marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
        marshaller.marshal(tss, new File(logFolder, JUNIT_REPORT_FILE_NAME));
    }

    public static void writeJUnitReport(TestSuiteLogRecord suiteLogEntity, File logFolder)
            throws JAXBException, IOException {

        JUnitReportObjectFactory factory = new JUnitReportObjectFactory();
        JUnitTestSuite ts = generateJUnitTestSuite(suiteLogEntity);

        // This is a single test suite. Thus, the info for the test suites is the same as test suite
        JUnitTestSuites tss = factory.createTestSuites();
        // errors: total number of tests with error result from all test suite
        tss.setErrors(ts.getErrors());
        // failures: total number of failed tests from all test suite
        tss.setFailures(ts.getFailures());
        // tests: total number of tests from all test suite
        tss.setTests(ts.getTests());
        // time: in seconds to execute all test suites
        tss.setTime(ts.getTime());
        // name
        tss.setName(ts.getName());

        tss.getTestsuite().add(ts);

        JAXBContext context = createJAXBContext();
        Marshaller marshaller = context.createMarshaller();
        marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
        marshaller.marshal(tss, new File(logFolder, JUNIT_REPORT_FILE_NAME));
    }

    public static String getTestCaseLogRecordProperty(TestCaseLogRecord testCaseLogRecord, String property) {
        if (testCaseLogRecord == null || StringUtils.isBlank(property)) {
            return null;
        }
        Map<String, String> testCaseLogRecordProperties = testCaseLogRecord.getProperties();
        if (testCaseLogRecordProperties == null) {
            return null;
        }
        return testCaseLogRecordProperties.get(property);
    }

    public static List<VariableReport> parseDataBinding(String rawDataBinding) {
        if (StringUtils.isBlank(rawDataBinding)) {
            return null;
        }
        Type listType = new TypeToken<List<VariableReport>>() {}.getType();
        return JsonUtil.fromJson(rawDataBinding, listType);
    }

    public static Map<String, Object> parseDataBindingAsMap(String rawDataBinding) {
        List<VariableReport> variables = parseDataBinding(rawDataBinding);
        Map<String, Object> dataBinding = new HashMap<String, Object>();
        if (variables == null) {
            return dataBinding;
        }
        variables.forEach(variable -> {
            dataBinding.put(variable.getDisplayReportName(), variable.isMasked() ? "********" : variable.getValue());
        });
        return dataBinding;
    }

    private static JAXBContext createJAXBContext() throws JAXBException {
        JAXBContext context = JAXBContextFactory.createContext(
                new Class[] { JUnitError.class, JUnitFailure.class, JUnitProperties.class, JUnitProperty.class,
                        JUnitTestCase.class, JUnitTestSuites.class, JUnitTestSuite.class, JUnitSkipped.class },
                new HashMap<>());
        return context;
    }

    public static void writeExecutionUUIDToFile(String UUID, File logFolder) throws IOException, URISyntaxException {
        FileUtils.writeStringToFile(new File(logFolder, "execution.uuid"), UUID, StringConstants.DF_CHARSET);
    }

    public static void writeCSVReport(TestSuiteLogRecord suiteLogEntity, File logFolder) throws IOException {
        CsvWriter.writeCsvReport(suiteLogEntity, new File(logFolder, logFolder.getName() + ".csv"),
                Arrays.asList(suiteLogEntity.getChildRecords()));
    }

    public static void writeLogRecordToHTMLFile(SuiteReportGenerationOptions options)
            throws IOException, URISyntaxException {
        var testSuiteLogRecord = options.getSuiteLogRecord();
        List<ILogRecord> infoLogs = new ArrayList<ILogRecord>();
        collectInfoLines(testSuiteLogRecord, infoLogs);
        for (ILogRecord infoLog : infoLogs) {
            infoLog.getParentLogRecord().removeChildRecord(infoLog);
        }
        ReportWriterUtil.writeHTMLReport(options);
    }

    public static List<XmlLogRecord> getAllLogRecords(String logFolder)
            throws XMLParserException, IOException, XMLStreamException {
        return XMLLoggerParser.readFromLogFolder(logFolder);
    }
}
