package com.kms.katalon.core.reporting.newreport.analyzer;

import java.util.Arrays;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kms.katalon.core.logging.model.ILogRecord;
import com.kms.katalon.core.logging.model.TestCaseLogRecord;
import com.kms.katalon.core.logging.model.TestStatus.TestStatusValue;
import com.kms.katalon.core.logging.model.TestSuiteLogRecord;
import com.kms.katalon.core.reporting.newreport.extractor.FailureReasonExtractors;
import com.kms.katalon.core.reporting.service.IFailureAnalysisService;
import com.kms.katalon.core.reporting.service.dto.FailureAnalysisResponse;
import com.kms.katalon.report.core.models.common.ExecutedTestResult;
import com.kms.katalon.report.core.models.entities.ExecutedTestEntity;
import com.kms.katalon.report.core.models.entities.FailureAnalysis;

public class FailureAnalyzer {
    private static final Logger logger = LoggerFactory.getLogger(FailureAnalyzer.class);

    private final FailureReasonExtractors failureReasonExtractors = FailureReasonExtractors.getDefault();

    private final boolean analyzeByAIEnabled;

    private final IFailureAnalysisService failureAnalysisService;

    private final FailedEntityBuilder failedEntityBuilder = new FailedEntityBuilder();

    public FailureAnalyzer(boolean analyzeByAIEnabled, IFailureAnalysisService failureAnalysisService) {
        this.analyzeByAIEnabled = analyzeByAIEnabled;
        this.failureAnalysisService = failureAnalysisService;
    }

    public void analyzeAndEnrich(ExecutedTestEntity entity, ILogRecord logRecord) {
        if (!isFailedEntity(entity)) {
            return;
        }

        if (isAiAnalysisEnabled()) {
            enrichWithAiAnalysis(entity, logRecord);
        } else {
            enrichWithBasicExtraction(entity, logRecord);
        }
    }

    private boolean isAiAnalysisEnabled() {
        return analyzeByAIEnabled && failureAnalysisService != null;
    }

    private boolean isFailedEntity(ExecutedTestEntity entity) {
        return entity != null
                && (entity.result == ExecutedTestResult.FAILED || entity.result == ExecutedTestResult.ERROR);
    }

    private void enrichWithAiAnalysis(ExecutedTestEntity entity, ILogRecord logRecord) {
        try {
            ExecutedTestEntity failureContext = failedEntityBuilder.buildFailedTestEntity(entity);
            FailureAnalysisResponse response = failureAnalysisService.executeAnalysis(failureContext);

            if (response != null) {
                applyAnalysisResponse(entity, response);
                return;
            }
        } catch (Exception e) {
            logger.warn("AI failure analysis failed for entity: " + entity.id, e);
        }

        enrichWithBasicExtraction(entity, logRecord);
    }

    private FailureAnalysis mapToFailureAnalysis(FailureAnalysisResponse response) {
        if (response == null) {
            return null;
        }
        var failureAnalysis = new FailureAnalysis();
        failureAnalysis.setImpact(response.getImpact());
        failureAnalysis.setRecommendation(response.getRecommendation());
        failureAnalysis.setConfidence(response.getConfidence());
        return failureAnalysis;
    }

    private void enrichWithBasicExtraction(ExecutedTestEntity entity, ILogRecord logRecord) {
        String failedReason = StringUtils.EMPTY;

        if (logRecord instanceof TestSuiteLogRecord suiteLogRecord) {
            ILogRecord[] childRecords = suiteLogRecord.getChildRecords();
            if (childRecords == null || childRecords.length == 0) {
                entity.failedReason = failedReason;
                return;
            }

            Optional<TestCaseLogRecord> firstFailedTestCase = Arrays.stream(childRecords)
                    .filter(TestCaseLogRecord.class::isInstance)
                    .map(TestCaseLogRecord.class::cast)
                    .filter(tc -> isFailedLogRecord(tc))
                    .findFirst();

            if (firstFailedTestCase.isPresent()) {
                String message = firstFailedTestCase.get().getMessage();
                if (StringUtils.isNotBlank(message)) {
                    failedReason = extractFailedReasonFromMessage(message);
                }
            }
        } else if (logRecord instanceof TestCaseLogRecord testCaseLogRecord) {
            String message = testCaseLogRecord.getMessage();
            if (StringUtils.isNotBlank(message)) {
                failedReason = extractFailedReasonFromMessage(message);
            }
        }

        entity.failedReason = failedReason;
    }

    private String extractFailedReasonFromMessage(String message) {
        return failureReasonExtractors.extract(message).orElse(StringUtils.EMPTY);
    }

    private void applyAnalysisResponse(ExecutedTestEntity entity, FailureAnalysisResponse response) {
        entity.failedReason = response.getFailedReason();
        entity.failureAnalysis = mapToFailureAnalysis(response);
    }

    private boolean isFailedLogRecord(ILogRecord logRecord) {
        return logRecord.getStatus().getStatusValue() == TestStatusValue.FAILED
                || logRecord.getStatus().getStatusValue() == TestStatusValue.ERROR;
    }

}
