package com.kms.katalon.core.testobject;

import static com.kms.katalon.core.constants.StringConstants.ID_SEPARATOR;

import java.io.File;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.enums.mobile.LocatorStrategy;
import com.kms.katalon.core.enums.mobile.MobilePlatform;
import com.kms.katalon.core.http.HttpUtils;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.logging.LogLevel;
import com.kms.katalon.core.main.ScriptEngine;
import com.kms.katalon.core.testobject.authorization.AwsSignatureAuthorization;
import com.kms.katalon.core.testobject.authorization.BasicRequestAuthorization;
import com.kms.katalon.core.testobject.authorization.DigestAuthorization;
import com.kms.katalon.core.testobject.authorization.NTLMAuthorization;
import com.kms.katalon.core.testobject.authorization.RequestAuthorization;
import com.kms.katalon.core.testobject.impl.HttpTextBodyContent;
import com.kms.katalon.core.testobject.internal.impl.HttpBodyContentReader;
import com.kms.katalon.core.testobject.internal.impl.WindowsObjectRepository;
import com.kms.katalon.core.util.ObjectUtil;
import com.kms.katalon.core.util.StrSubstitutor;
import com.kms.katalon.core.util.TestObjectDeserializer;
import com.kms.katalon.core.util.internal.Base64;
import com.kms.katalon.core.util.internal.ExceptionsUtil;
import com.kms.katalon.util.CryptoUtil;
import com.kms.katalon.util.SAXReaderProvider;
import com.kms.katalon.util.URLBuilder;
import com.kms.katalon.util.collections.NameValuePair;

import groovy.lang.Binding;
import groovy.util.ResourceException;
import groovy.util.ScriptException;

public class ObjectRepository {

    private static final KeywordLogger logger = KeywordLogger.getInstance(ObjectRepository.class);

    private static final String TEST_OBJECT_ROOT_FOLDER_NAME = "Object Repository";

    private static final String TEST_OBJECT_ID_PREFIX = TEST_OBJECT_ROOT_FOLDER_NAME + ID_SEPARATOR;

    private static final String WEB_SERVICES_TYPE_NAME = "WebServiceRequestEntity";

    private static final String WEB_ELEMENT_TYPE_NAME = "WebElementEntity";

    private static final String MOBILE_ELEMENT_TYPE_NAME = "MobileElementEntity";

    private static final String WEBELEMENT_FILE_EXTENSION = ".rs";

    private static final String WEB_ELEMENT_PROPERTY_NODE_NAME = "webElementProperties";

    private static final String WEB_ELEMENT_XPATH_NODE_NAME = "webElementXpaths";

    private static final String PROPERTY_NAME = "name";

    private static final String PROPERTY_CONDITION = "matchCondition";

    private static final String PROPERTY_VALUE = "value";

    private static final String PROPERTY_IS_SELECTED = "isSelected";

    private static final String[] PARENT_FRAME_ATTRS = new String[] { "ref_element", "parent_frame" };

    private static final String PARENT_SHADOW_ROOT_ATTRIBUTE = "ref_element_is_shadow_root";

    private static final String PROPERTY_SELECTOR_METHOD = "selectorMethod";

    private static final String PROPERTY_SELECTOR_COLLECTION = "selectorCollection";

    private static final String PROPERTY_LOCATOR_COLLECTION = "locatorCollection";

    private static final String PROPERTY_ELEMENT_PROPERTIES = "webElementProperties";

    private static final String SMART_PROPERTY_SELECTOR_COLLECTION = "smartLocatorCollection";

    private static final String PROPERTY_ENTRY = "entry";

    private static final String PROPERTY_KEY = "key";

    private static final String API_COLLECTION_META_EXTENSION = ".meta";

    private static final String AUTHORIZATION_HEADER = "Authorization";

    // Authorization type constants
    private static final String AUTH_TYPE_BASIC = "Basic";

    private static final String AUTH_TYPE_BEARER = "Bearer";

    private static final String AUTH_TYPE_OAUTH_1_0 = "OAuth 1.0";

    private static final String AUTH_TYPE_OAUTH_2_0 = "OAuth 2.0";

    private static final String AUTH_TYPE_NTLM = "NTLM";

    private static final String AUTH_TYPE_AWS_SIGNATURE = "AWS Signature";

    private static final String AUTH_TYPE_DIGEST = "Digest";

    private static Object globalVariable;

    private static ScriptEngine scriptEngine;

    private static Map<String, TestObject> recordedTestObjects;

    private static ScriptEngine getScriptEngine() {
        if (scriptEngine != null) {
            return scriptEngine;
        }

        try {
            scriptEngine = ScriptEngine.getDefault(ObjectRepository.class.getClassLoader());
        } catch (IOException e) {
            throw new RuntimeException("Cannot create script engine", e);
        }

        return scriptEngine;
    }

    /**
     * Get global variables.
     * 
     * @return a map of global variables.
     */
    public static Object getGlobalVariable() {
        if (globalVariable != null) {
            return globalVariable;
        }

        try {
            var scriptEngine = getScriptEngine();
            globalVariable = scriptEngine.runScriptWithoutLogging("internal.GlobalVariable", new Binding());
        } catch (ClassNotFoundException | ResourceException | ScriptException | IOException e) {
            throw new RuntimeException("Cannot evaluate global variable", e);
        }

        return globalVariable;
    }

    /**
     * Returns test object id of a its relative id.
     * 
     * @param testObjectRelativeId
     * Relative test object's id.
     * @returnString of test object id, <code>null</code> if <code>testObjectRelativeId</code> is null.
     */
    public static String getTestObjectId(final String testObjectRelativeId) {
        if (testObjectRelativeId == null) {
            return null;
        }

        if (testObjectRelativeId.startsWith(TEST_OBJECT_ID_PREFIX)) {
            return testObjectRelativeId;
        }
        return TEST_OBJECT_ID_PREFIX + testObjectRelativeId;
    }

    /**
     * Returns relative id of a test object's id. The relative id is cut <code>"Object Repository/"</code> prefix from
     * the
     * test object's id.
     * 
     * @param testObjectId
     * Full test object's id.
     * @return String of test object relative id, <code>null</code> if <code>testObjectId</code> is null.
     */
    public static String getTestObjectRelativeId(final String testObjectId) {
        if (testObjectId == null) {
            return null;
        }
        return testObjectId.replaceFirst(TEST_OBJECT_ID_PREFIX, StringUtils.EMPTY);
    }

    /**
     * Finds {@link TestObject} by its id or relative id
     * 
     * @param testObjectRelativeId
     * Can be test object full id or test object relative id
     * <p>
     * Eg: Using "Object Repository/Sample Test Object" (full id) OR "Sample Test Object" (relative id) as
     * <code>testObjectRelativeId</code> is accepted for the test object with id "Object Repository/Sample Test Object"
     * 
     * @return an instance of {@link TestObject} or <code>null</code> if the parameter is null or test object doesn't
     * exist
     * @see {@link #findTestObject(String, Map) findTestObject} for parameterizing test object
     */
    public static TestObject findTestObject(String testObjectRelativeId) {
        return findTestObject(testObjectRelativeId, new HashMap<String, Object>());
    }

    /**
     * Finds {@link TestObject} by its id or relative id using a variables map to parameterized its properties values
     * <p>
     * Object properties values are parameterized using the ${variable} syntax
     * <p>
     * For example: the test object has a xpath property with value ".//div[@class='class']//a[text()='${Variable}']"
     * <p>
     * If the test object is created using findTestObject(testObjectId, ['Variable': 'Text']) then the result xpath
     * property of the created test object will be ".//div[@class='class']//a[text()='Text']"
     * <p>
     * Use "$" to escape the "${" special characters, for example: "The variable $${${name}} must be used."
     * 
     * @param testObjectRelativeId
     * Can be test object full id or test object relative id
     * <p>
     * Eg: Using "Object Repository/Sample Test Object" (full id) OR "Sample Test Object" (relative id) as
     * <code>testObjectRelativeId</code> is accepted for the test object with id "Object Repository/Sample Test Object"
     * 
     * @param variables the variables map to parameterized the found test object
     * 
     * @return an instance of {@link TestObject} or <code>null</code> if test object id is null
     */
    public static TestObject findTestObject(String testObjectRelativeId, Map<String, Object> variables) {
        if (testObjectRelativeId == null) {
            logger.logWarning(StringConstants.TO_LOG_WARNING_TEST_OBJ_NULL);
            return null;
        }

        String testObjectId = getTestObjectId(testObjectRelativeId);
        logger.logDebug(MessageFormat.format(StringConstants.TO_LOG_INFO_FINDING_TEST_OBJ_W_ID, testObjectId));

        if (RunConfiguration.isPlayback()) {
            // Read test objects cached in temporary in record session.
            Map<String, TestObject> testObjectsCached = getCapturedTestObjects();

            if (testObjectRelativeId != null && testObjectsCached != null
                    && testObjectsCached.containsKey(testObjectRelativeId)) {
                return testObjectsCached.get(testObjectRelativeId);
            }
        }

        File objectFile = new File(RunConfiguration.getProjectDir(), testObjectId + WEBELEMENT_FILE_EXTENSION);
        if (!objectFile.exists()) {
            logger.logWarning(
                    MessageFormat.format(StringConstants.TO_LOG_WARNING_TEST_OBJ_DOES_NOT_EXIST, testObjectId));
            return null;
        }

        return readTestObjectFile(testObjectId, objectFile, RunConfiguration.getProjectDir(), variables);
    }

    public static WindowsTestObject findWindowsObject(final String windowsObjectRelativeId) {
        return findWindowsObject(windowsObjectRelativeId, Collections.emptyMap());
    }

    public static WindowsTestObject findWindowsObject(final String windowsObjectRelativeId,
            Map<String, Object> variables) {
        String windowsObjectId = getTestObjectId(windowsObjectRelativeId);
        File objectFile = new File(RunConfiguration.getProjectDir(), windowsObjectId + ".wrs");
        return WindowsObjectRepository.readWindowsTestObjectFile(windowsObjectId, objectFile,
                RunConfiguration.getProjectDir(), variables);
    }

    private static Map<String, TestObject> getCapturedTestObjects() {
        if (recordedTestObjects != null) {
            return recordedTestObjects;
        }

        try {
            String capturedObjectCacheFilePath = StringUtils
                    .defaultString(RunConfiguration.getCapturedObjectsCacheFile());
            if (!capturedObjectCacheFilePath.isEmpty()) {
                File capturedObjectCacheFile = new File(capturedObjectCacheFilePath);

                ObjectMapper mapper = new ObjectMapper();
                SimpleModule module = new SimpleModule();
                module.addDeserializer(TestObject.class, new TestObjectDeserializer());
                mapper.registerModule(module);

                Map<String, Map<String, Object>> rawObjects = mapper.readValue(
                        FileUtils.readFileToString(capturedObjectCacheFile, StringConstants.DF_CHARSET),
                        new TypeReference<Map<String, Map<String, Object>>>() {});

                recordedTestObjects = new LinkedHashMap<>();
                rawObjects.forEach((objectId, raw) -> {
                    recordedTestObjects.put(objectId, parseTestObjectsFromJson(objectId, raw));
                });
            }
        } catch (IOException ignored) {
            recordedTestObjects = Collections.emptyMap();
        }

        return recordedTestObjects;
    }

    public static TestObject readTestObjectFile(String testObjectId, File objectFile, String projectDir) {
        return readTestObjectFile(testObjectId, objectFile, projectDir, Collections.emptyMap());
    }

    public static TestObject readTestObjectFile(String testObjectId, File objectFile, String projectDir,
            Map<String, Object> variables) {
        try {
            Element rootElement = new SAXReader().read(objectFile).getRootElement();
            String elementName = rootElement.getName();
            if (WEB_ELEMENT_TYPE_NAME.equals(elementName)) {
                return findWebUIObject(testObjectId, rootElement, variables);
            }

            if (WEB_SERVICES_TYPE_NAME.equals(elementName)) {
                return findRequestObject(testObjectId, rootElement, projectDir, variables);
            }

            if (MOBILE_ELEMENT_TYPE_NAME.equals(elementName)) {
                return findMobileTestObject(testObjectId, rootElement, projectDir, variables);
            }

            return null;
        } catch (DocumentException e) {
            logger.logWarning(MessageFormat.format(StringConstants.TO_LOG_WARNING_CANNOT_GET_TEST_OBJECT_X_BECAUSE_OF_Y,
                    testObjectId, ExceptionsUtil.getMessageForThrowable(e)), null, e);
            return null;
        }
    }

    private static TestObject findWebUIObject(String testObjectId, Element element, Map<String, Object> variables) {
        TestObject testObject = new TestObject(testObjectId);

        // For image
        Element imagePathElement = element.element("imagePath");
        if (imagePathElement != null) {
            String imagePath = imagePathElement.getText();
            testObject.setImagePath(imagePath);
        }

        Element relativeImagePathElement = element.element("useRalativeImagePath");
        if (relativeImagePathElement != null) {
            String useRelavitePathString = relativeImagePathElement.getText();
            testObject.setUseRelativeImagePath(Boolean.parseBoolean(useRelavitePathString));
        }

        Element elementIsSmartLocatorEnabled = element.element("smartLocatorEnabled");
        boolean isSmartLocatorEnabled = false;
        if (elementIsSmartLocatorEnabled != null) {
            isSmartLocatorEnabled = Boolean.valueOf(elementIsSmartLocatorEnabled.getText());
        }

        Element dfSelectorMethodElement = element.element(PROPERTY_SELECTOR_METHOD);
        if (dfSelectorMethodElement != null) {
            if (isSmartLocatorEnabled) {
                testObject.setSelectorMethod(SelectorMethod.SMART_LOCATOR);
            } else {
                testObject.setSelectorMethod(SelectorMethod.valueOf(dfSelectorMethodElement.getText()));
            }
        }

        Map<String, Object> variablesStringMap = new HashMap<String, Object>();
        for (Entry<String, Object> entry : variables.entrySet()) {
            variablesStringMap.put(String.valueOf(entry.getKey()), entry.getValue());
        }

        var globalVariable = getGlobalVariable();
        if (globalVariable != null) {
            variablesStringMap.put("GlobalVariable", globalVariable);
        }

        StrSubstitutor strSubtitutor = new StrSubstitutor(variablesStringMap);

        Element propertySelectorCollection = element.element(PROPERTY_SELECTOR_COLLECTION);
        if (propertySelectorCollection != null) {
            List<?> selectorEntry = propertySelectorCollection.elements(PROPERTY_ENTRY);
            if (selectorEntry != null) {
                selectorEntry.forEach(entry -> {
                    Element selectorMethodElement = ((Element) entry);
                    SelectorMethod entryKey = SelectorMethod.valueOf(selectorMethodElement.elementText(PROPERTY_KEY));
                    String entryValue = strSubtitutor.replace(selectorMethodElement.elementText(PROPERTY_VALUE));
                    testObject.setSelectorValue(entryKey, entryValue);
                });
            }
        }

        Element smartSelectorCollection = element.element(SMART_PROPERTY_SELECTOR_COLLECTION);
        if (smartSelectorCollection != null) {
            List<?> selectorEntry = smartSelectorCollection.elements(PROPERTY_ENTRY);
            Map<SelectorMethod, String> map = new HashMap<>();
            if (selectorEntry != null) {
                selectorEntry.forEach(entry -> {
                    Element selectorMethodElement = ((Element) entry);
                    SelectorMethod entryKey = SelectorMethod.valueOf(selectorMethodElement.elementText(PROPERTY_KEY));
                    String entryValue = strSubtitutor.replace(selectorMethodElement.elementText(PROPERTY_VALUE));
                    map.put(entryKey, entryValue);
                });
                testObject.setSmartSelectorCollection(map);
            }
        }

        for (Object propertyElementObject : element.elements(WEB_ELEMENT_PROPERTY_NODE_NAME)) {
            TestObjectProperty objectProperty = new TestObjectProperty();
            Element propertyElement = (Element) propertyElementObject;

            String propertyName = StringEscapeUtils.unescapeXml(propertyElement.elementText(PROPERTY_NAME));
            ConditionType propertyCondition = ConditionType
                    .fromValue(StringEscapeUtils.unescapeXml(propertyElement.elementText(PROPERTY_CONDITION)));
            String propertyValue = StringEscapeUtils.unescapeXml(propertyElement.elementText(PROPERTY_VALUE));
            boolean isPropertySelected = Boolean
                    .valueOf(StringEscapeUtils.unescapeXml(propertyElement.elementText(PROPERTY_IS_SELECTED)));

            objectProperty.setName(propertyName);
            objectProperty.setCondition(propertyCondition);
            objectProperty.setValue(strSubtitutor.replace(propertyValue));
            objectProperty.setActive(isPropertySelected);

            // Check if this element is inside a frame
            if (Arrays.asList(PARENT_FRAME_ATTRS).contains(propertyName) && isPropertySelected) {
                TestObject parentObject = findTestObject(propertyValue);
                testObject.setParentObject(parentObject);
            } else if (PARENT_SHADOW_ROOT_ATTRIBUTE.equals(propertyName)) {
                testObject.setParentObjectShadowRoot(true);
            } else {
                testObject.addProperty(objectProperty);
            }
        }

        for (Object xpathElementObject : element.elements(WEB_ELEMENT_XPATH_NODE_NAME)) {
            TestObjectXpath objectXpath = new TestObjectXpath();
            Element xpathElement = (Element) xpathElementObject;

            String propertyName = StringEscapeUtils.unescapeXml(xpathElement.elementText(PROPERTY_NAME));
            ConditionType propertyCondition = ConditionType
                    .fromValue(StringEscapeUtils.unescapeXml(xpathElement.elementText(PROPERTY_CONDITION)));
            String propertyValue = StringEscapeUtils.unescapeXml(xpathElement.elementText(PROPERTY_VALUE));
            boolean isPropertySelected = Boolean
                    .valueOf(StringEscapeUtils.unescapeXml(xpathElement.elementText(PROPERTY_IS_SELECTED)));

            objectXpath.setName(propertyName);
            objectXpath.setCondition(propertyCondition);
            objectXpath.setValue(strSubtitutor.replace(propertyValue));
            objectXpath.setActive(isPropertySelected);

            // Check if this element is inside a frame
            if (Arrays.asList(PARENT_FRAME_ATTRS).contains(propertyName) && isPropertySelected) {
                TestObject parentObject = findTestObject(propertyValue);
                testObject.setParentObject(parentObject);
            } else if (PARENT_SHADOW_ROOT_ATTRIBUTE.equals(propertyName)) {
                testObject.setParentObjectShadowRoot(true);
            } else {
                testObject.addXpath(objectXpath);
            }
        }

        return testObject;
    }

    private static MobileTestObject findMobileTestObject(String mobileObjectId, Element element, String projectDir,
            Map<String, Object> variables) {
        MobileTestObject mobileTestObject = new MobileTestObject(mobileObjectId);
        Map<String, Object> variablesStringMap = new HashMap<String, Object>();
        for (Entry<String, Object> entry : variables.entrySet()) {
            variablesStringMap.put(String.valueOf(entry.getKey()), entry.getValue());
        }

        var globalVariable = getGlobalVariable();
        if (globalVariable != null) {
            variablesStringMap.put("GlobalVariable", globalVariable);
        }

        MobilePlatform platform = null;
        String propertyPlatform = element.elementText("platform");
        if (StringUtils.isNotEmpty(propertyPlatform)) {
            platform = MobilePlatform.valueOf(propertyPlatform);
        } else {
            String devicePlatform = RunConfiguration.getDevicePlatform();
            if (StringUtils.isNotEmpty(devicePlatform)) {
                if ("ANDROID_DRIVER".equals(devicePlatform)) {
                    platform = MobilePlatform.ANDROID;
                } else if ("IOS_DRIVER".equals(devicePlatform)) {
                    platform = MobilePlatform.IOS;
                }
            }
        }

        if (platform == null) {
            List<TestObjectProperty> properties = mobileTestObject.getProperties();
            boolean isAndroid = properties.stream().anyMatch(p -> p.getName().equals("class"));
            platform = isAndroid == true ? MobilePlatform.ANDROID : MobilePlatform.IOS;
        }

        mobileTestObject.setPlatform(platform);

        StrSubstitutor strSubstitutor = new StrSubstitutor(variablesStringMap);
        String locatorStrategyStr = element.elementText("locatorStrategy");
        LocatorStrategy locatorStrategy = LocatorStrategy.valueOf(locatorStrategyStr);
        mobileTestObject.setLocatorStrategy(locatorStrategy);

        List<Element> elementProperties = element.elements(PROPERTY_ELEMENT_PROPERTIES);
        List<TestObjectProperty> properties = elementProperties.stream().map(e -> {
            String isSelected = e.elementText("isSelected");
            String name = e.elementText("name");
            String value = e.elementText("value");
            String condition = e.elementText("matchCondition");

            return new TestObjectProperty(name, ConditionType.fromValue(condition), value,
                    Boolean.parseBoolean(isSelected));
        }).collect(Collectors.toList());
        mobileTestObject.setProperties(properties);

        Element propertyLocatorCollection = element.element(PROPERTY_LOCATOR_COLLECTION);
        if (propertyLocatorCollection != null) {
            List<?> locatorEntry = propertyLocatorCollection.elements(PROPERTY_ENTRY);
            if (locatorEntry != null) {
                locatorEntry.forEach(entry -> {
                    Element locatoStrategyElement = ((Element) entry);
                    String key = locatoStrategyElement.elementText(PROPERTY_KEY);
                    if (StringUtils.isNotEmpty(key)) {
                        LocatorStrategy entryKey = LocatorStrategy.valueOf(key);
                        String entryValue = strSubstitutor.replace(locatoStrategyElement.elementText(PROPERTY_VALUE));
                        mobileTestObject.setLocatorValue(entryKey, entryValue);
                    }
                });
            }
        }

        return mobileTestObject;
    }

    @SuppressWarnings("unlikely-arg-type")
    private static RequestObject findRequestObject(String requestObjectId, Element reqElement, String projectDir,
            Map<String, Object> variables) {
        RequestObject requestObject = new RequestObject(requestObjectId);
        requestObject.setName(reqElement.elementText("name"));

        String serviceType = reqElement.elementText("serviceType");
        requestObject.setServiceType(serviceType);

        Map<String, Object> defaultVariables = new HashMap<>();
        // Use default value of variables if available in case user passes nothing or null

        List<Element> variableElements = reqElement.elements("variables");
        if (variableElements != null && variableElements.size() > 0) {
            Map<String, String> rawVariables = new HashMap<>();
            for (Element variableElement : variableElements) {
                if (variableElement != null) {
                    Element defaultValue = variableElement.element("defaultValue");
                    Element name = variableElement.element("name");

                    if (!defaultValue.equals(StringUtils.EMPTY)) {
                        rawVariables.put(name.getData().toString(), defaultValue.getData().toString());
                    }
                }
            }
            boolean exception = false;
            try {
                defaultVariables = evaluateVariables(rawVariables);
            } catch (Exception e) {
                exception = true;
            } finally {
                if (exception == true) {
                    defaultVariables = new HashMap<>();
                }
            }
        }

        Map<String, Object> mergedVariables = new HashMap<>();
        mergedVariables.putAll(defaultVariables);

        if (variables != null && variables.size() > 0) {
            mergedVariables.putAll(variables);
        }

        var globalVariable = getGlobalVariable();
        if (globalVariable != null) {
            mergedVariables.put("GlobalVariable", globalVariable);
        }

        StrSubstitutor substitutor = new StrSubstitutor(mergedVariables);
        if ("SOAP".equals(serviceType)) {
            requestObject.setWsdlAddress(substitutor.replace(reqElement.elementText("wsdlAddress")));
            requestObject.setSoapRequestMethod(reqElement.elementText("soapRequestMethod"));
            requestObject.setSoapServiceFunction(reqElement.elementText("soapServiceFunction"));
            requestObject
                    .setHttpHeaderProperties(parseProperties(reqElement.elements("httpHeaderProperties"), substitutor));
            requestObject.setSoapBody(substitutor.replace(reqElement.elementText("soapBody")));
            String useServiceInfoFromWsdlValue = reqElement.elementText("useServiceInfoFromWsdl");
            if (StringUtils.isNotBlank(useServiceInfoFromWsdlValue)) {
                requestObject.setUseServiceInfoFromWsdl(
                        Boolean.valueOf(StringEscapeUtils.unescapeXml(useServiceInfoFromWsdlValue)));
            } else {
                requestObject.setUseServiceInfoFromWsdl(false);
            }

            requestObject.setSoapServiceEndpoint(substitutor.replace(reqElement.elementText("soapServiceEndpoint")));
        } else if ("RESTful".equals(serviceType)) {
            String rawUrl = reqElement.elementText("restUrl");
            String url = buildUrlFromRaw(rawUrl, substitutor);
            requestObject.setRestUrl(url);
            String requestMethod = reqElement.elementText("restRequestMethod");
            requestObject.setRestRequestMethod(requestMethod);
            requestObject.setRestParameters(parseProperties(reqElement.elements("restParameters")));
            requestObject
                    .setHttpHeaderProperties(parseProperties(reqElement.elements("httpHeaderProperties"), substitutor));
            // requestObject.setHttpBody(reqElement.elementText("httpBody"));

            String httpBodyType = reqElement.elementText("httpBodyType");
            String oldVersionBodyContent = reqElement.elementText("httpBody");
            if (StringUtils.isNotBlank(oldVersionBodyContent)) {
                // migrated from 5.3.1 (KAT-3200)
                httpBodyType = "text";
                String body = reqElement.elementText("httpBody");
                HttpTextBodyContent httpBodyContent = new HttpTextBodyContent(body);
                requestObject.setBodyContent(httpBodyContent);
            } else if (StringUtils.isNotBlank(httpBodyType)) {
                String httpBodyContent = reqElement.elementText("httpBodyContent");
                HttpBodyContent bodyContent = HttpBodyContentReader.fromSource(httpBodyType, httpBodyContent,
                        projectDir, substitutor);
                requestObject.setBodyContent(bodyContent);
            }
        }

        requestObject.setVariables(mergedVariables);

        RequestAuthorization requestAuthorization = parseAuthorization(reqElement.elements("authorizationRequest"));

        // If no explicit authorization in the request, check for inherited authorization from API Collection
        // Authorization inheritance is only for RESTful requests (API Collections), not SOAP
        if ("RESTful".equals(serviceType) && requestAuthorization == null && !hasAuthorizationHeaders(requestObject)) {
            BasicRequestAuthorization inheritedAuth = findInheritedAuthorizationFromPath(requestObjectId, projectDir);
            if (inheritedAuth != null) {
                applyInheritedAuthorization(requestObject, inheritedAuth);
                requestAuthorization = inheritedAuth;
            }
        }

        requestObject.setRequestAuthorization(requestAuthorization);

        String verificationScript = reqElement.elementText("verificationScript");
        requestObject.setVerificationScript(verificationScript);

        List<Element> rawValidationSteps = reqElement.elements("validationSteps");
        if (rawValidationSteps != null) {
            List<ValidationStep> validationSteps = rawValidationSteps.stream().map((rawStepI) -> {
                return parseValidationStep(rawStepI, substitutor);
            }).collect(Collectors.toList());
            requestObject.setValidationSteps(validationSteps);
        }

        boolean followRedirects = Boolean.valueOf(reqElement.elementText("followRedirects"));
        requestObject.setFollowRedirects(followRedirects);

        return requestObject;
    }

    private static ValidationStep parseValidationStep(Element rawStep, StrSubstitutor substitutor) {
        ValidationStep step = new ValidationStep();
        List<Element> rawProps = rawStep.elements();
        rawProps.forEach(propI -> {
            String propName = propI.getName();
            String propValue = propI.getStringValue();
            String parseValue = substitutor.replace(propValue);
            ObjectUtil.parseAndSet(step, propName, parseValue);
        });
        return step;
    }

    private static String buildUrlFromRaw(String rawUrl, StrSubstitutor substitutor) {
        URLBuilder urlBuilder = new URLBuilder(rawUrl);

        List<NameValuePair> rawQueryParams = urlBuilder.getQueryParams();

        List<NameValuePair> processedQueryParams = new ArrayList<>();

        rawQueryParams.stream().forEach(p -> {
            String variableExpandedName = substitutor.replace(p.getName());
            String variableExpandedValue = substitutor.replace(p.getValue());
            String escapedName = HttpUtils.escapePathSegment(variableExpandedName);
            String escapedValue = HttpUtils.escapePathSegment(variableExpandedValue);
            processedQueryParams.add(new NameValuePair(escapedName, escapedValue));
        });

        urlBuilder.setParameters(processedQueryParams);

        String url = urlBuilder.buildString();
        url = substitutor.replace(url); // replace the last time if there is still variable expansions in other parts of
                                        // the URL

        return url;
    }

    public static RequestObject findRequestObject(String requestObjectId, File objectFile) {
        try {
            Element reqElement = new SAXReader().read(objectFile).getRootElement();

            List<Element> variableElements = reqElement.elements("variables");
            Map<String, Object> variables = Collections.emptyMap();
            if (variableElements != null) {
                Map<String, String> rawVariables = parseRequestObjectVariables(variableElements);
                variables = evaluateVariables(rawVariables);
            }

            return findRequestObject(requestObjectId, reqElement, RunConfiguration.getProjectDir(), variables);
        } catch (Exception e) {
            logger.logWarning(MessageFormat.format(StringConstants.TO_LOG_WARNING_CANNOT_GET_TEST_OBJECT_X_BECAUSE_OF_Y,
                    requestObjectId, ExceptionsUtil.getMessageForThrowable(e)), null, e);
            return null;
        }
    }

    private static Map<String, String> parseRequestObjectVariables(List<Element> elements) {
        Map<String, String> variableMap = elements.stream()
                .collect(Collectors.toMap(element -> ((Element) element).elementText("name"),
                        element -> ((Element) element).elementText("defaultValue")));
        return variableMap;
    }

    private static Map<String, Object> evaluateVariables(Map<String, String> rawVariables)
            throws IOException, ClassNotFoundException, ResourceException, ScriptException {
        var scriptEngine = getScriptEngine();
        Map<String, Object> evaluatedVariables = new HashMap<>();
        for (Map.Entry<String, String> variableEntry : rawVariables.entrySet()) {
            String variableName = variableEntry.getKey();
            String variableValue = variableEntry.getValue();
            Object evaluatedValue = scriptEngine.runScriptWithoutLogging(variableValue, new Binding());
            evaluatedVariables.put(variableName, evaluatedValue);
        }
        return evaluatedVariables;
    }

    private static List<TestObjectProperty> parseProperties(List<Element> elements) {
        return parseProperties(elements, new StrSubstitutor());
    }

    private static List<TestObjectProperty> parseProperties(List<Element> elements, StrSubstitutor substitutor) {
        List<TestObjectProperty> props = new ArrayList<TestObjectProperty>();
        for (Element propertyElement : elements) {
            TestObjectProperty objectProperty = new TestObjectProperty();

            String propertyName = propertyElement.elementText(PROPERTY_NAME);
            ConditionType propertyCondition = ConditionType.fromValue(propertyElement.elementText(PROPERTY_CONDITION));
            String propertyValue = propertyElement.elementText(PROPERTY_VALUE);
            boolean isPropertySelected = Boolean.valueOf(propertyElement.elementText(PROPERTY_IS_SELECTED));

            objectProperty.setName(substitutor.replace(propertyName));
            objectProperty.setCondition(propertyCondition);
            objectProperty.setValue(substitutor.replace(propertyValue));
            objectProperty.setActive(isPropertySelected);

            props.add(objectProperty);
        }
        return props;
    }

    private static RequestAuthorization parseAuthorization(List<Element> requestElements) {
        RequestAuthorization retValue = null;
        Map<String, String> variables = new HashMap<String, String>();
        for (Element authorizationElement : requestElements) {
            String authorizationType = authorizationElement.elementText("authorizationType");
            if (authorizationElement.elements("authorizationInfo").size() > 0) {
                Element authorizationInfo = (Element) authorizationElement.elements("authorizationInfo").get(0);
                List<Element> elementObjects = authorizationInfo.elements("entry");
                for (Element element : elementObjects) {
                    variables.put(element.elementText("key"), element.elementText("value"));
                }

                if (authorizationType.equals(NTLMAuthorization.NTLM)) {
                    CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(variables.get("password"));
                    String password = StringUtils.EMPTY;
                    try {
                        password = CryptoUtil.decode(cryptoInfo);
                    } catch (GeneralSecurityException | IOException e) {
                        return null;
                    }

                    String username = variables.get("username");
                    String domain = variables.get("domain");
                    String workstation = variables.get("workstation");
                    retValue = new NTLMAuthorization(username, password, domain, workstation);
                } else if (authorizationType.equals(DigestAuthorization.AUTHORIZATION_TYPE)) {
                    try {
                        retValue = new DigestAuthorization(variables);
                    } catch (Exception e) {
                        logger.logMessage(LogLevel.WARNING, String.format(
                                "Fails to constructing the DigestAuthorization from the text. The Authorization of the test object will be null. Text: %s",
                                authorizationInfo.getText()), e);
                    }
                } else if (authorizationType.equals(AwsSignatureAuthorization.AUTHORIZATION_TYPE_NAME)) {
                    try {
                        retValue = AwsSignatureAuthorization.adapt(variables);
                    } catch (Exception e) {
                        logger.logMessage(LogLevel.WARNING, String.format(
                                "Fails to constructing the AwsSignatureAuthorization from the text. The Authorization of the test object will be null. Text: %s",
                                authorizationInfo.getText()), e);
                    }
                }
            }
        }

        return retValue;
    }

    /**
     * Check if the request object already has authorization headers
     */
    private static boolean hasAuthorizationHeaders(RequestObject requestObject) {
        if (requestObject.getHttpHeaderProperties() == null) {
            return false;
        }

        return requestObject.getHttpHeaderProperties()
                .stream()
                .anyMatch(header -> AUTHORIZATION_HEADER.equalsIgnoreCase(header.getName())
                        || header.getName().startsWith("Authorization:"));
    }

    /**
     * Find inherited authorization by traversing up the folder path looking for API Collection .meta files
     */
    private static BasicRequestAuthorization findInheritedAuthorizationFromPath(String requestObjectId,
            String projectDir) {
        try {
            // Convert requestObjectId to file path: "Object Repository/MyCollection/MyRequest" -> "/project/path/Object
            // Repository/MyCollection/MyRequest.rs"
            File requestFile = new File(projectDir, requestObjectId + WEBELEMENT_FILE_EXTENSION);
            File currentDir = requestFile.getParentFile();

            // Traverse up the directory hierarchy
            while (currentDir != null) {
                // Check if current directory has a .meta file
                File metaFile = new File(currentDir, API_COLLECTION_META_EXTENSION);
                if (metaFile.exists()) {
                    BasicRequestAuthorization auth = parseApiCollectionAuthorization(metaFile);
                    if (auth != null) {
                        // Found authorization in this API Collection
                        return auth;
                    }
                    // .meta exists but no authorization - continue looking up
                } else {
                    // No .meta file found - stop looking
                    break;
                }
                currentDir = currentDir.getParentFile();
            }

            // Reached Object Repository or no .meta files found - no inheritance
            return null;

        } catch (Exception e) {
            logger.logWarning("Failed to find inherited authorization: " + ExceptionsUtil.getMessageForThrowable(e),
                    null, e);
            return null;
        }
    }

    /**
     * Parse API Collection .meta file and extract authorization
     */
    private static BasicRequestAuthorization parseApiCollectionAuthorization(File metaFile) {
        try {
            SAXReader reader = SAXReaderProvider.newInstance();
            Element rootElement = reader.read(metaFile).getRootElement();

            // Check if this is an API Collection entity
            if (!"ApiCollectionEntity".equals(rootElement.getName())) {
                return null;
            }

            // Look for authorization element
            Element authorizationElement = rootElement.element("authorization");
            if (authorizationElement == null) {
                return null;
            }

            // Parse authorization directly from API Collection format
            return parseApiCollectionAuthorizationDirect(authorizationElement);

        } catch (DocumentException e) {
            throw new IllegalStateException("Authorization info is not available or is corrupted in API Collection at "
                    + metaFile.getAbsolutePath(), e);
        } catch (Exception e) {
            logger.logWarning(
                    "Failed to parse API Collection authorization: " + ExceptionsUtil.getMessageForThrowable(e), null,
                    e);
            return null;
        }
    }

    /**
     * Parse authorization element directly from API Collection .meta format
     */
    private static BasicRequestAuthorization parseApiCollectionAuthorizationDirect(Element authorizationElement) {
        try {
            String authorizationType = authorizationElement.elementText("authorizationType");
            if (authorizationType == null) {
                return null;
            }

            // Extract authorization info
            Map<String, String> variables = new HashMap<>();
            Element authorizationInfo = authorizationElement.element("authorizationInfo");
            if (authorizationInfo != null) {
                List<Element> entryElements = authorizationInfo.elements("entry");
                for (Element entry : entryElements) {
                    String key = entry.elementText("key");
                    String value = entry.elementText("value");
                    if (key != null && value != null) {
                        variables.put(key, value);
                    }
                }
            }

            // Create BasicRequestAuthorization with the parsed data
            BasicRequestAuthorization auth = new BasicRequestAuthorization();
            auth.setAuthorizationType(authorizationType);
            auth.setAuthorizationInfo(variables);

            return auth;

        } catch (Exception e) {
            logger.logWarning(
                    "Failed to parse API Collection authorization element: " + ExceptionsUtil.getMessageForThrowable(e),
                    null, e);
            return null;
        }
    }

    /**
     * Apply inherited authorization to request object based on authorization type mapping
     */
    private static void applyInheritedAuthorization(RequestObject requestObject,
            BasicRequestAuthorization inheritedAuth) {
        String authType = inheritedAuth.getAuthorizationType();
        if (authType == null) {
            return;
        }

        Map<String, String> authInfo = inheritedAuth.getAuthorizationInfo();
        if (authInfo == null) {
            return;
        }

        // Apply authorization based on type mapping
        switch (authType) {
            case AUTH_TYPE_BASIC:
                // Basic: httpHeaderProperties only
                addBasicAuthHeader(requestObject, authInfo);
                break;
            case AUTH_TYPE_BEARER:
                // Bearer: Both authorizationRequest + httpHeaderProperties
                requestObject.setRequestAuthorization(inheritedAuth);
                addBearerAuthHeader(requestObject, authInfo);
                break;
            case AUTH_TYPE_OAUTH_1_0:
                // OAuth 1.0: Both authorizationRequest + httpHeaderProperties
                requestObject.setRequestAuthorization(inheritedAuth);
                addOAuth1AuthHeaders(requestObject, authInfo);
                break;
            case AUTH_TYPE_OAUTH_2_0:
                // OAuth 2.0: Both authorizationRequest + httpHeaderProperties
                requestObject.setRequestAuthorization(inheritedAuth);
                addOAuth2AuthHeaders(requestObject, authInfo);
                break;
            case AUTH_TYPE_NTLM:
            case AUTH_TYPE_AWS_SIGNATURE:
            case AUTH_TYPE_DIGEST:
                // These types: authorizationRequest only (no headers needed)
                requestObject.setRequestAuthorization(inheritedAuth);
                break;
            // No Authorization and unknown types: handled by not setting any authorization
        }
    }

    /**
     * Add Basic authorization header
     */
    private static void addBasicAuthHeader(RequestObject requestObject, Map<String, String> authInfo) {
        String username = authInfo.get("username");
        String password = authInfo.get("password");
        if (username != null && password != null) {
            String encoded = Base64.basicEncode(username, password);
            addAuthorizationHeader(requestObject, AUTH_TYPE_BASIC + " " + encoded);
        }
    }

    /**
     * Add Bearer authorization header
     */
    private static void addBearerAuthHeader(RequestObject requestObject, Map<String, String> authInfo) {
        String token = authInfo.get("bearerToken");
        if (token != null) {
            addAuthorizationHeader(requestObject, AUTH_TYPE_BEARER + " " + token);
        }
    }

    /**
     * Add OAuth 1.0 authorization headers
     */
    private static void addOAuth1AuthHeaders(RequestObject requestObject, Map<String, String> authInfo) {
        // Add OAuth1 specific headers using exact field names from OAuth1Authorization class
        String consumerKey = authInfo.get("consumer_key");
        if (consumerKey != null) {
            addHeader(requestObject, "Authorization:oauth_consumer_key", consumerKey);
        }

        String consumerSecret = authInfo.get("consumer_secret");
        if (consumerSecret != null) {
            addHeader(requestObject, "Authorization:oauth_consumer_secret", consumerSecret);
        }

        String signatureMethod = authInfo.get("signature_method");
        if (signatureMethod != null) {
            addHeader(requestObject, "Authorization:oauth_signature_method", signatureMethod);
        }

        String token = authInfo.get("token");
        if (token != null) {
            addHeader(requestObject, "Authorization:oauth_token", token);
        }

        String tokenSecret = authInfo.get("token_secret");
        if (tokenSecret != null) {
            addHeader(requestObject, "Authorization:oauth_token_secret", tokenSecret);
        }

        String realm = authInfo.get("realm");
        if (realm != null) {
            addHeader(requestObject, "Authorization:oauth_realm", realm);
        }
    }

    /**
     * Add OAuth 2.0 authorization headers
     */
    private static void addOAuth2AuthHeaders(RequestObject requestObject, Map<String, String> authInfo) {
        // Add OAuth2 headers using exact field names from OAuth2Constants
        String grantType = authInfo.get("grant_type");
        if (grantType != null) {
            addHeader(requestObject, "Authorization:type", AUTH_TYPE_OAUTH_2_0);
        }

        String accessToken = authInfo.get("access_token");
        if (accessToken != null) {
            addAuthorizationHeader(requestObject, AUTH_TYPE_BEARER + " " + accessToken);
        }

        String clientId = authInfo.get("client_id");
        if (clientId != null) {
            addHeader(requestObject, "Authorization:oauth_consumer_key", clientId);
        }

        String clientSecret = authInfo.get("client_secret");
        if (clientSecret != null) {
            addHeader(requestObject, "Authorization:oauth_consumer_secret", clientSecret);
        }

        String tokenType = authInfo.get("token_type");
        if (tokenType != null) {
            addHeader(requestObject, "Authorization:token_type", tokenType);
        }

        String scope = authInfo.get("scope");
        if (scope != null) {
            addHeader(requestObject, "Authorization:scope", scope);
        }

        String state = authInfo.get("state");
        if (state != null) {
            addHeader(requestObject, "Authorization:state", state);
        }

        // Additional OAuth2 fields that might be present
        String username = authInfo.get("username");
        if (username != null) {
            addHeader(requestObject, "Authorization:username", username);
        }

        String password = authInfo.get("password");
        if (password != null) {
            addHeader(requestObject, "Authorization:password", password);
        }

        String authUrl = authInfo.get("auth_url");
        if (authUrl != null) {
            addHeader(requestObject, "Authorization:auth_url", authUrl);
        }

        String accessTokenUrl = authInfo.get("access_token_url");
        if (accessTokenUrl != null) {
            addHeader(requestObject, "Authorization:access_token_url", accessTokenUrl);
        }

        String redirectUri = authInfo.get("redirect_uri");
        if (redirectUri != null) {
            addHeader(requestObject, "Authorization:redirect_uri", redirectUri);
        }

        String authorizationCode = authInfo.get("code");
        if (authorizationCode != null) {
            addHeader(requestObject, "Authorization:authorization_code", authorizationCode);
        }

        String refreshToken = authInfo.get("refresh_token");
        if (refreshToken != null) {
            addHeader(requestObject, "Authorization:refresh_token", refreshToken);
        }
    }

    /**
     * Add authorization header to request object
     */
    private static void addAuthorizationHeader(RequestObject requestObject, String headerValue) {
        addHeader(requestObject, AUTHORIZATION_HEADER, headerValue);
    }

    /**
     * Add header to request object
     */
    private static void addHeader(RequestObject requestObject, String name, String value) {
        TestObjectProperty header = new TestObjectProperty(name, ConditionType.EQUALS, value, true);
        requestObject.getHttpHeaderProperties().add(header);
    }

    private static TestObject parseTestObjectsFromJson(String objectId, Map<String, Object> node) {
        boolean isMobile = node.containsKey("platform") || node.containsKey("locatorStrategy")
                || node.containsKey("locatorCollection");
        if (isMobile) {
            return buildMobileTestObject(objectId, node);
        }

        return buildTestObject(objectId, node);
    }

    @SuppressWarnings("unchecked")
    private static MobileTestObject buildMobileTestObject(String objectId, Map<String, Object> node) {
        MobileTestObject to = new MobileTestObject(objectId);

        // platform
        if (node.containsKey("platform")) {
            to.setPlatform(MobilePlatform.valueOf((String) node.get("platform")));
        }

        // locatorStrategy
        if (node.containsKey("locatorStrategy")) {
            to.setLocatorStrategy(LocatorStrategy.valueOf((String) node.get("locatorStrategy")));
        }

        // locatorCollection
        Map<String, String> rawLocatorCollection = (Map<String, String>) node.get("locatorCollection");
        if (rawLocatorCollection != null && !rawLocatorCollection.isEmpty()) {
            Map<LocatorStrategy, String> locatorCollection = new EnumMap<>(LocatorStrategy.class);
            rawLocatorCollection.forEach((k, v) -> {
                if (v != null && !v.isEmpty()) {
                    locatorCollection.put(LocatorStrategy.valueOf(k), v);
                }
            });

            to.setLocatorCollection(locatorCollection);
        }

        // -----------------------
        // properties
        // -----------------------
        applyProperties(to, (List<Map<String, Object>>) node.get("properties"));

        // -----------------------
        // xpaths
        // -----------------------
        applyXpaths(to, (List<Map<String, Object>>) node.get("xpaths"));
        return to;
    }

    @SuppressWarnings("unchecked")
    private static TestObject buildTestObject(String objectId, Map<String, Object> node) {
        TestObject to = new TestObject(objectId);

        // -----------------------
        // selectorMethod
        // -----------------------
        if (node.containsKey("selectorMethod")) {
            to.setSelectorMethod(SelectorMethod.valueOf((String) node.get("selectorMethod")));
        }

        // -----------------------
        // selectorCollection
        // -----------------------
        Map<String, String> rawSelectorCollection = (Map<String, String>) node.get("selectorCollection");
        if (rawSelectorCollection != null && !rawSelectorCollection.isEmpty()) {
            Map<SelectorMethod, String> selectorCollection = new EnumMap<>(SelectorMethod.class);
            rawSelectorCollection.forEach((k, v) -> {
                if (v != null && !v.isEmpty()) {
                    selectorCollection.put(SelectorMethod.valueOf(k), v);
                }
            });

            to.setSelectorCollection(selectorCollection);
        }

        // -----------------------
        // smartSelectorCollection
        // -----------------------
        Map<String, String> rawSmartSelectorCollection = (Map<String, String>) node.get("smartSelectorCollection");
        if (rawSmartSelectorCollection != null && !rawSmartSelectorCollection.isEmpty()) {
            Map<SelectorMethod, String> converted = new HashMap<>();
            rawSmartSelectorCollection.forEach((k, v) -> {
                if (v != null && !v.isEmpty()) {
                    converted.put(SelectorMethod.valueOf(k), v);
                }
            });

            to.setSmartSelectorCollection(converted);
        }

        // -----------------------
        // properties
        // -----------------------
        applyProperties(to, (List<Map<String, Object>>) node.get("properties"));

        // -----------------------
        // xpaths
        // -----------------------
        applyXpaths(to, (List<Map<String, Object>>) node.get("xpaths"));

        // -----------------------
        // image
        // -----------------------
        if (node.containsKey("imagePath")) {
            to.setImagePath((String) node.get("imagePath"));
        }

        if (node.containsKey("useRelativeImagePath")) {
            to.setUseRelativeImagePath(Boolean.TRUE.equals(node.get("useRelativeImagePath")));
        }

        // -----------------------
        // shadow root
        // -----------------------
        if (node.containsKey("isParentObjectShadowRoot")) {
            to.setParentObjectShadowRoot(Boolean.TRUE.equals(node.get("isParentObjectShadowRoot")));
        }

        return to;
    }

    private static void applyProperties(TestObject to, List<Map<String, Object>> props) {
        if (props == null) {
            return;
        }

        for (Map<String, Object> p : props) {
            to.addProperty((String) p.get("name"), ConditionType.valueOf((String) p.get("condition")),
                    (String) p.get("value"), Boolean.TRUE.equals(p.get("isActive")));
        }
    }

    private static void applyXpaths(TestObject to, List<Map<String, Object>> xpaths) {
        if (xpaths == null) {
            return;
        }

        for (Map<String, Object> x : xpaths) {
            to.addXpath((String) x.get("name"), ConditionType.valueOf((String) x.get("condition")),
                    (String) x.get("value"), Boolean.TRUE.equals(x.get("isActive")));
        }
    }
}
