package com.kms.katalon.core.testobject.authorization;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.GeneralSecurityException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;
import com.kms.katalon.util.CryptoUtil;

public class DigestAuthorization extends BasicRequestAuthorization {
	private static final long serialVersionUID = 1L;
	private static final String USERNAME_KEY = "username";
	private static final String PASSWORD_KEY = "password";
	private static final String REALM_KEY = "realm";
	private static final String NONCE_KEY = "nonce";
	private static final String ALGORITHM_KEY = "algorithm";
	private static final String QOP_KEY = "qop";
	private static final String NONCE_COUNT_KEY = "nonce-count";
	private static final String CLIENT_NONCE_KEY = "client-nonce";
	private static final String OPAQUE_KEY = "opaque";
	private static final String OVERRIDE_CHALLENGE_KEY = "override-challenge";

	public static final String AUTHORIZATION_TYPE = "Digest";


	public DigestAuthorization(String username, String password) throws UnsupportedEncodingException, GeneralSecurityException {
		var authorizationInfo = new HashMap<String, String>();
		authorizationInfo.put(USERNAME_KEY, username);
		authorizationInfo.put(PASSWORD_KEY, encrypt(password));
		init(authorizationInfo);
	}

	public DigestAuthorization(String username, String password, String realm, String nonce, DigestAlgorithmDirective algorithm, DigestQualityOfProtectionDirective qop,
			String nonceCount, String clientNounce, String opaque) throws UnsupportedEncodingException, GeneralSecurityException {
		var authorizationInfo = new HashMap<String, String>();
		authorizationInfo.put(USERNAME_KEY, username);
		authorizationInfo.put(PASSWORD_KEY, encrypt(password));
		authorizationInfo.put(OVERRIDE_CHALLENGE_KEY, Boolean.toString(true));
		if (Objects.nonNull(realm)) {
			authorizationInfo.put(REALM_KEY, realm);
		}

		if (Objects.nonNull(nonce)) {
			authorizationInfo.put(NONCE_KEY, nonce);
		}

		if (Objects.nonNull(algorithm)) {
			authorizationInfo.put(ALGORITHM_KEY, algorithm.getName());
		}

		if (Objects.nonNull(qop) && qop != DigestQualityOfProtectionDirective.UNSPECIFIED) {
			authorizationInfo.put(QOP_KEY, qop.getName());
		}

		if (StringUtils.isNotBlank(nonceCount)) {
		    try {
		        Integer.parseInt(nonceCount);
		        authorizationInfo.put(NONCE_COUNT_KEY, nonceCount);
		    }
		    catch (RuntimeException e) {
		        throw new IllegalArgumentException(String.format("The nonceCount text value %s isn't in number format", nonceCount), e);
		    }
		}

		if (Objects.nonNull(clientNounce)) {
			authorizationInfo.put(CLIENT_NONCE_KEY, clientNounce);
		}

		if (Objects.nonNull(opaque)) {
			authorizationInfo.put(OPAQUE_KEY, opaque);
		}

		init(authorizationInfo);
	}

	public DigestAuthorization(RequestAuthorization source) throws UnsupportedEncodingException, GeneralSecurityException {
		this(source.getAuthorizationInfo());
	}
	
	public DigestAuthorization(Map<String, String> props) throws UnsupportedEncodingException, GeneralSecurityException {
		this.init(props);
	}

	public String getUsername() {
		return getWithEmpty(USERNAME_KEY);
	}

	public String getPassword() throws GeneralSecurityException, IOException {
		String encryptedValue = getWithEmpty(PASSWORD_KEY);

		if (StringUtils.isNotBlank(encryptedValue)) {
			CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(encryptedValue);
			return CryptoUtil.decode(cryptoInfo);
		}

		return StringUtils.EMPTY;
	}

	public String getRealm() {
		return getWithEmpty(REALM_KEY);
	}

	public String getNonce() {
		return getWithEmpty(NONCE_KEY);
	}

	public DigestAlgorithmDirective getAlgorithm() {
		var vl = getAuthorizationInfo().get(ALGORITHM_KEY);
		if (StringUtils.isBlank(vl)) {
			return DigestAlgorithmDirective.MD5;
		}

		return DigestAlgorithmDirective.fromString(vl);
	}

	public DigestQualityOfProtectionDirective getQop() {
		var vl = getAuthorizationInfo().get(QOP_KEY);
		if (StringUtils.isBlank(vl)) {
			return DigestQualityOfProtectionDirective.UNSPECIFIED;
		}

		return DigestQualityOfProtectionDirective.fromString(vl);

	}

	public Integer getNonceCount() {
		var vl = getWithEmpty(NONCE_COUNT_KEY);
		if (StringUtils.isNotBlank(vl)) {
			return Integer.parseInt(vl);
		}

		return null;
	}

	public String getClientNounce() {
		return getWithEmpty(CLIENT_NONCE_KEY);
	}

	public String getOpaque() {
		return getWithEmpty(OPAQUE_KEY);
	}

	public void validate() {
		if (StringUtils.isBlank(getUsername())) {
			throw new IllegalArgumentException("The directive 'username' is required but unavailable");
		}

		// We only need to check data availability for the password, no need to decode it
		if (Objects.isNull(getAuthorizationInfo().get(PASSWORD_KEY))) {
			throw new IllegalArgumentException("The directive 'password' is required but unavailable");
		}

		if (!useChallengeResponseMechanism()) {
			if (StringUtils.isBlank(getRealm())) {
				throw new IllegalArgumentException("The directive 'realm' is required but unavailable");
			}

			if (StringUtils.isBlank(getNonce())) {
				throw new IllegalArgumentException("The directive 'nonce' is required but unavailable");
			}
		}
	}

	public boolean useChallengeResponseMechanism() {
		var dict = getAuthorizationInfo();
		if (dict.containsKey(OVERRIDE_CHALLENGE_KEY)) {
			var vl = dict.get(OVERRIDE_CHALLENGE_KEY);
			if (StringUtils.isNotBlank(vl)) {
				return !Boolean.valueOf(vl);
			}
		}
		
		return true;
	}
	
	public static boolean is(RequestAuthorization ra) {
	    if (Objects.nonNull(ra)) {
	        var at = ra.getAuthorizationType();
	        if (StringUtils.isNotBlank(at)) return at.equals(AUTHORIZATION_TYPE);
	    }
	    
		return false;
	}

	private String encrypt(String vl) throws UnsupportedEncodingException, GeneralSecurityException {
		CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(vl);
		String encryptedValue = StringUtils.EMPTY;
		return CryptoUtil.encode(cryptoInfo);
	}

	private void init(Map<String, String> info) {
		this.setAuthorizationType(AUTHORIZATION_TYPE);
		this.setAuthorizationInfo(info);
	}
	
	private String getWithEmpty(String key) {
		var dict = getAuthorizationInfo();
		if (!dict.containsKey(key)) return StringUtils.EMPTY;
		
		var vl = dict.get(key);
		return StringUtils.isBlank(vl) ? StringUtils.EMPTY : vl;
	}
}
