package com.kms.katalon.core.trace;

import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.context.internal.ExecutionListenerEvent;
import com.kms.katalon.core.context.internal.ExecutionListenerEventHandler;
import com.kms.katalon.core.context.internal.InternalTestCaseContext;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.util.VideoRecorderUtil;
import com.kms.katalon.core.util.internal.TestOpsUtil;
import java.io.File;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Stack;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Java reimplementation of TraceRecorderService to avoid Groovy compilation gaps during Tycho builds.
 * Uses reflection to interact with Groovy-based trace classes at runtime.
 */
public class TraceRecorderService implements ExecutionListenerEventHandler {
    private static final String TRACE_KEYWORD_NAME = "Trace";

    private final KeywordLogger logger = KeywordLogger.getInstance(this.getClass());

    private Object session;
    private final Map<Integer, String> callIds = new ConcurrentHashMap<>();
    private ScheduledExecutorService screencastExec;

    @Override
    public void handleListenerEvent(ExecutionListenerEvent listenerEvent, Object[] injectedObjects) {
        try {
            switch (listenerEvent) {
                case BEFORE_TEST_CASE:
                    onBeforeTestCase((InternalTestCaseContext) injectedObjects[0]);
                    break;
                case AFTER_TEST_CASE:
                    onAfterTestCase((InternalTestCaseContext) injectedObjects[0]);
                    break;
                case BEFORE_TEST_STEP:
                    onBeforeTestStep(injectedObjects);
                    break;
                case AFTER_TEST_STEP:
                    onAfterTestStep(injectedObjects);
                    break;
                default:
                    break;
            }
        } catch (Throwable t) {
            traceDebug("TraceRecorderService error: " + t.getMessage());
        }
    }

    private void onBeforeTestCase(InternalTestCaseContext ctx) {
        boolean decision = RunConfiguration.shouldTraceFor(ctx);
        int suiteRerunIndex = RunConfiguration.getSuiteRerunIndex();
        int tcRetry = ctx != null ? ctx.getRetryIndex() : 0;
        traceDebug("Trace decision: mode=" + RunConfiguration.getTraceMode()
                + " tcRetry=" + tcRetry
                + " suiteRerunIndex=" + suiteRerunIndex
                + " retryRun=" + Math.max(tcRetry, suiteRerunIndex)
                + " decision=" + decision);
        if (!decision) {
            traceHolderSetSession(null);
            session = null;
            return;
        }
        // Reset any lingering HAR capture state from previous test cases.
        try {
            Class<?> har = Class.forName("com.kms.katalon.core.webui.trace.HarTracer");
            har.getMethod("stopAndFlush", long.class).invoke(null, 0L);
        } catch (Throwable ignored) { }
        File reportDir = new File(RunConfiguration.getReportFolder());
        String safeId = sanitize(ctx.getTestCaseId());
        File tempTracingDir = new File(reportDir, "tmp/tracing/" + safeId);
        traceDebugInit(new File(tempTracingDir, "trace-debug-" + safeId + ".log"));
        File tracesDir = tempTracingDir;
        session = newTraceSession(tracesDir);
        traceHolderSetSession(session);
        invoke(session, "start", ctx.getTestCaseId());
        traceDebug("TraceRecorderService: session created at " + tracesDir.getAbsolutePath());
        try {
            Class<?> ki = Class.forName("com.kms.katalon.core.webui.trace.KeywordInterceptor");
            ki.getMethod("reset").invoke(null);
            ki.getMethod("install").invoke(null);
        } catch (Throwable ignored) { /* best effort */ }
        screencastExec = Executors.newSingleThreadScheduledExecutor();
        screencastExec.scheduleAtFixedRate(() -> {
            try { invoke(session, "captureScreencastFrame"); } catch (Throwable ignored) {}
        }, 500, 1000, TimeUnit.MILLISECONDS);
    }

    private void onAfterTestCase(InternalTestCaseContext ctx) {
        if (session == null) return;
        try {
            File reportDir = new File(RunConfiguration.getReportFolder());
            String safeId = sanitize(ctx.getTestCaseId());
            // Where this test case's temporary trace workspace lived:
            // Reports/<...>/tmp/tracing/<safeId>/
            File tempTracingDir = new File(reportDir, "tmp/tracing/" + safeId);
            File tracesOutDir = new File(reportDir, "traces");
            // Trace file naming:
            // - Keep it readable (no ellipsis "...").
            // - Avoid characters that are problematic on filesystems.
            // - Preserve the conventional suffix: <name>_<index>_<retry>.ktrace
            String baseName = buildTraceFileName(ctx);
            File out = new File(tracesOutDir, baseName);
            int n = 1;
            while (out.exists() && n < 1000) {
                String candidate = baseName.replaceFirst("\\.ktrace$", "-" + n + ".ktrace");
                out = new File(tracesOutDir, candidate);
                n++;
            }
            invoke(session, "stopChunk", out);
            traceDebug("TraceRecorderService: packaged -> " + out.getAbsolutePath());
            // Best-effort cleanup of intermediate trace artifacts under tmp/tracing,
            // now that the .ktrace bundle has been produced. We preserve trace-debug
            // logs but remove the bulk data to keep report folders small.
            cleanupTempTracingDir(tempTracingDir);
            maybeDeleteOnPass(ctx, out);
            logTraceRecordingStep(ctx, out);
        } finally {
            traceHolderSetSession(null);
            session = null;
            try { if (screencastExec != null) screencastExec.shutdownNow(); } catch (Throwable ignored) {}
        }
    }

    private void cleanupTempTracingDir(File tempTracingDir) {
        if (tempTracingDir == null || !tempTracingDir.exists()) {
            return;
        }
        File[] children = tempTracingDir.listFiles();
        if (children == null) {
            return;
        }
        for (File child : children) {
            String name = child.getName();
            // Keep debug logs for troubleshooting; remove everything else.
            if (child.isFile() && name.startsWith("trace-debug-") && name.endsWith(".log")) {
                continue;
            }
            deleteRecursively(child);
        }
    }

    private void deleteRecursively(File file) {
        if (file == null || !file.exists()) {
            return;
        }
        if (file.isDirectory()) {
            File[] children = file.listFiles();
            if (children != null) {
                for (File child : children) {
                    deleteRecursively(child);
                }
            }
        }
        try {
            file.delete();
        } catch (Throwable ignored) {
            // Best effort; trace cleanup must not interfere with test results.
        }
    }

    private void onBeforeTestStep(Object[] info) {
        if (session == null || !getRecordingFlag(session)) return;
        int stepIdx = (int) info[0];
        String description = info.length > 1 ? String.valueOf(Objects.requireNonNullElse(info[1], "")) : "";
        String keywordName = info.length > 2 ? String.valueOf(Objects.requireNonNullElse(info[2], "")) : "";
        Map<String, Object> params = new java.util.LinkedHashMap<>();
        if (!description.isEmpty()) params.put("runtimeDescription", description);
        if (!keywordName.isEmpty()) params.put("runtimeAction", keywordName);
        params.put("runtimeStepIndex", stepIdx);
        params.put("runtimePhase", "Before");
        String runtimeTitle = !description.isEmpty() ? description : (!keywordName.isEmpty() ? keywordName : "Step " + stepIdx);
        String title = "Before " + runtimeTitle;
        String callId = (String) invoke(session, "beginStep", title, params);
        callIds.put(stepIdx, callId);
    }

    private void onAfterTestStep(Object[] info) {
        if (session == null || !getRecordingFlag(session)) return;
        int stepIdx = (int) info[0];
        String callId = callIds.remove(stepIdx);
        if (callId == null) return;
        String description = info.length > 1 ? String.valueOf(Objects.requireNonNullElse(info[1], "")) : "";
        String keywordName = info.length > 2 ? String.valueOf(Objects.requireNonNullElse(info[2], "")) : "";
        Map<String, Object> meta = new java.util.LinkedHashMap<>();
        meta.put("runtimePhase", "After");
        meta.put("runtimeStepIndex", stepIdx);
        if (!description.isEmpty()) meta.put("runtimeDescription", description);
        if (!keywordName.isEmpty()) meta.put("runtimeAction", keywordName);
        invoke(session, "endStep", callId, "ok", meta);
    }

    private static Object newTraceSession(File dir) {
        try {
            Class<?> cls = Class.forName("com.kms.katalon.core.webui.trace.TraceSession");
            return cls.getDeclaredConstructor(File.class).newInstance(dir);
        } catch (Throwable t) {
            traceDebugStatic("TraceRecorderService: cannot instantiate TraceSession " + t.getMessage());
            return null;
        }
    }

    private static Object invoke(Object target, String method, Object... args) {
        if (target == null) return null;
        try {
            // Prefer Groovy's invokeMethod if present
            Method m = target.getClass().getMethod("invokeMethod", String.class, Object.class);
            return m.invoke(target, method, args);
        } catch (NoSuchMethodException e) {
            // Fallback to reflection by name/arity
            for (Method cand : target.getClass().getMethods()) {
                if (!cand.getName().equals(method)) continue;
                if (cand.getParameterCount() != args.length) continue;
                try {
                    cand.setAccessible(true);
                    return cand.invoke(target, args);
                } catch (Throwable ignored) { /* try next */ }
            }
            traceDebugStatic("TraceRecorderService invoke " + method + " missing");
            return null;
        } catch (Throwable t) {
            traceDebugStatic("TraceRecorderService invoke " + method + " error " + t.getMessage());
            return null;
        }
    }

    private static boolean getRecordingFlag(Object session) {
        try {
            Method m = session.getClass().getMethod("getRecording");
            Object val = m.invoke(session);
            return Boolean.TRUE.equals(val);
        } catch (Throwable ignored) {
            try {
                Method m = session.getClass().getMethod("isRecording");
                Object val = m.invoke(session);
                return Boolean.TRUE.equals(val);
            } catch (Throwable ignored2) {
                return false;
            }
        }
    }

    private static String sanitize(String s) {
        return (s == null ? "" : s).replaceAll("[\\\\/:*?\"<>|\\n\\r\\t ]+", "-");
    }

    private static String buildTraceFileName(InternalTestCaseContext ctx) {
        String raw = ctx != null ? ctx.getTestCaseName() : null;
        String base = VideoRecorderUtil.getTestCaseNameFromFullPath(raw);
        base = sanitize(base).replaceAll("-{2,}", "-");
        base = base.replaceAll("^-", "").replaceAll("-$", "");
        if (base.isEmpty()) {
            base = "trace";
        }

        // Keep a generous cap to avoid OS path limits, but do not insert "...".
        // If very long, keep the start and append a stable hash.
        final int max = 120;
        if (base.length() > max) {
            String hash = Integer.toHexString(base.hashCode());
            int keep = Math.max(1, max - (hash.length() + 2));
            base = base.substring(0, keep) + "--" + hash;
        }

        int index = ctx != null ? ctx.getTestCaseIndex() : 0;
        int retryIdx = ctx != null ? ctx.getRetryIndex() : 0;
        return base + "_" + (index + 1) + "_" + retryIdx + ".ktrace";
    }

    private static void traceHolderSetSession(Object s) {
        try {
            Class<?> cls = Class.forName("com.kms.katalon.core.trace.TraceHolder");
            cls.getMethod("setSession", Object.class).invoke(null, s);
        } catch (Throwable ignored) { }
    }

    private static void traceDebugInit(File file) {
        try {
            Class<?> cls = Class.forName("com.kms.katalon.core.trace.TraceDebug");
            cls.getMethod("init", File.class).invoke(null, file);
        } catch (Throwable ignored) { }
    }

    private static void traceDebug(String msg) {
        traceDebugStatic(msg);
    }

    private static void traceDebugStatic(String msg) {
        try {
            Class<?> cls = Class.forName("com.kms.katalon.core.trace.TraceDebug");
            cls.getMethod("writeLine", String.class).invoke(null, msg);
        } catch (Throwable ignored) { }
    }
    private void maybeDeleteOnPass(InternalTestCaseContext ctx, File out) {
        RunConfiguration.TraceMode mode = RunConfiguration.getTraceMode();
        if (mode == RunConfiguration.TraceMode.ON_FAILURE || mode == RunConfiguration.TraceMode.ON_FIRST_FAILURE) {
            String status = ctx != null ? ctx.getTestCaseStatus() : null;
            if (status != null && "PASSED".equalsIgnoreCase(status) && out != null && out.exists()) {
                out.delete();
            }
        }
    }

    private void logTraceRecordingStep(InternalTestCaseContext ctx, File out) {
        if (ctx == null || out == null || !out.exists()) {
            return;
        }
        String relativePath = TestOpsUtil.getRelativePathForLog(out.getAbsolutePath());
        if (relativePath == null || relativePath.isEmpty()) {
            return;
        }
        Map<String, String> attributes = new HashMap<>();
        attributes.put(StringConstants.XML_LOG_ATTACHMENT_PROPERTY, relativePath);
        String message = "Trace recording for test case '" + ctx.getTestCaseId() + "'.";
        logger.startKeyword(TRACE_KEYWORD_NAME, new HashMap<>(), new Stack<>());
        logger.logInfo(message, attributes);
        logger.endKeyword(TRACE_KEYWORD_NAME, new HashMap<>(), new Stack<>());
    }
}
