package com.kms.katalon.core.trace;

import com.kms.katalon.core.util.internal.JsonUtil;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

public final class TraceUtils {
    private TraceUtils() {
    }

    public static long monotonicMs() {
        return TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
    }

    public static String sha1Hex(byte[] bytes) {
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-1");
            byte[] digest = md.digest(bytes);
            StringBuilder sb = new StringBuilder(digest.length * 2);
            for (byte b : digest) {
                sb.append(String.format("%02x", b));
            }
            return sb.toString();
        } catch (NoSuchAlgorithmException e) {
            return "";
        }
    }

    public static void ensureDir(File dir) {
        if (dir == null) {
            return;
        }
        if (!dir.exists()) {
            dir.mkdirs();
        }
    }

    public static void writeBytes(File file, byte[] bytes) {
        if (file == null || bytes == null) {
            return;
        }
        ensureDir(file.getParentFile());
        try (FileOutputStream fos = new FileOutputStream(file)) {
            fos.write(bytes);
        } catch (IOException ignored) {
        }
    }

    public static void appendJsonLine(File file, java.util.Map<String, Object> event) {
        if (file == null || event == null) {
            return;
        }
        ensureDir(file.getParentFile());
        try (FileOutputStream fos = new FileOutputStream(file, true)) {
            // Playwright expects NDJSON: one compact JSON object per line.
            String line = JsonUtil.toJson(event, false) + "\n";
            fos.write(line.getBytes(StandardCharsets.UTF_8));
        } catch (Throwable ignored) {
            // ignore
        }
        try {
            Object type = event.get("type");
            Object snapshot = event.get("snapshot");
            String kind;
            if (type != null) {
                kind = String.valueOf(type);
            } else if (snapshot != null) {
                kind = "frame-snapshot";
            } else {
                kind = "unknown";
            }
            TraceDebug.writeLine("appendJsonLine: " + kind + " -> " + file.getName());
        } catch (Throwable ignored) {
        }
    }

    public static byte[] pngToJpeg(byte[] pngBytes) {
        try {
            ByteArrayInputStream bais = new ByteArrayInputStream(pngBytes);
            BufferedImage img = ImageIO.read(bais);
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write(img, "jpeg", baos);
            return baos.toByteArray();
        } catch (Throwable t) {
            return pngBytes;
        }
    }

    public static void zipEntries(File zipFile, List<java.util.Map<String, Object>> entries) {
        if (zipFile == null) {
            return;
        }
        ensureDir(zipFile.getParentFile());
        try (ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(zipFile))) {
            for (java.util.Map<String, Object> e : entries) {
                Object nameObj = e.get("name");
                Object fileObj = e.get("file");
                if (!(nameObj instanceof String) || !(fileObj instanceof File)) {
                    continue;
                }
                String name = (String) nameObj;
                File src = (File) fileObj;
                if (src == null || !src.exists()) {
                    continue;
                }
                try {
                    TraceDebug.writeLine("zip: add " + name + " size=" + src.length());
                } catch (Throwable ignored) {
                }
                zos.putNextEntry(new ZipEntry(name.replace('\\', '/')));
                try (FileInputStream fis = new FileInputStream(src)) {
                    fis.transferTo(zos);
                }
                zos.closeEntry();
            }
        } catch (Throwable ignored) {
        }
        try {
            TraceDebug.writeLine("zip: wrote " + zipFile.getAbsolutePath() + " size=" + zipFile.length());
        } catch (Throwable ignored) {
        }
    }
}
