package com.kms.katalon.core.util;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.Strings;
import org.jsoup.nodes.Element;

public class MobilePageSourceMinifier extends HTMLMinifier {

    // Android default values
    private static final Map<String, String> ANDROID_DEFAULT_VALUES = new HashMap<>();
    static {
        // Basic Android attributes
        ANDROID_DEFAULT_VALUES.put("class", null);
        ANDROID_DEFAULT_VALUES.put("package", null);
        ANDROID_DEFAULT_VALUES.put("text", null);
        ANDROID_DEFAULT_VALUES.put("content-desc", null);
        ANDROID_DEFAULT_VALUES.put("resource-id", null);
        ANDROID_DEFAULT_VALUES.put("index", "0");

        // Geometry
        ANDROID_DEFAULT_VALUES.put("bounds", "[0,0][0,0]");
        ANDROID_DEFAULT_VALUES.put("x", "0");
        ANDROID_DEFAULT_VALUES.put("y", "0");
        ANDROID_DEFAULT_VALUES.put("width", "0");
        ANDROID_DEFAULT_VALUES.put("height", "0");
        ANDROID_DEFAULT_VALUES.put("displayed", "false");

        // Standard state attributes (Accessibility flags)
        ANDROID_DEFAULT_VALUES.put("checkable", "false");
        ANDROID_DEFAULT_VALUES.put("checked", "false");
        ANDROID_DEFAULT_VALUES.put("clickable", "false");
        ANDROID_DEFAULT_VALUES.put("enabled", "true");
        ANDROID_DEFAULT_VALUES.put("focusable", "false");
        ANDROID_DEFAULT_VALUES.put("focused", "false");
        ANDROID_DEFAULT_VALUES.put("scrollable", "false");
        ANDROID_DEFAULT_VALUES.put("longClickable", "false");
        ANDROID_DEFAULT_VALUES.put("long-clickable", "false"); // kebab-case variant
        ANDROID_DEFAULT_VALUES.put("selected", "false");
        ANDROID_DEFAULT_VALUES.put("password", "false");

        // Appium / UIAutomator2 custom enrichment
        ANDROID_DEFAULT_VALUES.put("xpath", null);
        ANDROID_DEFAULT_VALUES.put("drawing-order", "0");
        ANDROID_DEFAULT_VALUES.put("text-entry-key", "false");
        ANDROID_DEFAULT_VALUES.put("dismissable", "false");
        ANDROID_DEFAULT_VALUES.put("context-clickable", "false");
        ANDROID_DEFAULT_VALUES.put("showing-hint", "false");

        // A11y extended attributes (Android 9+)
        ANDROID_DEFAULT_VALUES.put("a11y-important", "false");
        ANDROID_DEFAULT_VALUES.put("a11y-focused", "false");
        ANDROID_DEFAULT_VALUES.put("screen-reader-focusable", "false");
        ANDROID_DEFAULT_VALUES.put("live-region", "0");
        ANDROID_DEFAULT_VALUES.put("heading", "false");
        ANDROID_DEFAULT_VALUES.put("content-invalid", "false");

        // Rare / OEM / Android system attributes
        ANDROID_DEFAULT_VALUES.put("hintText", null);
        ANDROID_DEFAULT_VALUES.put("clickableHost", "false");
        ANDROID_DEFAULT_VALUES.put("importantForAutofill", "0");
        ANDROID_DEFAULT_VALUES.put("autofillHints", null);
        ANDROID_DEFAULT_VALUES.put("accessibilityTraversalBefore", null);
        ANDROID_DEFAULT_VALUES.put("accessibilityTraversalAfter", null);
        ANDROID_DEFAULT_VALUES.put("class-chain", null);
        ANDROID_DEFAULT_VALUES.put("android-widget-hash", null);
        ANDROID_DEFAULT_VALUES.put("clickableRegion", null);
        ANDROID_DEFAULT_VALUES.put("visibleToUser", "true");
    }

    // iOS default values
    private static final Map<String, String> IOS_DEFAULT_VALUES = new HashMap<>();
    static {
        // Basic iOS attributes
        IOS_DEFAULT_VALUES.put("type", null);
        IOS_DEFAULT_VALUES.put("name", null);
        IOS_DEFAULT_VALUES.put("label", null);
        IOS_DEFAULT_VALUES.put("value", null);
        IOS_DEFAULT_VALUES.put("hint", null);

        // Geometry
        IOS_DEFAULT_VALUES.put("x", "0");
        IOS_DEFAULT_VALUES.put("y", "0");
        IOS_DEFAULT_VALUES.put("width", "0");
        IOS_DEFAULT_VALUES.put("height", "0");
        IOS_DEFAULT_VALUES.put("visible", "false");
        IOS_DEFAULT_VALUES.put("accessible", "true");

        // State attributes
        IOS_DEFAULT_VALUES.put("enabled", "true");
        IOS_DEFAULT_VALUES.put("selected", "false");

        // Appium custom
        IOS_DEFAULT_VALUES.put("xpath", null);
        IOS_DEFAULT_VALUES.put("index", "0");
    }

    public MobilePageSourceMinifier(String html) {
        super(html);
    }

    @Override
    protected String minifyDOM(String locator) {
        super.minifyDOM(locator);

        runDOMMinizationPhase("Remove hidden or invisible or zero bounds elements", () -> {
            this.removeInvisibleElements();
        });

        runDOMMinizationPhase("Remove redundant mobile attributes", () -> {
            this.removeMobileRedundantAttributes();
        });

        runDOMMinizationPhase("Remove attributes with default values", () -> {
            this.removeAttributesWithDefaultValues();
        });

        return this.buildHTML();
    }

    @Override
    protected String minifyHTML(String html) {
        html = removeRedundantSpaces(html);
        return html;
    }

    private String removeRedundantSpaces(String html) {
        html = html.replaceAll("\r\n", "").replaceAll(">\\s*", ">");
        return html;
    }

    private void removeMobileRedundantAttributes() {
        this.removeMobileRedundantAttributes(getRootElement());
    }

    private void removeMobileRedundantAttributes(Element root) {
        var attributes = root.attributes();
        for (var attribute : attributes.clone()) {
            var attributeName = attribute.getKey();

            // Remove redundant attributes that don't provide semantic value for AI models
            if (Strings.CI.equalsAny(attributeName, "class",                          // Duplicate of tag name
                    "package",                        // Same for all elements in app
                    "xpath",                          // Generated locator, no semantic value
                    "bounds",                         // Duplicate of x,y,width,height
                    "drawing-order",                  // Rendering metadata
                    "android-widget-hash",            // Technical hash
                    "class-chain",                    // iOS locator strategy
                    "accessibilityTraversalBefore",   // Navigation hints
                    "accessibilityTraversalAfter",    // Navigation hints
                    "clickableRegion",                // Duplicate of clickable
                    "importantForAutofill",           // Autofill metadata
                    "autofillHints")) {               // Autofill hints
                root.removeAttr(attributeName);
            }
        }
        root.children().forEach((child) -> {
            removeMobileRedundantAttributes(child);
        });
    }

    private void removeAttributesWithDefaultValues() {
        this.removeAttributesWithDefaultValues(getRootElement());
    }

    private void removeAttributesWithDefaultValues(Element root) {
        var attributes = root.attributes();

        for (var attribute : attributes.clone()) {
            var attributeName = attribute.getKey();
            var attributeValue = attribute.getValue();

            // Check Android defaults
            if (ANDROID_DEFAULT_VALUES.containsKey(attributeName)) {
                String defaultValue = ANDROID_DEFAULT_VALUES.get(attributeName);
                if (isMatchingDefaultValue(attributeValue, defaultValue)) {
                    root.removeAttr(attributeName);
                    continue;
                }
            }

            // Check iOS defaults
            if (IOS_DEFAULT_VALUES.containsKey(attributeName)) {
                String defaultValue = IOS_DEFAULT_VALUES.get(attributeName);
                if (isMatchingDefaultValue(attributeValue, defaultValue)) {
                    root.removeAttr(attributeName);
                    continue;
                }
            }
        }

        root.children().forEach((child) -> {
            removeAttributesWithDefaultValues(child);
        });
    }

    private boolean isMatchingDefaultValue(String actualValue, String defaultValue) {
        // Handle null defaults
        if (defaultValue == null) {
            return actualValue == null || actualValue.isEmpty() || actualValue.equals("null");
        }

        // Handle empty string
        if (actualValue == null || actualValue.isEmpty()) {
            return defaultValue.isEmpty();
        }

        // Direct string comparison
        if (actualValue.equals(defaultValue)) {
            return true;
        }

        // Case-insensitive comparison for boolean values
        if (defaultValue.equals("true") || defaultValue.equals("false")) {
            return actualValue.equalsIgnoreCase(defaultValue);
        }

        return false;
    }

    private void removeInvisibleElements() {
        this.removeInvisibleElements(getRootElement());
    }

    private void removeInvisibleElements(Element root) {
        // Remove invisible children first (bottom-up approach)
        var children = root.children();
        for (int i = children.size() - 1; i >= 0; i--) {
            Element child = children.get(i);
            removeInvisibleElements(child);

            if (isInvisibleElement(child)) {
                child.remove();
            }
        }
    }

    private boolean isInvisibleElement(Element element) {
        // Android: check displayed, visible, visibleToUser attributes
        String displayed = element.attr("displayed");
        if ("false".equalsIgnoreCase(displayed)) {
            return true;
        }

        String visibleToUser = element.attr("visibleToUser");
        if ("false".equalsIgnoreCase(visibleToUser)) {
            return true;
        }

        // iOS: check visible attribute
        String visible = element.attr("visible");
        if ("false".equalsIgnoreCase(visible)) {
            return true;
        }

        // Check for zero bounds (Android format: [x,y][x2,y2])
        String bounds = element.attr("bounds");
        if (bounds != null && !bounds.isEmpty()) {
            if (isZeroBounds(bounds)) {
                return true;
            }
        }

        // Check for zero width or height
        String width = element.attr("width");
        String height = element.attr("height");

        if (width != null && !width.isEmpty()) {
            try {
                if (Integer.parseInt(width) == 0) {
                    return true;
                }
            } catch (NumberFormatException e) {
                // Ignore parsing errors
            }
        }

        if (height != null && !height.isEmpty()) {
            try {
                if (Integer.parseInt(height) == 0) {
                    return true;
                }
            } catch (NumberFormatException e) {
                // Ignore parsing errors
            }
        }

        return false;
    }

    private boolean isZeroBounds(String bounds) {
        // Android bounds format: [x1,y1][x2,y2]
        // Zero bounds means x1==x2 and y1==y2
        try {
            // Extract coordinates
            String[] parts = bounds.replace("][", ",").replace("[", "").replace("]", "").split(",");
            if (parts.length == 4) {
                int x1 = Integer.parseInt(parts[0].trim());
                int y1 = Integer.parseInt(parts[1].trim());
                int x2 = Integer.parseInt(parts[2].trim());
                int y2 = Integer.parseInt(parts[3].trim());

                // Zero bounds if width or height is zero
                return (x2 - x1 == 0) || (y2 - y1 == 0);
            }
        } catch (NumberFormatException e) {
            // Ignore parsing errors
        }
        return false;
    }
}
