package com.kms.katalon.core.util.internal;

import java.io.Reader;
import java.lang.reflect.Type;
import java.text.MessageFormat;
import java.time.OffsetDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.google.gson.*;
import org.apache.commons.lang3.StringUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import com.kms.katalon.core.constants.StringConstants;
import com.networknt.schema.Schema;
import com.networknt.schema.SchemaRegistry;
import com.networknt.schema.dialect.Dialects;

public class JsonUtil {

    /**
     * Create a GsonBuilder with ZonedDateTime serializer registered
     *
     * @return GsonBuilder with ZonedDateTime support
     */
    private static GsonBuilder createGsonBuilderWithIsoDateTimeSupport() {
        return new GsonBuilder()
                .registerTypeAdapter(ZonedDateTime.class, new JsonSerializer<ZonedDateTime>() {
                    @Override
                    public JsonElement serialize(ZonedDateTime src, Type typeOfSrc, JsonSerializationContext context) {
                        return new JsonPrimitive(src.format(DateTimeFormatter.ISO_ZONED_DATE_TIME));
                    }
                })
                .registerTypeAdapter(OffsetDateTime.class, new JsonSerializer<OffsetDateTime>() {
                    @Override
                    public JsonElement serialize(OffsetDateTime src, Type typeOfSrc, JsonSerializationContext context) {
                        return new JsonPrimitive(src.format(DateTimeFormatter.ISO_OFFSET_DATE_TIME));
                    }
                })
                .registerTypeAdapter(OffsetDateTime.class, new JsonDeserializer<OffsetDateTime>() {
                    @Override
                    public OffsetDateTime deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
                        return OffsetDateTime.parse(json.getAsString(), DateTimeFormatter.ISO_OFFSET_DATE_TIME);
                    }
                });
    }

    /**
     * Convert an object to JSON string. Pretty print by default.
     * 
     * @param src source object
     * @return JSON string
     */
    public static String toJson(Object src) {
        return toJson(src, true);
    }

    /**
     * Convert an object to JSON string.
     * 
     * @param src source object
     * @param prettyPrint pretty print
     * @return JSON string
     */
    public static String toJson(Object src, boolean prettyPrint) {
        return toJson(src, prettyPrint, false);
    }
    
    /**
     * Convert an object to JSON string.
     * 
     * @param src source object
     * @param prettyPrint pretty print
     * @param excludeFieldsWithoutExposeAnnotation exclude the fields without expose annotation
     * @return JSON string
     */
    public static String toJson(Object src, boolean prettyPrint, boolean excludeFieldsWithoutExposeAnnotation) {
        return toJson(src, prettyPrint, excludeFieldsWithoutExposeAnnotation, false);
    }

    /**
     * Convert an object to JSON string.
     * 
     * @param src source object
     * @param prettyPrint pretty print
     * @param excludeFieldsWithoutExposeAnnotation exclude the fields without expose annotation
     * @param disableHtmlEscaping - By default, Gson encode characters such as <, >, =, etc. Use this flag to configure Gson to leave these characters unchanged.
     * @return JSON string
     */
    public static String toJson(Object src, boolean prettyPrint, boolean excludeFieldsWithoutExposeAnnotation, boolean disableHtmlEscaping) {
        GsonBuilder gsonBuilder = createGsonBuilderWithIsoDateTimeSupport();
        if (disableHtmlEscaping) {
            gsonBuilder.disableHtmlEscaping();
        }
        
        Gson gson;
        if (prettyPrint || excludeFieldsWithoutExposeAnnotation) {
            if (prettyPrint) {
                gsonBuilder = gsonBuilder.setPrettyPrinting();
            }
            if (excludeFieldsWithoutExposeAnnotation) {
                gsonBuilder = gsonBuilder.excludeFieldsWithoutExposeAnnotation();
            }
            gson = gsonBuilder.create();
        } else {
            gson = gsonBuilder.create();
        }
        return gson.toJson(src);
    }

    /**
     * Convert an object to JSON string. Pretty print by default.
     * 
     * @param src source object
     * @param typeOfSrc type of source
     * @return JSON string
     */
    public static String toJson(Object src, Type typeOfSrc) {
        return toJson(src, typeOfSrc, true);
    }

    /**
     * Convert an object to JSON string.
     * 
     * @param src source object
     * @param typeOfSrc type of source
     * @param prettyPrint pretty print
     * @return JSON string
     */
    public static String toJson(Object src, Type typeOfSrc, boolean prettyPrint) {
        GsonBuilder gsonBuilder = createGsonBuilderWithIsoDateTimeSupport();
        if (prettyPrint) {
            gsonBuilder.setPrettyPrinting();
        }
        Gson gson = gsonBuilder.create();
        return gson.toJson(src, typeOfSrc);
    }

    @SuppressWarnings("unchecked")
    public static <T> List<T> fromJsonArray(String json, Class<T[]> arrayType) {
        Object[] items = (Object[]) fromJson(json, arrayType);
        return (List<T>) Arrays.asList(items);
    }

    /**
     * Convert JSON string to Object
     * 
     * @param json JSON string
     * @param typeOfT type of target object
     * @return target object
     * @throws IllegalArgumentException if the input is invalid JSON syntax
     */
    public static <T> T fromJson(String json, Type typeOfT) throws IllegalArgumentException {
        try {
            return createGsonBuilderWithIsoDateTimeSupport().create().fromJson(json, typeOfT);
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    public static <T> T fromJson(Reader reader, Type typeOfT) throws IllegalArgumentException {
        try {
            return createGsonBuilderWithIsoDateTimeSupport().create().fromJson(reader, typeOfT);
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    public static <T> T fromJson(Reader reader, Class<T> classOfT) throws IllegalArgumentException {
        try {
            return createGsonBuilderWithIsoDateTimeSupport().create().fromJson(reader, classOfT);
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    /**
     * Convert JSON string to Object
     * 
     * @param json JSON string
     * @param classOfT class of target object
     * @return target object
     * @throws IllegalArgumentException if the input is invalid JSON syntax
     */
    public static <T> T fromJson(String json, Class<T> classOfT) throws IllegalArgumentException {
        try {
            return fromJson(null, json, classOfT);
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    /**
     * Convert JSON string to Object containing Date with given format
     * 
     * @param dateTimeFormat Datetime format 
     * @param json JSON string
     * @param classOfT class of target object
     * @return target object
     * @throws IllegalArgumentException if the input is invalid JSON syntax
     */
    public static <T> T fromJson(String dateTimeFormat, String json, Class<T> classOfT) throws IllegalArgumentException {
        try {
            if (StringUtils.isBlank(dateTimeFormat)) {
                return (T) createGsonBuilderWithIsoDateTimeSupport().create().fromJson(json, classOfT);
            } else {
                return (T) createGsonBuilderWithIsoDateTimeSupport().setDateFormat(dateTimeFormat).create().fromJson(json, classOfT);
            }
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    public static JsonObject toJsonObject(Object src) {
        Gson gson = createGsonBuilderWithIsoDateTimeSupport().create();
        return gson.toJsonTree(src).getAsJsonObject();
    }

    public static void mergeJsonObject(JsonObject src, JsonObject dst) {
        for (Map.Entry<String, JsonElement> entry : src.entrySet()) {
            dst.add(entry.getKey(), entry.getValue());
        }
    }

    public static boolean isValidJsonSchema(String aSchema) {
        ObjectMapper mapper = new ObjectMapper();
        JsonNode schemaNode;
        try {
            schemaNode = mapper.readTree(aSchema);

            if (schemaNode == null) {
                return false;
            }

            Schema jsonSchema = parseJsonSchema(schemaNode);

            if (jsonSchema == null) {
                return false;
            }

            return true;
        } catch (JsonProcessingException e) {
            return false;
        }
    }
    
    private static Schema parseJsonSchema(JsonNode schemaNode) {
        if (schemaNode == null) {
            return null;
        }

        return SchemaRegistry.withDialect(Dialects.getDraft202012()).getSchema(schemaNode);
    }
}
