package com.kms.katalon.core.util.internal;

import com.kms.katalon.core.constants.CoreConstants;
import com.kms.katalon.core.keyword.internal.IKeywordLoader;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.wiring.BundleWiring;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class KeywordLoader implements IKeywordLoader {

    private static final Logger logger = LoggerFactory.getLogger(KeywordLoader.class);

    private String keywordPlatform;

    private ClassLoader classLoader;

    public KeywordLoader(String keywordPlatform) {
        this.keywordPlatform = keywordPlatform;
    }

    private ClassLoader fetchClassLoader() {
        if (null == classLoader) {
            try {
                Bundle targetBundle = fetchBundle();
                if (targetBundle != null) {
                    BundleWiring wiring = targetBundle.adapt(BundleWiring.class);
                    this.classLoader = wiring.getClassLoader();
                }
            } catch (Throwable e) {
                logger.debug("Cannot load bundle, fallback to loading from class | keywordPlatform = {}", keywordPlatform);
                final String className = getClassName();
                try {
                    this.classLoader = Class.forName(className).getClassLoader();
                } catch (Exception ex) {
                    logger.error("Cannot class loader | keywordPlatform = {} | className = {}", keywordPlatform, className, ex);
                }
            }
        }

        return this.classLoader;
    }
    
    private String getClassName() {
        if (StringUtils.equals(CoreConstants.PLATFORM_BUILT_IN, keywordPlatform)) {
            return "com.kms.katalon.core.keyword.BuiltinKeywords";
        } else if (StringUtils.equals(CoreConstants.PLATFORM_WEB, keywordPlatform)) {
            return "com.kms.katalon.core.webui.keyword.WebUiBuiltInKeywords";
        } else if (StringUtils.equals(CoreConstants.PLATFORM_MOBILE, keywordPlatform)) {
            return "com.kms.katalon.core.mobile.keyword.MobileBuiltInKeywords";
        } else if (StringUtils.equals(CoreConstants.PLATFORM_WEB_SERVICE, keywordPlatform)) {
            return "com.kms.katalon.core.webservice.keyword.WSBuiltInKeywords";
        } else if (StringUtils.equals(CoreConstants.PLATFORM_WINDOWS, keywordPlatform)) {
            return "com.kms.katalon.core.windows.keyword.WindowsBuiltinKeywords";
        } else {
            throw new IllegalArgumentException("Unsupported keyword platform: " + keywordPlatform);
        }
    }

    private Bundle fetchBundle() {
        String bundleName = getBundleName();
        return Platform.getBundle(bundleName);
    }

    private String getBundleName() {
        if (StringUtils.equals(CoreConstants.PLATFORM_BUILT_IN, keywordPlatform)) {
            return "com.kms.katalon.core";
        } else if (StringUtils.equals(CoreConstants.PLATFORM_WEB, keywordPlatform)) {
            return "com.kms.katalon.core.webui";
        } else if (StringUtils.equals(CoreConstants.PLATFORM_MOBILE, keywordPlatform)) {
            return "com.kms.katalon.core.mobile";
        } else if (StringUtils.equals(CoreConstants.PLATFORM_WEB_SERVICE, keywordPlatform)) {
            return "com.kms.katalon.core.webservice";
        } else if (StringUtils.equals(CoreConstants.PLATFORM_WINDOWS, keywordPlatform)) {
            return "com.kms.katalon.core.windows";
        } else {
            throw new IllegalArgumentException("Unsupported keyword platform: " + keywordPlatform);
        }
    }

    public List<Class<?>> loadAllClass(String... packageNames) {
        List<Class<?>> cls = new ArrayList<Class<?>>();
        for (String packageName : packageNames) {
            cls.addAll(getAllClassInPackage(packageName));
        }

        return cls;
    }

    public List<Class<?>> loadAllClass(Class<?> aClass) {
        return getAllClassInPackage(aClass.getPackage().getName());
    }

    private List<Class<?>> getAllClassInPackage(String packageName) {
        try {
            ArrayList<Class<?>> classes = new ArrayList<>();
            ClassLoader cld = fetchClassLoader();
            Enumeration<URL> urls = cld.getResources(packageName.replace('.', '/'));
            while (urls.hasMoreElements()) {
                URL url = urls.nextElement();
                if (url.getProtocol().contains("jar")) {
                    String file = url.getFile();
                    file = file.substring(5, file.indexOf("!"));
                    return getAllClassInPackageJar(file, packageName);
                } else if ("bundleresource".equalsIgnoreCase(url.getProtocol())) {
                    return getAllClassInBundleResource(url);
                }

                String path = URLDecoder.decode(url.getFile(), StandardCharsets.UTF_8.name());
                if (path.startsWith("/") && path.contains(":")) {
                    path = path.substring(1);
                }

                File f = new File(path);
                if (!f.isDirectory()) {
                    continue;
                }

                for (String fileName : f.list()) {
                    if (!fileName.endsWith(".class")) {
                        continue;
                    }

                    classes.add(cld.loadClass((packageName + "." + fileName).replaceAll("\\.class", "")));
                }
            }

            return classes;
        } catch (Exception ex) {
            return Collections.emptyList();
        }
    }

    private List<Class<?>> getAllClassInPackageJar(String jarFile, String packageName) throws Exception {
        JarFile jar = null;
        try {
            jar = new JarFile(URLDecoder.decode(jarFile, StandardCharsets.UTF_8.name()));
            Pattern pat = Pattern.compile(packageName.replace('.', '/') + "/[^/]+");
            ArrayList<Class<?>> classes = new ArrayList<Class<?>>();
            ClassLoader cld = fetchClassLoader();
            Enumeration<JarEntry> entries = jar.entries();
            while (entries.hasMoreElements()) {
                String name = entries.nextElement().getName();
                if (pat.matcher(name).matches()) {
                    classes.add(cld.loadClass(name.replace('/', '.').replaceAll("\\.class", "")));
                }
            }

            return classes;
        } catch (Exception ex) {
            return Collections.emptyList();
        } finally {
            if (jar != null) {
                jar.close();
            }
        }
    }

    private List<Class<?>> getAllClassInBundleResource(URL url) throws Exception {
        Bundle bundle = fetchBundle();
        if (bundle == null) {
            throw new IllegalStateException("Cannot resolve current bundle");
        }

        BundleWiring wiring = bundle.adapt(BundleWiring.class);
        if (wiring == null) {
            return Collections.emptyList();
        }

        // 3. Extract path (e.g. /com/kms/katalon/core/mobile/keyword/builtin/)
        String path = url.getPath();
        if (path.startsWith("/")) {
            path = path.substring(1);
        }

        // 4. List all .class entries in that path
        Collection<String> resources = wiring.listResources(path, "*.class", BundleWiring.LISTRESOURCES_LOCAL);
        if (resources == null) {
            return Collections.emptyList();
        }

        ClassLoader cl = wiring.getClassLoader();
        return resources.stream().map(r -> r.replace('/', '.').replaceAll("\\.class$", "")).map(name -> {
            try {
                return cl.loadClass(name);
            } catch (ClassNotFoundException e) {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }
}
