package com.kms.katalon.core.cucumber.keyword

import io.cucumber.core.plugin.KHtmlFormatter
import io.cucumber.core.plugin.KUnitFormatter
import java.util.Arrays;
import java.text.MessageFormat;

import org.apache.commons.lang3.StringUtils
import org.junit.runner.Computer;
import org.junit.runner.JUnitCore;
import org.junit.runner.Result;
import org.junit.runner.notification.Failure

import com.kms.katalon.core.annotation.Keyword;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.cucumber.keyword.internal.CucumberRunnerResultImpl
import com.kms.katalon.core.cucumber.keyword.internal.CucumberTargetType
import com.kms.katalon.core.keyword.BuiltinKeywords;
import com.kms.katalon.core.keyword.internal.KeywordMain;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.model.FailureHandling;
import com.kms.katalon.core.model.RunningMode

import io.cucumber.core.cli.Main
import groovy.transform.CompileStatic
import io.cucumber.core.plugin.CucumberReporter

public class CucumberBuiltinKeywords extends BuiltinKeywords {
    
    private static final String CUCUMBER_PLATFORM_NAME = "cucumber";
    private static final String STATUS_PASSED = "passed";
    private static final String STATUS_FAILED = "failed";

    /**
     * Key for the failed test file path in cucumberRunOptions map
     */
    private static final String FAILED_TEST_FILE_PATH_KEY = "failedTestFilePath";

    private static final KeywordLogger logger = KeywordLogger.getInstance(CucumberBuiltinKeywords.class);

    public static List GLUE = [""]

    /**
     * Builds the Cucumber CLI arguments with glue paths and plugins
     */
    private static String[] buildCucumberArguments(String featurePath, String reportDir, String rerunFilePath, String[] tags) {
        RunningMode runningMode = RunConfiguration.getRunningMode()
        String[] argv = []
        
        // Add glue paths
        GLUE.each { g -> argv += ['--glue', g] }
        
        // Add feature path
        argv += [featurePath]
        
        // Add standard plugins
        argv += buildPluginArguments(reportDir, rerunFilePath)
        
        // Add tags if provided
        if (tags != null) {
            tags.each { tag ->
                argv += ["--tags", tag]
            }
        }
        
        // Add monochrome for console mode
        if (runningMode == RunningMode.CONSOLE) {
            argv += ["--monochrome"]
        }
        
        return argv
    }

    /**
     * Builds plugin arguments for Cucumber reports
     */
    private static String[] buildPluginArguments(String reportDir, String rerunFilePath) {
        List<String> plugins = []

        // Only add rerun plugin if rerunFilePath is provided
        if (StringUtils.isNotEmpty(rerunFilePath)) {
            plugins += ["--plugin", "rerun:${rerunFilePath}"]
        }

        plugins += [
            "--plugin", "${KHtmlFormatter.class.getName()}:${reportDir}/cucumber.html",
            "--plugin", "${KUnitFormatter.class.getName()}:${reportDir}/cucumber.xml",
            "--plugin", "${CucumberReporter.class.getName()}:${reportDir}/k-cucumber.json"
        ]
        
        return plugins as String[]
    }

    /**
     * Executes Cucumber tests with optional rerun file generation.
     * <p>
     * This method handles the complete execution flow including:
     * <ul>
     * <li>Execution of the specified feature file or folder</li>
     * <li>Optional generation of rerun file for failed scenarios (if failedTestFilePath is provided)</li>
     * <li>Generation of timestamped reports</li>
     * <li>Logging of execution progress and results</li>
     * </ul>
     * 
     * @param targetPath the relative path to the feature file or folder to execute
     * @param tags optional array of Cucumber tags to filter scenarios (e.g., ['@smoke', 'not @slow'])
     * @param cucumberRunOptions optional map containing execution options. Supported keys:
     *        - failedTestFilePath: relative path from project folder where failed scenarios will be written (optional)
     * @param targetType the type of target being executed (FEATURE_FILE or FEATURE_FOLDER)
     * @param flowControl controls the running flow and failure handling behavior
     * @return a CucumberRunnerResultImpl containing the execution status and report directory path
     * @throws IllegalArgumentException if targetPath is null or empty (via validatePath)
     * @since 11.0.0
     */
    private static CucumberRunnerResultImpl executeCucumber(
            String targetPath,
            String[] tags,
            Map cucumberRunOptions,
            CucumberTargetType targetType,
            FailureHandling flowControl) {

        String reportDir = generateCucumberReportDir()
        String projectDir = RunConfiguration.getProjectDir()
        String fullPath =  targetPath.startsWith('@') ? targetPath : projectDir + "/" + targetPath

        String rerunFilePath = null
        if (cucumberRunOptions != null && cucumberRunOptions.containsKey(FAILED_TEST_FILE_PATH_KEY)) {
            String failedTestFilePath = cucumberRunOptions[FAILED_TEST_FILE_PATH_KEY]
            if (!StringUtils.isEmpty(failedTestFilePath)) {
                rerunFilePath = projectDir + "/" + failedTestFilePath
            }
        }

        logger.logInfo(MessageFormat.format(
            "Starting run keyword run{0}: ''{1}'' and extract report to folder: ''{2}''...",
            targetType.getDisplayName(), targetPath, reportDir))
        logger.logDebug('Glue: ' + GLUE)
        
        if (rerunFilePath != null) {
            logger.logInfo(MessageFormat.format("Failed scenarios will be written to: ''{0}''", rerunFilePath))
        }

        String[] argv = buildCucumberArguments(fullPath, reportDir, rerunFilePath, tags)
        logCucumberArguments(argv)

        boolean runSuccess = executeCucumberCLI(argv)
        CucumberRunnerResultImpl cucumberResult = new CucumberRunnerResultImpl(
            runSuccess ? STATUS_PASSED : STATUS_FAILED, reportDir)
        
        logFinalResult(runSuccess, targetPath, targetType, flowControl)
        
        return cucumberResult
    }

    /**
     * Executes Cucumber CLI and returns success status
     */
    private static boolean executeCucumberCLI(String[] argv) {
        return Main.run(argv, CucumberBuiltinKeywords.class.getClassLoader()) == 0
    }

    /**
     * Generates a timestamped report directory path
     */
    private static String generateCucumberReportDir() {
        return RunConfiguration.getReportFolder() + "/cucumber_report/" + System.currentTimeMillis()
    }

    /**
     * Logs Cucumber arguments for debugging
     */
    private static void logCucumberArguments(String[] argv) {
        logger.logDebug("Cucumber argv: " + Arrays.toString(argv))
    }

    /**
     * Logs the final result and handles failure if needed
     */
    private static void logFinalResult(boolean runSuccess, String targetPath, CucumberTargetType targetType, FailureHandling flowControl) {
        if (runSuccess) {
            String message = targetType.isFolder() ?
                MessageFormat.format("All feature files in ''{0}'' were passed", targetPath) :
                MessageFormat.format("Feature file: ''{0}'' was passed", targetPath)
            logger.logPassed(message)
        } else {
            String message = targetType.isFolder() ?
                MessageFormat.format("Run feature folder ''{0}'' failed", targetPath) :
                MessageFormat.format("Feature file ''{0}'' was failed", targetPath)
            KeywordMain.stepFailed(message, flowControl)
        }
    }

    /**
     * Validates that the path parameter is not null or empty
     */
    private static void validatePath(String path, String paramName) {
        if (StringUtils.isEmpty(path)) {
            throw new IllegalArgumentException(MessageFormat.format("{0} param must not be null or empty", paramName))
        }
    }


    // ===========================
    // Feature File Methods
    // ===========================

    /**
     * Runs the given Feature file with <code>featureId</code> by invoking
     * {@link cucumber.api.cli.Main#run(String[], ClassLoader)}. 
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p>
     * 
     * Examples:
     * <ul>
     * <li>Example #1: Run a single feature file
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFile('Include/features/New Feature File.feature', FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * <li>Example #2: Run a single feature file and the step definitions is in pre-defined packages
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFile('Include/features/New Feature File.feature', FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param relativeFilePath relativeFilePath of Feature file
     * @param flowControl an instance {@link FailureHandling} that controls the running flow
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 5.7
     * @see CucumberRunnerResult
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFile(String relativeFilePath, FailureHandling flowControl) {
        return runFeatureFile(relativeFilePath, null, flowControl)
    }

    /**
     * Runs the given Feature file with optional rerun file generation.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p>
     * 
     * Examples:
     * <ul>
     * <li>Example #1: Run a single feature file with rerun file generation
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFile('Include/features/operations/New Feature File 1.feature', 
     *     [failedTestFilePath: 'failed-new-testcase.txt'], 
     *     FailureHandling.CONTINUE_ON_FAILURE)
     * </pre>
     * </li>
     * <li>Example #2: Rerun failed scenarios from a previous execution
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * // Use @ prefix to run from rerun file
     * CucumberKW.runFeatureFile('@failed-new-testcase.txt', 
     *     [failedTestFilePath: 'failed-new-testcase-2.txt'])
     * </pre>
     * </li>
     * <li>Example #3: Run with custom glue path
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFile('Include/features/New Feature File.feature', 
     *     [failedTestFilePath: 'Include/features/rerun.txt'], 
     *     FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param relativeFilePath relativeFilePath of Feature file, or @path/to/rerun.txt to rerun failed scenarios
     * @param cucumberRunOptions optional map containing execution options (e.g., [failedTestFilePath: 'path/to/rerun.txt'])
     * @param flowControl an instance {@link FailureHandling} that controls the running flow
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 11.0.0
     * @see CucumberRunnerResult
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFile(String relativeFilePath, Map cucumberRunOptions, FailureHandling flowControl) {
        return KeywordMain.runKeyword(CUCUMBER_PLATFORM_NAME, {
            validatePath(relativeFilePath, "relativeFilePath")
            return executeCucumber(relativeFilePath, null, cucumberRunOptions, CucumberTargetType.FEATURE_FILE, flowControl)
        }, flowControl, "Keyword runFeatureFile was failed")
    }

    /**
     * Runs the given Feature file with <code>featureId</code> by invoking
     * {@link cucumber.api.cli.Main#run(String[], ClassLoader)}.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions
     * path easier.
     * </p>
     * Examples:
     * <ul>
     * <li>Example #1: Run a single feature file
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFile('Include/features/New Feature File.feature')
     * </pre>
     * </li>
     * <li>Example #2: Run a single feature file and the step definitions is in pre-defined packages
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFile('Include/features/New Feature File.feature')
     * </pre>
     * </li>
     * </ul>
     * 
     * @param relativeFilePath relativeFilePath of Feature file
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 5.7
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFile(String relativeFilePath) {
        return runFeatureFile(relativeFilePath, RunConfiguration.getDefaultFailureHandling())
    }

    /**
     * Runs the given Feature file with optional rerun file generation.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * Examples:
     * <ul>
     * <li>Example #1: Run a single feature file with rerun file generation
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFile('Include/features/operations/New Feature File 1.feature', 
     *     [failedTestFilePath: 'failed-new-testcase.txt'])
     * </pre>
     * </li>
     * <li>Example #2: Rerun failed scenarios from a previous execution
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * // Use @ prefix to run from rerun file
     * CucumberKW.runFeatureFile('@failed-new-testcase.txt', 
     *     [failedTestFilePath: 'failed-new-testcase-2.txt'])
     * </pre>
     * </li>
     * <li>Example #3: Run a single feature file without rerun file
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFile('Include/features/New Feature File.feature', null)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param relativeFilePath relativeFilePath of Feature file, or @path/to/rerun.txt to rerun failed scenarios
     * @param cucumberRunOptions optional map containing execution options (e.g., [failedTestFilePath: 'path/to/rerun.txt'])
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 11.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFile(String relativeFilePath, Map cucumberRunOptions) {
        return runFeatureFile(relativeFilePath, cucumberRunOptions, RunConfiguration.getDefaultFailureHandling())
    }

    // ===========================
    // Feature File With Tags Methods
    // ===========================

    /**
     * Runs the given Feature file with <code>featureId</code> by invoking
     * {@link cucumber.api.cli.Main#run(String[], ClassLoader)}.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p>
     * 
     * Examples:
     * <ul>
     * <li>Example #1: Run a single feature file with selected tags
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFileWithTags('Include/features/New Feature File.feature', ['@foo', 'not @bar'] as [], FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * <li>Example #2: Run a single feature file with selected tags and the step definitions is in pre-defined packages
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFileWithTags('Include/features/New Feature File.feature', ['@foo', 'not @bar'] as [], FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param relativeFilePath relativeFilePath of the feature file
     * @param tags what tags in the features should be executed. Eg: ['@foo', 'not @bar']: Select all foo tags but not bar tags
     * @param flowControl an instance {@link FailureHandling} that controls the running flow
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 7.0.0
     * @see CucumberRunnerResult
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFileWithTags(String relativeFilePath, String[] tags, FailureHandling flowControl) {
        return runFeatureFileWithTags(relativeFilePath, tags, null, flowControl)
    }

    /**
     * Runs the given Feature file with tags and optional rerun file generation.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p>
     * 
     * Examples:
     * <ul>
     * <li>Example #1: Run a single feature file with tags and rerun file generation
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFileWithTags('Include/features/operations/New Feature File 1.feature', 
     *     ['@smoke', 'not @slow'] as String[], 
     *     [failedTestFilePath: 'failed-new-testcase.txt'], 
     *     FailureHandling.CONTINUE_ON_FAILURE)
     * </pre>
     * </li>
     * <li>Example #2: Rerun failed scenarios with tags
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * // Use @ prefix to run from rerun file
     * CucumberKW.runFeatureFileWithTags('@failed-new-testcase.txt', 
     *     ['@smoke'] as String[], 
     *     [failedTestFilePath: 'failed-new-testcase-2.txt'])
     * </pre>
     * </li>
     * <li>Example #3: Run with tags without rerun file
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFileWithTags('Include/features/New Feature File.feature', 
     *     ['@foo', 'not @bar'] as String[], 
     *     null, 
     *     FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param relativeFilePath relativeFilePath of the feature file, or @path/to/rerun.txt to rerun failed scenarios
     * @param tags what tags in the features should be executed. Eg: ['@foo', 'not @bar']: Select all foo tags but not bar tags
     * @param cucumberRunOptions optional map containing execution options (e.g., [failedTestFilePath: 'path/to/rerun.txt'])
     * @param flowControl an instance {@link FailureHandling} that controls the running flow
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 11.0.0
     * @see CucumberRunnerResult
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFileWithTags(String relativeFilePath, String[] tags, Map cucumberRunOptions, FailureHandling flowControl) {
        return KeywordMain.runKeyword(CUCUMBER_PLATFORM_NAME, {
            validatePath(relativeFilePath, "relativeFilePath")
            return executeCucumber(relativeFilePath, tags, cucumberRunOptions, CucumberTargetType.FEATURE_FILE, flowControl)
        }, flowControl, "Keyword runFeatureFileWithTags was failed")
    }

    /**
     * Runs the given Feature file with <code>featureId</code> by invoking
     * {@link cucumber.api.cli.Main#run(String[], ClassLoader)}.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions
     * path easier.
     * 
     * <ul>
     * <li>Example #1: Run a single feature file with selected tags
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFileWithTags('Include/features/New Feature File.feature', ["@foo", "not @bar"] as [])
     * </pre>
     * </li>
     * <li>Example #2: Run a single feature file with selected tags and the step definitions is in pre-defined packages
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFileWithTags('Include/features/New Feature File.feature', ["@foo", "not @bar"] as [])
     * </pre>
     * </li>
     * </ul>
     * 
     * @param relativeFilePath relativeFilePath of Feature file
     * @param tags what tags in the features should be executed. Eg: ['@foo', 'not @bar']: Select all foo tags but not bar tags
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 7.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFileWithTags(String relativeFilePath, String[] tags) {
        return runFeatureFileWithTags(relativeFilePath, tags, RunConfiguration.getDefaultFailureHandling())
    }

    /**
     * Runs the given Feature file with tags and optional rerun file generation.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * 
     * <ul>
     * <li>Example #1: Run a single feature file with selected tags and rerun file generation
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFileWithTags('Include/features/operations/New Feature File 1.feature', 
     *     ['@smoke', 'not @slow'] as String[], 
     *     [failedTestFilePath: 'failed-new-testcase.txt'])
     * </pre>
     * </li>
     * <li>Example #2: Rerun failed scenarios with tags
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFileWithTags('@failed-new-testcase.txt', 
     *     ['@smoke'] as String[], 
     *     [failedTestFilePath: 'failed-new-testcase-2.txt'])
     * </pre>
     * </li>
     * </ul>
     * 
     * @param relativeFilePath relativeFilePath of Feature file, or @path/to/rerun.txt to rerun failed scenarios
     * @param tags what tags in the features should be executed. Eg: ['@foo', 'not @bar']: Select all foo tags but not bar tags
     * @param cucumberRunOptions optional map containing execution options (e.g., [failedTestFilePath: 'path/to/rerun.txt'])
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 11.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFileWithTags(String relativeFilePath, String[] tags, Map cucumberRunOptions) {
        return runFeatureFileWithTags(relativeFilePath, tags, cucumberRunOptions, RunConfiguration.getDefaultFailureHandling())
    }


    // ===========================
    // Feature Folder Methods
    // ===========================

    /**
     * Runs the given Feature folder and its nested sub-folder with <code>folderRelativePath</code>
     * by invoking {@link cucumber.api.cli.Main#run(String[], ClassLoader)}.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p> 
     * Examples:
     * <ul>
     * <li>Example #1: Run all features file in the specified folder
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFolder('Include/features', FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * <li>Example #2: Run all features file in the specified folder and the step definitions is in pre-defined packages
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFolder('Include/features', FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param folderRelativePath folder relative path that starts from the current project location
     * @param flowControl an instance {@link FailureHandling} that controls the running flow
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 5.7
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFolder(String folderRelativePath, FailureHandling flowControl) {
        return runFeatureFolder(folderRelativePath, null, flowControl)
    }

    /**
     * Runs the given Feature folder and its nested sub-folder with optional rerun file generation.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p> 
     * Examples:
     * <ul>
     * <li>Example #1: Run all features file in the specified folder with rerun file generation
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFolder('Include/features', 
     *     [failedTestFilePath: 'failed-features.txt'], 
     *     FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * <li>Example #2: Rerun failed scenarios from previous folder execution
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFolder('@failed-features.txt', 
     *     [failedTestFilePath: 'failed-features-2.txt'])
     * </pre>
     * </li>
     * <li>Example #3: Run all features file without rerun file
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFolder('Include/features', null, FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param folderRelativePath folder relative path that starts from the current project location, or @path/to/rerun.txt to rerun failed scenarios
     * @param cucumberRunOptions optional map containing execution options (e.g., [failedTestFilePath: 'path/to/rerun.txt'])
     * @param flowControl an instance {@link FailureHandling} that controls the running flow
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 11.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFolder(String folderRelativePath, Map cucumberRunOptions, FailureHandling flowControl) {
        return KeywordMain.runKeyword(CUCUMBER_PLATFORM_NAME, {
            validatePath(folderRelativePath, "folderRelativePath")
            return executeCucumber(folderRelativePath, null, cucumberRunOptions, CucumberTargetType.FEATURE_FOLDER, flowControl)
        }, flowControl, "Keyword runFeatureFolder was failed")
    }

    /**
     * Runs the given Feature folder and its nested sub-folder with <code>folderRelativePath</code>
     * by invoking {@link cucumber.api.cli.Main#run(String[], ClassLoader)}.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * 
     * Examples</code>:
     * <ul>
     * <li>Example #1: Run all features file in the specified folder
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFolder('Include/features')
     * </pre>
     * </li>
     * <li>Example #2: Run all features file in the specified folder and the step definitions is in pre-defined packages
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFolder('Include/features')
     * </pre>
     * </li>
     * </ul>
     * 
     * @param folderRelativePath folder relative path that starts from current project location
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 5.7
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFolder(String folderRelativePath) {
        return runFeatureFolder(folderRelativePath, RunConfiguration.getDefaultFailureHandling())
    }

    /**
     * Runs the given Feature folder with optional rerun file generation.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * 
     * Examples:
     * <ul>
     * <li>Example #1: Run all features file in the specified folder with rerun file generation
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFolder('Include/features', 
     *     [failedTestFilePath: 'failed-features.txt'])
     * </pre>
     * </li>
     * <li>Example #2: Rerun failed scenarios from previous folder execution
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFolder('@failed-features.txt', 
     *     [failedTestFilePath: 'failed-features-2.txt'])
     * </pre>
     * </li>
     * </ul>
     * 
     * @param folderRelativePath folder relative path that starts from current project location, or @path/to/rerun.txt to rerun failed scenarios
     * @param cucumberRunOptions optional map containing execution options (e.g., [failedTestFilePath: 'path/to/rerun.txt'])
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 11.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFolder(String folderRelativePath, Map cucumberRunOptions) {
        return runFeatureFolder(folderRelativePath, cucumberRunOptions, RunConfiguration.getDefaultFailureHandling())
    }

    // ===========================
    // Feature Folder With Tags Methods
    // ===========================

    /**
     * Runs the given Feature folder and its nested sub-folder with <code>folderRelativePath</code>
     * by invoking {@link cucumber.api.cli.Main#run(String[], ClassLoader)}.
     * </p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p>
     * <ul>
     * <li>Example #1: Run a single feature file with selected tags
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFolderWithTags('Include/features', ["@foo", "not @bar"] as [], FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * <li>Example #2: Run a single feature file with selected tags and the step definitions is in pre-defined packages
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFolderWithTags('Include/features', ["@foo", "not @bar"] as [], FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param folderRelativePath folder relative path that starts from the current project location
     * @param tags what tags in the features should be executed. Eg: ['@foo', 'not @bar']: Select all foo tags but not bar tags
     * @param flowControl an instance {@link FailureHandling} that controls the running flow
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 7.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFolderWithTags(String folderRelativePath, String[] tags, FailureHandling flowControl) {
        return runFeatureFolderWithTags(folderRelativePath, tags, null, flowControl)
    }

    /**
     * Runs the given Feature folder and its nested sub-folder with tags and optional rerun file generation.
     * </p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p>
     * <ul>
     * <li>Example #1: Run feature folder with tags and rerun file generation
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFolderWithTags('Include/features', 
     *     ['@smoke', 'not @slow'] as String[], 
     *     [failedTestFilePath: 'failed-features.txt'], 
     *     FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * <li>Example #2: Rerun failed scenarios from folder with tags
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFolderWithTags('@failed-features.txt', 
     *     ['@smoke'] as String[], 
     *     [failedTestFilePath: 'failed-features-2.txt'])
     * </pre>
     * </li>
     * <li>Example #3: Run feature folder with tags without rerun file
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * import com.kms.katalon.core.model.FailureHandling
     * 
     * CucumberKW.runFeatureFolderWithTags('Include/features', 
     *     ['@foo', 'not @bar'] as String[], 
     *     null, 
     *     FailureHandling.STOP_ON_FAILURE)
     * </pre>
     * </li>
     * </ul>
     * 
     * @param folderRelativePath folder relative path that starts from the current project location, or @path/to/rerun.txt to rerun failed scenarios
     * @param tags what tags in the features should be executed. Eg: ['@foo', 'not @bar']: Select all foo tags but not bar tags
     * @param cucumberRunOptions optional map containing execution options (e.g., [failedTestFilePath: 'path/to/rerun.txt'])
     * @param flowControl an instance {@link FailureHandling} that controls the running flow
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 11.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFolderWithTags(String folderRelativePath, String[] tags, Map cucumberRunOptions, FailureHandling flowControl) {
        return KeywordMain.runKeyword(CUCUMBER_PLATFORM_NAME, {
            validatePath(folderRelativePath, "folderRelativePath")
            return executeCucumber(folderRelativePath, tags, cucumberRunOptions, CucumberTargetType.FEATURE_FOLDER, flowControl)
        }, flowControl, "Keyword runFeatureFolderWithTags was failed")
    }

    /**
     * Runs the given Feature folder and its nested sub-folder with <code>folderRelativePath</code>
     * by invoking {@link cucumber.api.cli.Main#run(String[], ClassLoader)}
     *</p>
     * The generated reports will be extracted in the current report folder with the following path: <code>&lt;report_folder&gt;/cucumber_report/&lt;current_time_stamp&gt;<code>
     * </p>
     * <ul>
     * <li>Example #1: Run a single feature file with selected tags
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFolderWithTags('Include/features', ["@foo", "not @bar"] as [])
     * </pre>
     * </li>
     * <li>Example #2: Run a single feature file with selected tags and the step definitions is in pre-defined packages
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.GLUE = ['mypackage1', 'mypackage2']
     * CucumberKW.runFeatureFolderWithTags('Include/features', ["@foo", "not @bar"] as [])
     * </pre>
     * </li>
     * </ul>
     * 
     * @param folderRelativePath folder relative path that starts from current project location
     * @param tags what tags in the features should be executed. Eg: ['@foo', 'not @bar']: Select all foo tags but not bar tags
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 7.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFolderWithTags(String folderRelativePath, String[] tags) {
        return runFeatureFolderWithTags(folderRelativePath, tags, RunConfiguration.getDefaultFailureHandling())
    }

    /**
     * Runs the given Feature folder with tags and optional rerun file generation.
     * If your step definitions are in packages, you can set glue path to help system can detect your step definitions path easier.
     * </p>
     * 
     * Examples:
     * <ul>
     * <li>Example #1: Run feature folder with tags and rerun file generation
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFolderWithTags('Include/features', 
     *     ['@smoke', 'not @slow'] as String[], 
     *     [failedTestFilePath: 'failed-features.txt'])
     * </pre>
     * </li>
     * <li>Example #2: Rerun failed scenarios from folder with tags
     * <pre>
     * import com.kms.katalon.core.cucumber.keyword.CucumberBuiltinKeywords as CucumberKW
     * 
     * CucumberKW.runFeatureFolderWithTags('@failed-features.txt', 
     *     ['@smoke'] as String[], 
     *     [failedTestFilePath: 'failed-features-2.txt'])
     * </pre>
     * </li>
     * </ul>
     * 
     * @param folderRelativePath folder relative path that starts from current project location, or @path/to/rerun.txt to rerun failed scenarios
     * @param tags what tags in the features should be executed. Eg: ['@foo', 'not @bar']: Select all foo tags but not bar tags
     * @param cucumberRunOptions optional map containing execution options (e.g., [failedTestFilePath: 'path/to/rerun.txt'])
     * @return an instance of {@link CucumberRunnerResult} that includes status of keyword and report folder location.
     * @since 11.0.0
     */
    @Keyword
    public static CucumberRunnerResult runFeatureFolderWithTags(String folderRelativePath, String[] tags, Map cucumberRunOptions) {
        return runFeatureFolderWithTags(folderRelativePath, tags, cucumberRunOptions, RunConfiguration.getDefaultFailureHandling())
    }

    /**
     * Runs the given <code>cucumberRunnerClass</code> that is annotated with {@link Cucumber} runner by invoke JUnit
     * runner.
     * 
     * @param cucumberRunnerClass
     * a class that is annotated with {@link Cucumber} runner.
     * <p>
     * Example of <code>cucumberRunnerClass</code>:
     * <ul>
     * <li>Example #1: Run all Feature files in <b>Include/features</b> Folder
     * 
     * <pre>
     * import org.junit.runner.RunWith;
     * import cucumber.api.CucumberOptions;
     * import cucumber.api.junit.Cucumber;
     * 
     * &#64;RunWith(Cucumber.class)
     * &#64;CucumberOptions(features = "Include/features", glue = "")
     * public class MyCucumberRunner {}
     * </pre>
     * 
     * </li>
     * 
     * <li>Example #2: Run all Feature files in a specified file/folder
     * 
     * <pre>
     * import org.junit.runner.RunWith;
     * import cucumber.api.CucumberOptions;
     * import cucumber.api.junit.Cucumber;
     * 
     * &#64;RunWith(Cucumber.class)
     * &#64;CucumberOptions(features = "Your_Folder_Or_File_Path", glue = "")
     * public class MyCucumberRunner {}
     * </pre>
     * 
     * </li>
     * 
     * </li>
     * <li>Example #3: Run all Feature files in a specified file/folder, generate JUnit Cucumber report with XML pretty
     * format,
     * and copy to a specified folder
     * 
     * <pre>
     * import org.junit.runner.RunWith;
     * import cucumber.api.CucumberOptions;
     * import cucumber.api.junit.Cucumber;
     * &#64;RunWith(Cucumber.class)
     * &#64;CucumberOptions(features="Your_Folder_Path", glue="", plugin = ["pretty",
     *                      "junit:Folder_Name/cucumber.xml"])
     * public class MyCucumberRunner {
     * }
     * </pre>
     * 
     * </li>
     * 
     * </li>
     * <li>Example #4: Run all Feature files in a specified file/folder, generate multi Cucumber reports with XML,
     * HTML pretty format,
     * and copy to a specified folder
     * 
     * <pre>
     * import org.junit.runner.RunWith;
     * import cucumber.api.CucumberOptions;
     * import cucumber.api.junit.Cucumber;
     * &#64;RunWith(Cucumber.class)
     * &#64;CucumberOptions(features="Your_Folder_Path", glue="", plugin = ["pretty",
     *                      "junit:Folder_Name/cucumber.xml",
     *                      "html:Folder_Name"])
     * public class MyCucumberRunner {
     * }
     * </pre>
     * 
     * </li>
     * @param flowControl
     * an instance {@link FailureHandling} that controls the running flow
     * @return
     * an instance of {@link CucumberRunnerResult} that includes status of keyword and JUnit Runner result.
     * @since 5.7
     */
    @Keyword
    public static CucumberRunnerResult runWithCucumberRunner(Class cucumberRunnerClass, FailureHandling flowControl) {
        return KeywordMain.runKeyword(CUCUMBER_PLATFORM_NAME, {
            JUnitCore core = new JUnitCore();
            Computer computer = new Computer();
            Result result = core.run(computer, cucumberRunnerClass);
            boolean runSuccess = result.wasSuccessful();
            CucumberRunnerResultImpl cucumberResult = new CucumberRunnerResultImpl(
                    runSuccess ? 'passed' : 'failed', '', result)
            if (runSuccess) {
                logger.logPassed(MessageFormat.format("Run with ''{0}'' was passed", cucumberRunnerClass.getName()));
            } else {
                List failuresDescriptions = []
                for (Failure failure : result.getFailures()) {
                    failuresDescriptions.add(failure.getMessage())
                }
                KeywordMain.stepFailed(
                        MessageFormat.format("These following reason:\n {0}", failuresDescriptions), flowControl);
            }
            return cucumberResult;
        }, flowControl, "Keyword runWithCucumberRunner was failed");
    }

    /**
     * Runs the given <code>cucumberRunnerClass</code> that is annotated with {@link Cucumber} runner by invoke JUnit
     * runner.
     * 
     * @param cucumberRunnerClass
     * a class that is annotated with {@link Cucumber} runner.
     * <p>
     * Example of <code>cucumberRunnerClass</code>:
     * <ul>
     * <li>Example #1: Run all Feature files in <b>Include/features</b> Folder
     * 
     * <pre>
     * import org.junit.runner.RunWith;
     * import cucumber.api.CucumberOptions;
     * import cucumber.api.junit.Cucumber;
     * 
     * &#64;RunWith(Cucumber.class)
     * &#64;CucumberOptions(features = "Include/features", glue = "")
     * public class MyCucumberRunner {}
     * </pre>
     * 
     * </li>
     * 
     * <li>Example #2: Run all Feature files in a specified file/folder
     * 
     * <pre>
     * import org.junit.runner.RunWith;
     * import cucumber.api.CucumberOptions;
     * import cucumber.api.junit.Cucumber;
     * 
     * &#64;RunWith(Cucumber.class)
     * &#64;CucumberOptions(features = "Your_Folder_Or_File_Path", glue = "")
     * public class MyCucumberRunner {}
     * </pre>
     * 
     * </li>
     * 
     * </li>
     * <li>Example #3: Run all Feature files in a specified file/folder, generate JUnit Cucumber report with XML pretty
     * format,
     * and copy to a specified folder
     * 
     * <pre>
     * import org.junit.runner.RunWith;
     * import cucumber.api.CucumberOptions;
     * import cucumber.api.junit.Cucumber;
     * &#64;RunWith(Cucumber.class)
     * &#64;CucumberOptions(features="Your_Folder_Path", glue="", plugin = ["pretty",
     *                      "junit:Folder_Name/cucumber.xml"])
     * public class MyCucumberRunner {
     * }
     * </pre>
     * 
     * </li>
     * 
     * </li>
     * <li>Example #4: Run all Feature files in a specified file/folder, generate multi Cucumber reports with XML, JSON,
     * HTML pretty format,
     * and copy to a specified folder
     * 
     * <pre>
     * import org.junit.runner.RunWith;
     * import cucumber.api.CucumberOptions;
     * import cucumber.api.junit.Cucumber;
     * &#64;RunWith(Cucumber.class)
     * &#64;CucumberOptions(features="Your_Folder_Path", glue="", plugin = ["pretty",
     *                      "junit:Folder_Name/cucumber.xml",
     *                      "html:Folder_Name"])
     * public class MyCucumberRunner {
     * }
     * </pre>
     * 
     * </li>
     * an instance of {@link CucumberRunnerResult} that includes status of keyword and JUnit Runner result.
     * @since 5.7
     */
    @Keyword
    public static CucumberRunnerResult runWithCucumberRunner(Class cucumberRunnerClass) {
        return runWithCucumberRunner(cucumberRunnerClass, RunConfiguration.getDefaultFailureHandling());
    }
}

