package io.cucumber.core.plugin;

import static io.cucumber.core.exception.ExceptionUtils.printStackTrace;
import static io.cucumber.core.plugin.TestSourcesModel.getBackgroundForTestCase;
import static java.util.Collections.singletonList;
import static java.util.Locale.ROOT;
import static java.util.stream.Collectors.toList;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.net.URI;
import java.text.MessageFormat;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.UUID;

import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.logging.ErrorCollector;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.logging.LogLevel;
import com.kms.katalon.core.util.internal.ExceptionsUtil;

import io.cucumber.messages.types.Background;
import io.cucumber.messages.types.Feature;
import io.cucumber.messages.types.Scenario;
import io.cucumber.messages.types.Step;
import io.cucumber.plugin.EventListener;
import io.cucumber.plugin.event.*;

public class CucumberReporter implements EventListener {

    private static final String before = "before";

    private static final String after = "after";

    private final List<Map<String, Object>> featureMaps = new ArrayList<>();

    private final Map<String, Object> currentBeforeStepHookList = new HashMap<>();

    private final Writer writer;

    private final TestSourcesModel testSources = new TestSourcesModel();

    private URI currentFeatureFile;

    private List<Map<String, Object>> currentElementsList;

    private Map<String, Object> currentElementMap;

    private Map<String, Object> currentTestCaseMap;

    private List<Map<String, Object>> currentStepsList;

    private Map<String, Object> currentStepOrHookMap;

    private Map<String, Object> currentFeatureMap;

    private static final KeywordLogger logger = KeywordLogger.getInstance(CucumberReporter.class);

    public CucumberReporter(OutputStream out) {
        this.writer = new UTF8OutputStreamWriter(out);
    }

    @Override
    public void setEventPublisher(EventPublisher publisher) {
        publisher.registerHandlerFor(TestSourceRead.class, this::handleTestSourceRead);
        publisher.registerHandlerFor(TestCaseStarted.class, this::handleTestCaseStarted);
        publisher.registerHandlerFor(TestStepStarted.class, this::handleTestStepStarted);
        publisher.registerHandlerFor(TestStepFinished.class, this::handleTestStepFinished);
        publisher.registerHandlerFor(WriteEvent.class, this::handleWrite);
        publisher.registerHandlerFor(EmbedEvent.class, this::handleEmbed);
        publisher.registerHandlerFor(TestCaseFinished.class, this::onTestCaseFinished);
        publisher.registerHandlerFor(TestRunFinished.class, this::finishReport);
    }

    private void handleTestSourceRead(TestSourceRead event) {
        testSources.addTestSourceReadEvent(event.getUri(), event);
    }

    @SuppressWarnings("unchecked")
    private void handleTestCaseStarted(TestCaseStarted event) {
        if (currentFeatureFile == null || !currentFeatureFile.equals(event.getTestCase().getUri())) {
            currentFeatureFile = event.getTestCase().getUri();
            Map<String, Object> currentFeatureMap = createFeatureMap(event.getTestCase());
            featureMaps.add(currentFeatureMap);
            currentElementsList = (List<Map<String, Object>>) currentFeatureMap.get("elements");
        }
        currentTestCaseMap = createTestCase(event);
        // For Katalon report
        String uuid = UUID.randomUUID().toString();
        String bddTestrunUUIDPropName = "BDD_TESTRUN_UUID";
        currentTestCaseMap.put(bddTestrunUUIDPropName, uuid);
        String name = getTestCaseName(event.getTestCase());
        Map<String, String> attributes = new HashMap<>();
        attributes.put(bddTestrunUUIDPropName, uuid);
        if (currentFeatureMap != null && currentFeatureMap.get("name") != null) {
            attributes.put("BDD_FEATURE_NAME", logger.maskSecureValues(currentFeatureMap.get("name").toString()));
        }
        if (currentTestCaseMap.get("line") != null) {
            attributes.put("BDD_TESTCASE_LINE", currentTestCaseMap.get("line").toString());
        }
        if (currentTestCaseMap.get("name") != null) {
            attributes.put("BDD_TESTCASE_NAME", logger.maskSecureValues(currentTestCaseMap.get("name").toString()));
        }
        if (currentTestCaseMap.get("name") != null) {
            attributes.put("BDD_TESTCASE_DESCRIPTION", logger.maskSecureValues(currentTestCaseMap.get("description").toString()));
        }
        if (currentTestCaseMap.get("name") != null) {
            attributes.put("BDD_TESTCASE_TYPE", currentTestCaseMap.get("type").toString());
        }
        logger.startTest(name, attributes, new Stack<KeywordLogger.KeywordStackElement>());
        // end
        if (testSources.hasBackground(currentFeatureFile, event.getTestCase().getLocation().getLine())) {
            currentElementMap = createBackground(event.getTestCase());
            // For Katalon report
            if (currentElementMap != null) {
                currentElementMap.put(bddTestrunUUIDPropName, uuid);
            }
            // end
            currentElementsList.add(currentElementMap);
        } else {
            currentElementMap = currentTestCaseMap;
        }
        currentElementsList.add(currentTestCaseMap);
        currentStepsList = (List<Map<String, Object>>) currentElementMap.get("steps");
    }

    @SuppressWarnings("unchecked")
    private void handleTestStepStarted(TestStepStarted event) {
        if (event.getTestStep() instanceof PickleStepTestStep) {
            PickleStepTestStep testStep = (PickleStepTestStep) event.getTestStep();
            if (isFirstStepAfterBackground(testStep)) {
                currentElementMap = currentTestCaseMap;
                currentStepsList = (List<Map<String, Object>>) currentElementMap.get("steps");
            }
            currentStepOrHookMap = createTestStep(testStep);
            // add beforeSteps list to current step
            if (currentBeforeStepHookList.containsKey(before)) {
                currentStepOrHookMap.put(before, currentBeforeStepHookList.get(before));
                currentBeforeStepHookList.clear();
            }
            // For Katalon report
            String uuid = UUID.randomUUID().toString();
            String bddStepUUIDPropName = "BDD_STEP_UUID";
            currentStepOrHookMap.put(bddStepUUIDPropName, uuid);
            String stepText = testStep.getStep().getText();
            Map<String, String> attributes = new HashMap<>();
            attributes.put(bddStepUUIDPropName, uuid);
            if (currentStepOrHookMap.get("line") != null) {
                attributes.put("BDD_STEP_LINE", currentStepOrHookMap.get("line").toString());
            }
            if (currentStepOrHookMap.get("line") != null) {
                attributes.put("BDD_STEP_NAME", logger.maskSecureValues(currentStepOrHookMap.get("name").toString()));
            }
            if (currentStepOrHookMap.get("line") != null) {
                attributes.put("BDD_STEP_KEYWORD", currentStepOrHookMap.get("keyword").toString());
            }
            logger.startKeyword(stepText, attributes, null);
            // end
            currentStepsList.add(currentStepOrHookMap);
        } else if (event.getTestStep() instanceof HookTestStep) {
            HookTestStep hookTestStep = (HookTestStep) event.getTestStep();
            currentStepOrHookMap = createHookStep(hookTestStep);
            addHookStepToTestCaseMap(currentStepOrHookMap, hookTestStep.getHookType());
        } else {
            throw new IllegalStateException();
        }
    }

    @SuppressWarnings("unchecked")
    private void handleTestStepFinished(TestStepFinished event) {
        currentStepOrHookMap.put("match", createMatchMap(event.getTestStep(), event.getResult()));
        currentStepOrHookMap.put("result", createResultMap(event.getResult()));
        // For Katalon report
        if (event.getTestStep() instanceof PickleStepTestStep) {
            String stepText = ((PickleStepTestStep) event.getTestStep()).getStep().getText();
            Result result = event.getResult();
            logResult(stepText, result);
            Map<String, Object> r = (Map<String, Object>) currentStepOrHookMap.get("result");
            Map<String, String> attributes = new HashMap<>();
            if (r.get("status") != null) {
                attributes.put("BDD_STEP_STATUS", r.get("status").toString());
            }
            if (r.get("error_message") != null) {
                attributes.put("BDD_STEP_ERROR_MESSAGE", logger.maskSecureValues(r.get("error_message").toString()));
            }
            if (r.get("duration") != null) {
                attributes.put("BDD_STEP_DURATION", r.get("duration").toString());
            }
            logger.endKeyword(stepText, attributes);
        }
        // end
    }

    private void handleWrite(WriteEvent event) {
        addOutputToHookMap(event.getText());
    }

    private void handleEmbed(EmbedEvent event) {
        addEmbeddingToHookMap(event.getData(), event.getMediaType(), event.getName());
    }

    private void onTestCaseFinished(TestCaseFinished event) {
        TestCase testCase = event.getTestCase();
        String name = getTestCaseName(testCase);
        Result result = event.getResult();
        logResult(name, result);
        logger.endTest(name, new HashMap<String, String>());
    }

    private void finishReport(TestRunFinished event) {
        Throwable exception = event.getResult().getError();
        if (exception != null) {
            featureMaps.add(createDummyFeatureForFailure(event));
        }

        try {
            Jackson.OBJECT_MAPPER.writeValue(writer, featureMaps);
            writer.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Map<String, Object> createFeatureMap(TestCase testCase) {
        Map<String, Object> featureMap = new HashMap<>();
        featureMap.put("uri", TestSourcesModel.relativize(testCase.getUri()));
        featureMap.put("elements", new ArrayList<Map<String, Object>>());
        Feature feature = testSources.getFeature(testCase.getUri());
        if (feature != null) {
            featureMap.put("keyword", feature.getKeyword());
            featureMap.put("name", logger.maskSecureValues(feature.getName()));
            String description = feature.getDescription() != null ? feature.getDescription() : "";
            featureMap.put("description", logger.maskSecureValues(description));
            featureMap.put("line", feature.getLocation().getLine());
            featureMap.put("id", TestSourcesModel.convertToId(feature.getName()));
            featureMap.put("tags", feature.getTags().stream().map(tag -> {
                Map<String, Object> json = new LinkedHashMap<>();
                json.put("name", tag.getName());
                json.put("type", "Tag");
                Map<String, Object> location = new LinkedHashMap<>();
                location.put("line", tag.getLocation().getLine());
                location.put("column", tag.getLocation().getColumn());
                json.put("location", location);
                return json;
            }).collect(toList()));

        }
        return featureMap;
    }

    private Map<String, Object> createTestCase(TestCaseStarted event) {
        Map<String, Object> testCaseMap = new HashMap<>();

        testCaseMap.put("start_timestamp", getDateTimeFromTimeStamp(event.getInstant()));

        TestCase testCase = event.getTestCase();

        testCaseMap.put("name", logger.maskSecureValues(testCase.getName()));
        testCaseMap.put("line", testCase.getLine());
        testCaseMap.put("type", "scenario");
        TestSourcesModel.AstNode astNode = testSources.getAstNode(currentFeatureFile, testCase.getLine());
        if (astNode != null) {
            testCaseMap.put("id", TestSourcesModel.calculateId(astNode));
            Scenario scenarioDefinition = TestSourcesModel.getScenarioDefinition(astNode);
            testCaseMap.put("keyword", scenarioDefinition.getKeyword());
            String description = scenarioDefinition.getDescription() != null ? scenarioDefinition.getDescription() : "";
            testCaseMap.put("description", logger.maskSecureValues(description));
        }
        testCaseMap.put("steps", new ArrayList<Map<String, Object>>());
        if (!testCase.getTags().isEmpty()) {
            List<Map<String, Object>> tagList = new ArrayList<>();
            for (String tag : testCase.getTags()) {
                Map<String, Object> tagMap = new HashMap<>();
                tagMap.put("name", logger.maskSecureValues(tag));
                tagList.add(tagMap);
            }
            testCaseMap.put("tags", tagList);
        }
        return testCaseMap;
    }

    private Map<String, Object> createBackground(TestCase testCase) {
        TestSourcesModel.AstNode astNode = testSources.getAstNode(currentFeatureFile, testCase.getLocation().getLine());
        if (astNode != null) {
            Background background = getBackgroundForTestCase(astNode).get();
            Map<String, Object> testCaseMap = new HashMap<>();
            testCaseMap.put("name", logger.maskSecureValues(background.getName()));
            testCaseMap.put("line", background.getLocation().getLine());
            testCaseMap.put("type", "background");
            testCaseMap.put("keyword", background.getKeyword());
            String description = background.getDescription() != null ? background.getDescription() : "";
            testCaseMap.put("description", logger.maskSecureValues(description));
            testCaseMap.put("steps", new ArrayList<Map<String, Object>>());
            return testCaseMap;
        }
        return null;
    }

    private boolean isFirstStepAfterBackground(PickleStepTestStep testStep) {
        TestSourcesModel.AstNode astNode = testSources.getAstNode(currentFeatureFile, testStep.getStepLine());
        if (astNode == null) {
            return false;
        }
        return currentElementMap != currentTestCaseMap && !TestSourcesModel.isBackgroundStep(astNode);
    }

    private Map<String, Object> createTestStep(PickleStepTestStep testStep) {
        Map<String, Object> stepMap = new HashMap<>();
        stepMap.put("name", logger.maskSecureValues(testStep.getStepText()));
        stepMap.put("line", testStep.getStepLine());
        TestSourcesModel.AstNode astNode = testSources.getAstNode(currentFeatureFile, testStep.getStepLine());
        StepArgument argument = testStep.getStepArgument();
        if (argument != null) {
            if (argument instanceof DocStringArgument) {
                DocStringArgument docStringArgument = (DocStringArgument) argument;
                stepMap.put("doc_string", createDocStringMap(docStringArgument));
            } else if (argument instanceof DataTableArgument) {
                DataTableArgument dataTableArgument = (DataTableArgument) argument;
                stepMap.put("rows", createDataTableList(dataTableArgument));
            }
        }
        if (astNode != null) {
            Step step = (Step) astNode.node;
            stepMap.put("keyword", step.getKeyword());
        }

        return stepMap;
    }

    private Map<String, Object> createHookStep(HookTestStep hookTestStep) {
        return new HashMap<>();
    }

    private void addHookStepToTestCaseMap(Map<String, Object> currentStepOrHookMap, HookType hookType) {
        String hookName;
        if (hookType == HookType.AFTER || hookType == HookType.AFTER_STEP)
            hookName = after;
        else hookName = before;

        Map<String, Object> mapToAddTo;
        switch (hookType) {
            case BEFORE:
                mapToAddTo = currentTestCaseMap;
                break;
            case AFTER:
                mapToAddTo = currentTestCaseMap;
                break;
            case BEFORE_STEP:
                mapToAddTo = currentBeforeStepHookList;
                break;
            case AFTER_STEP:
                mapToAddTo = currentStepsList.get(currentStepsList.size() - 1);
                break;
            default:
                mapToAddTo = currentTestCaseMap;
        }

        if (!mapToAddTo.containsKey(hookName)) {
            mapToAddTo.put(hookName, new ArrayList<Map<String, Object>>());
        }
        ((List<Map<String, Object>>) mapToAddTo.get(hookName)).add(currentStepOrHookMap);
    }

    private Map<String, Object> createMatchMap(TestStep step, Result result) {
        Map<String, Object> matchMap = new HashMap<>();
        if (step instanceof PickleStepTestStep) {
            PickleStepTestStep testStep = (PickleStepTestStep) step;
            if (!testStep.getDefinitionArgument().isEmpty()) {
                List<Map<String, Object>> argumentList = new ArrayList<>();
                for (Argument argument : testStep.getDefinitionArgument()) {
                    Map<String, Object> argumentMap = new HashMap<>();
                    if (argument.getValue() != null) {
                        argumentMap.put("val", logger.maskSecureValues(argument.getValue()));
                        argumentMap.put("offset", argument.getStart());
                    }
                    argumentList.add(argumentMap);
                }
                matchMap.put("arguments", argumentList);
            }
        }
        if (!result.getStatus().is(Status.UNDEFINED)) {
            matchMap.put("location", step.getCodeLocation());
        }
        return matchMap;
    }

    private Map<String, Object> createResultMap(Result result) {
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("status", result.getStatus().name().toLowerCase(ROOT));
        if (result.getError() != null) {
            resultMap.put("error_message", result.getError().getMessage());
        }
        if (!result.getDuration().isZero()) {
            resultMap.put("duration", result.getDuration().toNanos());
        }
        return resultMap;
    }

    private void addOutputToHookMap(String text) {
        if (!currentStepOrHookMap.containsKey("output")) {
            currentStepOrHookMap.put("output", new ArrayList<String>());
        }
        ((List<String>) currentStepOrHookMap.get("output")).add(text);
    }

    private void addEmbeddingToHookMap(byte[] data, String mediaType, String name) {
        if (!currentStepOrHookMap.containsKey("embeddings")) {
            currentStepOrHookMap.put("embeddings", new ArrayList<Map<String, Object>>());
        }
        Map<String, Object> embedMap = createEmbeddingMap(data, mediaType, name);
        ((List<Map<String, Object>>) currentStepOrHookMap.get("embeddings")).add(embedMap);
    }

    private Map<String, Object> createDummyFeatureForFailure(TestRunFinished event) {
        Throwable exception = event.getResult().getError();

        Map<String, Object> feature = new LinkedHashMap<>();
        feature.put("line", 1);
        {
            Map<String, Object> scenario = new LinkedHashMap<>();
            feature.put("elements", singletonList(scenario));

            scenario.put("start_timestamp", getDateTimeFromTimeStamp(event.getInstant()));
            scenario.put("line", 2);
            scenario.put("name", "Failure while executing Cucumber");
            scenario.put("description", "");
            scenario.put("id", "failure;failure-while-executing-cucumber");
            scenario.put("type", "scenario");
            scenario.put("keyword", "Scenario");

            Map<String, Object> when = new LinkedHashMap<>();
            Map<String, Object> then = new LinkedHashMap<>();
            scenario.put("steps", Arrays.asList(when, then));
            {

                {
                    Map<String, Object> whenResult = new LinkedHashMap<>();
                    when.put("result", whenResult);
                    whenResult.put("duration", 0);
                    whenResult.put("status", "passed");
                }
                when.put("line", 3);
                when.put("name", "Cucumber failed while executing");
                Map<String, Object> whenMatch = new LinkedHashMap<>();
                when.put("match", whenMatch);
                whenMatch.put("arguments", new ArrayList<>());
                whenMatch.put("location", "io.cucumber.core.Failure.failure_while_executing_cucumber()");
                when.put("keyword", "When ");

                {
                    Map<String, Object> thenResult = new LinkedHashMap<>();
                    then.put("result", thenResult);
                    thenResult.put("duration", 0);
                    thenResult.put("error_message", printStackTrace(exception));
                    thenResult.put("status", "failed");
                }
                then.put("line", 4);
                then.put("name", "Cucumber will report this error:");
                Map<String, Object> thenMatch = new LinkedHashMap<>();
                then.put("match", thenMatch);
                thenMatch.put("arguments", new ArrayList<>());
                thenMatch.put("location", "io.cucumber.core.Failure.cucumber_reports_this_error()");
                then.put("keyword", "Then ");
            }

            feature.put("name", "Test run failed");
            feature.put("description", "There were errors during the execution");
            feature.put("id", "failure");
            feature.put("keyword", "Feature");
            feature.put("uri", "classpath:io/cucumber/core/failure.feature");
            feature.put("tags", new ArrayList<>());
        }

        return feature;
    }

    private String getDateTimeFromTimeStamp(Instant instant) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSXXX")
                .withZone(ZoneOffset.UTC);
        return formatter.format(instant);
    }

    private Map<String, Object> createDocStringMap(DocStringArgument docString) {
        Map<String, Object> docStringMap = new HashMap<>();
        docStringMap.put("value", logger.maskSecureValues(docString.getContent()));
        docStringMap.put("line", docString.getLine());
        docStringMap.put("content_type", docString.getMediaType());
        return docStringMap;
    }

    private List<Map<String, List<String>>> createDataTableList(DataTableArgument argument) {
        List<Map<String, List<String>>> rowList = new ArrayList<>();
        for (List<String> row : argument.cells()) {
            Map<String, List<String>> rowMap = new HashMap<>();
            rowMap.put("cells", new ArrayList<>(row));
            rowList.add(rowMap);
        }
        return rowList;
    }

    private Map<String, Object> createEmbeddingMap(byte[] data, String mediaType, String name) {
        Map<String, Object> embedMap = new HashMap<>();
        embedMap.put("mime_type", mediaType); // Should be media-type but not
                                              // worth migrating for
        embedMap.put("data", Base64.getEncoder().encodeToString(data));
        if (name != null) {
            embedMap.put("name", name);
        }
        return embedMap;
    }

    private String getTestCaseName(TestCase testCase) {
        String name = "SCENARIO " + testCase.getName();
        return name;
    }

    private void logResult(String name, Result result) {
        Status status = result.getStatus();
        if (Status.PASSED.equals(status)) {
            logger.logPassed(name);
        } else {
            Throwable t = result.getError();
            if (t == null) {
                LogLevel level;
                if (Status.FAILED.equals(status)) {
                    level = LogLevel.FAILED;
                } else {
                    level = LogLevel.NOT_RUN;
                }
                logger.logMessage(level, name);
            } else {
                String stackTraceForThrowable = ExceptionsUtil.getStackTraceForThrowable(t);
                String message = MessageFormat.format(StringConstants.MAIN_LOG_MSG_FAILED_BECAUSE_OF, name,
                        stackTraceForThrowable);
                logError(t, message);
            }
        }
    }

    private void logError(Throwable t, String message) {
        logger.logMessage(ErrorCollector.fromError(t), message, t);
    }
}