package com.kms.katalon.core.appium.driver;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.concurrent.atomic.AtomicBoolean;

public class AppiumLogWatcher {
    private static final int WATCH_SERVICE_POLL_TIMEOUT_IN_MILLISECONDS = 500;
    
    private final Path logFilePath;

    private final AppiumLogListener listener;

    private final AtomicBoolean running = new AtomicBoolean(false);

    private Thread thread;

    AppiumLogWatcher(String appiumLogFilePath, AppiumLogListener listener) {
        this.logFilePath = Paths.get(appiumLogFilePath);
        this.listener = listener;
    }

    public static AppiumLogWatcher create(String appiumLogFilePath, AppiumLogListener listener) {
        return new AppiumLogWatcher(appiumLogFilePath, listener);
    }

    public void start() {
        if (running.compareAndSet(false, true)) {
            thread = new Thread(this::watchLoop, "AppiumLogWatcher");
            thread.setDaemon(true);
            thread.start();
        }
    }

    public void stop() {
        running.set(false);
        if (thread != null) {
            thread.interrupt();
        }
    }

    private void watchLoop() {
        if (!Files.exists(logFilePath)) {
            if (listener != null) {
                listener.onLog("Log file not found: " + logFilePath);
            }
            running.set(false);
            return;
        }

        try (WatchService watchService = FileSystems.getDefault().newWatchService();
             RandomAccessFile raf = new RandomAccessFile(logFilePath.toFile(), "r")) {

            Path dir = logFilePath.getParent();
            dir.register(watchService, StandardWatchEventKinds.ENTRY_MODIFY);

            long filePointer = raf.length(); // start at end like tail -f

            while (running.get()) {
                boolean modified = false;

                // Poll for file system events with 500ms timeout to avoid blocking indefinitely
                WatchKey key = watchService.poll(WATCH_SERVICE_POLL_TIMEOUT_IN_MILLISECONDS,
                        java.util.concurrent.TimeUnit.MILLISECONDS);
                if (key != null) {
                    for (WatchEvent<?> event : key.pollEvents()) {
                        if (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY
                                && logFilePath.getFileName().equals(event.context())) {
                            modified = true;
                        }
                    }
                    
                    key.reset();
                }

                // Fallback for Windows where file system events may be unreliable - check file length directly
                if (modified || raf.length() > filePointer) {
                    raf.seek(filePointer);
                    String line;
                    while ((line = raf.readLine()) != null) {
                        if (listener != null) {
                            listener.onLog(line);
                        }
                    }
                    
                    filePointer = raf.getFilePointer();
                }
            }
        } catch (IOException | InterruptedException e) {
            if (running.get() && listener != null) {
                listener.onLog("Log watcher stopped due to error: " + e.getMessage());
            }
        }
    }
}
