package com.kms.katalon.core.appium.util;

import static java.util.Objects.requireNonNull;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.http.client.utils.URIBuilder;
import org.openqa.selenium.AcceptedW3CCapabilityKeys;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.http.HttpClient.Factory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.kms.katalon.core.appium.driver.AndroidMobileCapabilityType;
import com.kms.katalon.core.appium.driver.IOSMobileCapabilityType;
import com.kms.katalon.core.appium.driver.MobileCapabilityType;
import com.kms.katalon.core.appium.driver.YouiEngineCapabilityType;
import com.kms.katalon.core.network.ProxyInformation;
import com.kms.katalon.core.selenium.remote.http.ConfiguredHttpClientFactory;
import com.kms.katalon.selenium.exception.W3CCapabilityViolationException;
import com.kms.katalon.selenium.filter.SeleniumBasicAuthorizationFilter;
import com.kms.katalon.selenium.filter.SeleniumCreateSessionPayloadFilter;
import com.kms.katalon.selenium.filter.SeleniumCreateSessionResponseFilter;
import com.kms.katalon.selenium.filter.SeleniumProxyBasicAuthorizationFilter;

import io.appium.java_client.AppiumClientConfig;
import io.appium.java_client.MobileCommand;
import io.appium.java_client.remote.AppiumCommandExecutor;

public class AppiumDriverUtil {

    private static Logger logger = LoggerFactory.getLogger(AppiumDriverUtil.class);

    private static final String APPIUM_PREFIX = "appium:";

    private static final List<String> APPIUM_CAPABILITIES = ImmutableList.<String> builder()
            .addAll(getAppiumCapabilities(MobileCapabilityType.class))
            .addAll(getAppiumCapabilities(AndroidMobileCapabilityType.class))
            .addAll(getAppiumCapabilities(IOSMobileCapabilityType.class))
            .addAll(getAppiumCapabilities(YouiEngineCapabilityType.class))
            .build();

    private static List<String> getAppiumCapabilities(Class<?> capabilityList) {
        return Arrays.stream(capabilityList.getDeclaredFields()).map(field -> {
            field.setAccessible(true);
            try {
                return field.get(capabilityList).toString();
            } catch (IllegalAccessException e) {
                throw new IllegalArgumentException(e);
            }
        }).filter(s -> !MobileCapabilityType.FORCE_MJSONWP.equals(s)).collect(Collectors.toList());
    }

    /**
     * From Appium 9, it will validate the DesiredCapabilities if there is any key which violate the
     * {@link org.openqa.selenium.AcceptedW3CCapabilityKeys} it will throw the exception.
     * Please see the detail here {@link org.openqa.selenium.remote.NewSessionPayload#validate()}
     * <br>
     * Applied the naming convention follow W3C capability
     * <br>
     * 
     * @throws W3CCapabilityViolationException
     * @see <a href=
     * "https://github.com/appium/java-client/blob/v7.3.0/src/main/java/io/appium/java_client/remote/NewAppiumSessionPayload.java#L388}">NewAppiumSessionPayload</a>
     */
    public static MutableCapabilities filterByAcceptedW3CCapabilityKeys(MutableCapabilities capabilities)
            throws W3CCapabilityViolationException {

        AcceptedW3CCapabilityKeys acceptedW3CCapabilityKeys = new AcceptedW3CCapabilityKeys();

        Map<String, Object> filteredCapabilities = new HashMap<>();
        Map<String, String> backfillMap = new HashMap<>();
        List<String> uncompatibleCapabilityKeys = new ArrayList<>();

        capabilities.asMap().keySet().stream().forEach(key -> {
            if (acceptedW3CCapabilityKeys.test(key)) {
                filteredCapabilities.put(key, capabilities.getCapability(key));
            } else {
                if (APPIUM_CAPABILITIES.contains(key)) {
                    String backfillKey = APPIUM_PREFIX + key;
                    filteredCapabilities.put(backfillKey, capabilities.getCapability(key));
                    backfillMap.put(key, backfillKey);
                } else {
                    uncompatibleCapabilityKeys.add(key);
                }
            }
        });

        if (!backfillMap.isEmpty()) {
            logger.info(String.format("There are uncompatible capabilities keys which will be replaced: %s",
                    Joiner.on(", ").withKeyValueSeparator("=").join(backfillMap)));
        }

        /**
         * We let the Appium server throw the excption if any missing capabilities needed
         */
        // if (!uncompatibleCapabilityKeys.isEmpty()) {
        // throw new W3CCapabilityViolationException(uncompatibleCapabilityKeys);
        // }

        return new DesiredCapabilities(filteredCapabilities);
    }

    public static URL removeUserInfoPart(URL url) throws URISyntaxException, MalformedURLException {
        URIBuilder uriBuilder = new URIBuilder(url.toURI());
        uriBuilder.setUserInfo(null);
        return uriBuilder.build().toURL();
    }

    public static AppiumCommandExecutor getAppiumExecutorForRemoteDriver(URL remoteWebServerUrl,
            ProxyInformation proxyInfo, MutableCapabilities capabilities) throws URISyntaxException, IOException {
        URL cleanRemoteWebServerUrl = AppiumDriverUtil.removeUserInfoPart(remoteWebServerUrl);
        Factory clientFactory = ConfiguredHttpClientFactory.of(proxyInfo, remoteWebServerUrl, capabilities);

        AppiumClientConfig appiumClientConfig = AppiumClientConfig.defaultConfig()
                .baseUrl(requireNonNull(cleanRemoteWebServerUrl))
                .withFilter(new SeleniumProxyBasicAuthorizationFilter(proxyInfo.getUsername(), proxyInfo.getPassword())
                        .andThen(new SeleniumBasicAuthorizationFilter(remoteWebServerUrl))
                        .andThen(new SeleniumCreateSessionPayloadFilter(capabilities))
                        .andThen(new SeleniumCreateSessionResponseFilter()));

        return new AppiumCommandExecutor(MobileCommand.commandRepository, null, clientFactory, appiumClientConfig);
    }

    public static URL appendSessionToUrl(URL remoteServerUrl, String sessionId)
            throws URISyntaxException, MalformedURLException {
        URIBuilder builder = new URIBuilder(remoteServerUrl.toURI());
        String path = builder.getPath();
        if (!path.endsWith("/")) {
            path += "/";
        }
        builder.setPath(path + "session/" + sessionId);
        return builder.build().toURL();
    }
}
