package com.kms.katalon.core.mobile.helper;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.text.MessageFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.Point;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.interactions.Pause;
import org.openqa.selenium.interactions.PointerInput;
import org.openqa.selenium.interactions.Sequence;

import com.katalon.selfhealing.execution.healers.LLMTestObjectHealer.LLMCustomPrompts;
import com.katalon.selfhealing.execution.healers.MobileStandardTestObjectMapper;
import com.katalon.selfhealing.execution.healers.SelfHealingPipeline;
import com.katalon.selfhealing.execution.healers.SelfHealingPipeline.SelfHealingParams;
import com.kms.katalon.core.aut.MobileAUT;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.enums.mobile.LocatorStrategy;
import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.keyword.internal.KeywordExecutionContext;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.mobile.common.FindElementsResult;
import com.kms.katalon.core.mobile.common.MobileXPathBuilder;
import com.kms.katalon.core.mobile.common.internal.SelfHealingController;
import com.kms.katalon.core.mobile.constants.StringConstants;
import com.kms.katalon.core.mobile.driver.AppiumDriverSession;
import com.kms.katalon.core.mobile.driver.AppiumSessionCollector;
import com.kms.katalon.core.mobile.keyword.internal.AndroidProperties;
import com.kms.katalon.core.mobile.keyword.internal.GUIObject;
import com.kms.katalon.core.mobile.keyword.internal.MobileDriverFactory;
import com.kms.katalon.core.mobile.keyword.internal.MobileLocatorFinder;
import com.kms.katalon.core.mobile.keyword.internal.MobileSearchEngine;
import com.kms.katalon.core.testobject.BrokenTestObject;
import com.kms.katalon.core.testobject.MobileTestObject;
import com.kms.katalon.core.testobject.TestObject;
import com.kms.katalon.core.util.ConsoleCommandBuilder;
import com.kms.katalon.core.util.internal.ProcessUtil;

import io.appium.java_client.AppiumBy;
import io.appium.java_client.AppiumDriver;
import io.appium.java_client.android.AndroidDriver;
import io.appium.java_client.ios.IOSDriver;

public class MobileCommonHelper {

    private static final KeywordLogger logger = KeywordLogger.getInstance(MobileCommonHelper.class);

    private static final String ATTRIBUTE_NAME_FOR_ANDROID_RESOURCE_ID = "resourceId";

    private static final String ATTRIBUTE_NAME_FOR_ANDROID_CONTENT_DESC = "name";

    public static final String PROPERTY_NAME_DEVICE_PIXEL_RATIO = "devicePixelRatio";

    public static final String PROPERTY_NAME_OS_STATUS_BAR_HEIGHT = "osStatusBarHeight";

    public static final String PROPERTY_NAME_STATUS_BAR_HEIGHT = "statusBarHeight";

    public static final String PROPERTY_NAME_SCALE_FACTOR = "scaleFactor";

    public static final String PROPERTY_NAME_IOS_BUNDLE_ID = "iosBundleId";

    public static final int DEFAULT_TAP_DURATION = 50;

    public static final int DEFAULT_SWIPE_DURATION = 500;

    public static final int DEFAULT_PINCH_DURATION = 1000;

    public static final int DEFAULT_LONG_TAP_DURATION = 2000;

    public static final double MAX_FILE_SIZE_MB = 4.5;

    public static final String[] SUPPORTED_IMAGE_TYPES = { "png", "jpg", "jpeg" };

    public static final EnumSet<LocatorStrategy> VALID_LOCATOR_STRATEGIES_FOR_AI = EnumSet.of(LocatorStrategy.ID,
            LocatorStrategy.NAME, LocatorStrategy.ANDROID_UI_AUTOMATOR, LocatorStrategy.XPATH,
            LocatorStrategy.ANDROID_VIEWTAG, LocatorStrategy.IOS_CLASS_CHAIN, LocatorStrategy.IOS_PREDICATE_STRING,
            LocatorStrategy.CLASS_NAME);

    public static Map<String, String> deviceModels = new HashMap<String, String>();

    static {
        deviceModels.put("iPhone3,1", "iPhone 4");
        deviceModels.put("iPhone3,3", "iPhone 4");
        deviceModels.put("iPhone4,1", "iPhone 4S");

        deviceModels.put("iPhone5,1", "iPhone 5");
        deviceModels.put("iPhone5,2", "iPhone 5");
        deviceModels.put("iPhone5,3", "iPhone 5c");
        deviceModels.put("iPhone5,4", "iPhone 5c");
        deviceModels.put("iPhone6,1", "iPhone 5s");
        deviceModels.put("iPhone6,2", "iPhone 5s");
        deviceModels.put("iPhone7,1", "iPhone 6 Plus");
        deviceModels.put("iPhone7,2", "iPhone 6");
        deviceModels.put("iPad1,1", "iPad");
        deviceModels.put("iPad2,1", "iPad 2");
        deviceModels.put("iPad2,2", "iPad 2");
        deviceModels.put("iPad2,3", "iPad 2");
        deviceModels.put("iPad2,4", "iPad 2");
        deviceModels.put("iPad2,5", "iPad mini");
        deviceModels.put("iPad2,6", "iPad mini");
        deviceModels.put("iPad2,7", "iPad mini");

        deviceModels.put("iPad3,1", "iPad 3");
        deviceModels.put("iPad3,2", "iPad 3");
        deviceModels.put("iPad3,3", "iPad 3");
        deviceModels.put("iPad3,4", "iPad 4");
        deviceModels.put("iPad3,5", "iPad 4");
        deviceModels.put("iPad3,6", "iPad 4");
        deviceModels.put("iPad4,1", "iPad Air");
        deviceModels.put("iPad4,2", "iPad Air");
        deviceModels.put("iPad4,3", "iPad Air");
        deviceModels.put("iPad4,4", "iPad mini 2");
        deviceModels.put("iPad4,5", "iPad mini 2");
        deviceModels.put("iPad4,6", "iPad mini 2");
        deviceModels.put("iPad4,7", "iPad mini 3");
        deviceModels.put("iPad4,8", "iPad mini 3");
        deviceModels.put("iPad4,9", "iPad mini 3");
        deviceModels.put("iPad5,3", "iPad Air 2");
        deviceModels.put("iPad5,4", "iPad Air 2");

    }

    public static Map<String, String> airPlaneButtonCoords = new HashMap<String, String>();

    static {
        airPlaneButtonCoords.put("iPhone 5s", "40;195");
        airPlaneButtonCoords.put("iPhone 5", "40;195");

        airPlaneButtonCoords.put("iPad 2", "260;905");
        airPlaneButtonCoords.put("iPad 3", "260;905");
        airPlaneButtonCoords.put("iPad 4", "260;905");

        airPlaneButtonCoords.put("iPad Air", "260;905");
        airPlaneButtonCoords.put("iPad Air 2", "260;905");

        airPlaneButtonCoords.put("iPhone 6", "50;290");
        airPlaneButtonCoords.put("iPhone 6 Plus", "59;359");

        airPlaneButtonCoords.put("iPad mini", "265;905");
        airPlaneButtonCoords.put("iPad mini 2", "265;905");
        airPlaneButtonCoords.put("iPad mini 3", "265;905");
    }

    public static boolean canUseImageBasedTesting() {
        return RunConfiguration.canUseMobileImageBasedTesting();
    }

    public static int checkTimeout(int timeout) {
        logger.logDebug(com.kms.katalon.core.constants.StringConstants.COMM_LOG_INFO_CHECKING_TIMEOUT);
        if (timeout <= 0) {
            int defaultElementTimeout = RunConfiguration.getElementTimeoutForMobile();
            logger.logWarning(MessageFormat.format(
                    com.kms.katalon.core.constants.StringConstants.COMM_LOG_WARNING_INVALID_TIMEOUT, timeout,
                    defaultElementTimeout));
            return defaultElementTimeout;
        }
        return timeout;
    }

    public static void checkXAndY(Number x, Number y) {
        logger.logDebug(StringConstants.COMM_LOG_INFO_CHECKING_X);
        if (x == null) {
            throw new StepFailedException(
                    MessageFormat.format(StringConstants.KW_MSG_FAILED_PARAM_X_CANNOT_BE_NULL, "x"));
        }

        logger.logDebug(StringConstants.COMM_LOG_INFO_CHECKING_Y);
        if (y == null) {
            throw new StepFailedException(
                    MessageFormat.format(StringConstants.KW_MSG_FAILED_PARAM_X_CANNOT_BE_NULL, "y"));
        }
    }

    public static void doubleTap(AppiumDriver driver, Point point) {
        if (driver instanceof AndroidDriver) {
            PointerInput finger = new PointerInput(PointerInput.Kind.TOUCH, "finger");
            Sequence doubleTap = new Sequence(finger, 0);

            // First tap
            doubleTap.addAction(finger.createPointerMove(Duration.ZERO, PointerInput.Origin.viewport(), point));
            doubleTap.addAction(finger.createPointerDown(PointerInput.MouseButton.LEFT.asArg()));
            doubleTap.addAction(finger.createPointerUp(PointerInput.MouseButton.LEFT.asArg()));

            // Short pause between taps (not after pointer up, but before the next tap)
            doubleTap.addAction(new Pause(finger, Duration.ofMillis(100))); // Typical double-tap gap is ~100ms

            // Second tap
            doubleTap.addAction(finger.createPointerDown(PointerInput.MouseButton.LEFT.asArg()));
            doubleTap.addAction(finger.createPointerUp(PointerInput.MouseButton.LEFT.asArg()));

            // Perform the sequence
            driver.perform(List.of(doubleTap));
        } else if (driver instanceof IOSDriver) {
            Map<String, Object> args = new HashMap<>();
            args.put("x", point.x);
            args.put("y", point.y);
            driver.executeScript("mobile: doubleTap", args);
        }
    }

    public static WebElement findElement(AppiumDriver driver, TestObject testObject, int timeOut) throws Exception {
        WebElement cachedWebElement = testObject.getCachedWebElement();
        if (cachedWebElement != null) {
            logger.logDebug("Using cached element");
            return cachedWebElement;
        }

        logger.logDebug("Finding element from Test Object's properties");
        List<WebElement> elements = findElements(driver, testObject, timeOut);
        if (elements != null && elements.size() > 0) {
            return elements.get(0);
        }

        return null;
    }

    public static WebElement findElementByIosClassChain(IOSDriver iosDriver, String type, String name) {
        String locator = "**/%s[`name == '%s'`]";
        return iosDriver.findElement(AppiumBy.iOSClassChain(String.format(locator, type, name)));
    }

    public static WebElement findElementByIosClassChain(IOSDriver iosDriver, String type, String name, String label) {
        String locator = "**/%s[`name == '%s' OR label = '%s'`]";
        return iosDriver.findElement(AppiumBy.iOSClassChain(String.format(locator, type, name, label)));
    }

    public static List<WebElement> findElements(AppiumDriver driver, TestObject testObject, int timeout)
            throws Exception {

        // If the TestObject is not a MobileTestObject, we do not support Self-Healing.
        // In this case, we fall back to treating it as a Web TestObject and use the MobileSearchEngine
        // to locate elements accordingly.
        if (!(testObject instanceof MobileTestObject)) {
            logger.logWarning("The input TestObject is not a MobileTestObject. Self-Healing is not supported");
            driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(timeout));
            MobileSearchEngine searchEngine = new MobileSearchEngine(driver, testObject);
            return searchEngine.findWebElements(false);
        }

        final MobileTestObject mobileTestObject = (MobileTestObject) testObject;

        if (mobileTestObject.getLocatorStrategy() == null) {
            mobileTestObject.setLocatorStrategy(LocatorStrategy.ATTRIBUTES);
        }

        if (mobileTestObject.getLocatorCollection().isEmpty()) {
            List<LocatorStrategy> locatorStrategies = LocatorStrategy
                    .getLocatorStrategies(mobileTestObject.getPlatform());
            locatorStrategies.forEach(locatorStrategy -> {
                String locator = StringUtils.EMPTY;
                locator = MobileLocatorFinder.findLocator(mobileTestObject, locatorStrategy);
                mobileTestObject.setLocatorValue(locatorStrategy, locator);
            });
        }

        timeout = MobileCommonHelper.checkTimeout(timeout);
        boolean shouldApplySelfHealing = RunConfiguration.shouldApplySelfHealingForMobile();
        if (shouldApplySelfHealing) {
            return findElementsWithSelfHealing(driver, mobileTestObject, timeout);
        }

        List<WebElement> foundElements = new ArrayList<WebElement>();
        FindElementsResult findResult = findElementsByDefaultLocator(driver, mobileTestObject, timeout);
        if (findResult != null) {
            foundElements = findResult.getElements();
        }

        return foundElements;
    }

    public static FindElementsResult findElementsByDefaultLocator(AppiumDriver driver, TestObject testObject,
            int timeout) throws Exception {
        return findElementsByLocatorStrategy(driver, testObject, timeout,
                ((MobileTestObject) testObject).getLocatorStrategy());
    }

    public static FindElementsResult findElementsByLocatorStrategy(AppiumDriver driver, TestObject testObject,
            int timeout, LocatorStrategy locatorStrategy) throws Exception {
        // Disable driver timeout and handle timeout manually.
        long endTime = System.currentTimeMillis() + timeout * 1000L;
        Duration defaultImplicitlyWait = driver.manage().timeouts().getImplicitWaitTimeout();

        try {
            MobileSearchEngine searchEngine = new MobileSearchEngine(driver, testObject);
            List<WebElement> webElements = new ArrayList<WebElement>();
            while (webElements.size() == 0) {
                driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));
                webElements = searchEngine.findElementsByMobileLocator(locatorStrategy);

                if (System.currentTimeMillis() >= endTime) {
                    break;
                }

                // Throttle to reduce API spam
                if (webElements == null || webElements.isEmpty()) {
                    Thread.sleep(200);
                }
            }

            MobileTestObject mobileTestObject = (MobileTestObject) testObject;
            String locator = mobileTestObject.getLocatorCollection().get(locatorStrategy);
            return FindElementsResult.from(webElements, locator, locatorStrategy);
        } finally {
            driver.manage().timeouts().implicitlyWait(defaultImplicitlyWait);
        }
    }

    private static List<WebElement> findElementsWithSelfHealing(AppiumDriver driver, MobileTestObject testObject,
            int timeout) throws Exception {
        Exception defaultException = null;
        try {
            List<WebElement> foundElements = new ArrayList<WebElement>();
            FindElementsResult findResult = findElementsByDefaultLocator(driver, testObject, timeout);
            if (findResult != null) {
                foundElements = findResult.getElements();
            }

            if (foundElements != null && !foundElements.isEmpty()) {
                return foundElements;
            }
        } catch (Exception exception) {
            defaultException = exception;
        }
        if (defaultException != null && !(defaultException instanceof NoSuchElementException)) {
            throw defaultException;
        }

        var selfHealingParams = new SelfHealingParams();

        boolean isMobilePlatform = KeywordExecutionContext.isRunningMobile();
        String runningKeyword = KeywordExecutionContext.getRunningKeyword();
        List<String> excludedKeywords = RunConfiguration.getExcludedMobileKeywordsFromSelfHealing();

        var isSelfHealingEnabled = isMobilePlatform && !excludedKeywords.contains(runningKeyword)
                && RunConfiguration.shouldApplySelfHealing();
        if (!isSelfHealingEnabled) {
            return null;
        }

        selfHealingParams.useBasicSelfHealing = true;
        selfHealingParams.useAISelfHealing = RunConfiguration.getMobileAISelfHealingEnabled();
        selfHealingParams.aiSelfHealingInputSources = RunConfiguration.getMobileAISelfHealingInputSources();
        selfHealingParams.currentAction = runningKeyword;

        selfHealingParams.aut = new MobileAUT(driver);

        selfHealingParams.testObject = testObject;
        selfHealingParams.standardTestObjectMapper = (originalTestObject) -> {
            if (originalTestObject == null) {
                throw new IllegalArgumentException("TestObject cannot be null");
            }
            if (!(originalTestObject instanceof MobileTestObject)) {
                throw new IllegalArgumentException(
                        "Expected MobileTestObject but got " + originalTestObject.getClass().getName());
            }
            return new MobileStandardTestObjectMapper().map((MobileTestObject) originalTestObject);
        };

        selfHealingParams.basicSelfHealer = (tempTestObject) -> {
            var foundElements = findElementsWithAlternativeLocators(driver, testObject, 0);
            return foundElements;
        };

        selfHealingParams.elementsFinder = (foundLocator) -> {
            var tempTestObject = new MobileTestObject("self-healing-temp-test-object-id");
            var locatorType = LocatorStrategy.valueOf(foundLocator.type);
            tempTestObject.setLocatorStrategy(locatorType);
            tempTestObject.setLocatorValue(locatorType, foundLocator.value);
            var result = findElementsByDefaultLocator(driver, tempTestObject, 0);
            return result.getElements();
        };

        var selfHealingPipeline = new SelfHealingPipeline();
        selfHealingPipeline.registerPhase((context) -> {
            var foundElements = context.data.results.foundElements;
            var foundLocator = context.data.results.foundLocator;

            if (foundElements == null || foundElements.isEmpty()) {
                return;
            }

            var firstElement = foundElements.get(0);

            var healingResult = FindElementsResult.from(firstElement, foundLocator.value,
                    LocatorStrategy.valueOf(foundLocator.type), null);
            registerHealingObject(healingResult, testObject);
        });

        var customPrompts = new LLMCustomPrompts();
        var prioritizedLocatorStrategies = RunConfiguration
                .getPrioritizedLocatorStrategies(((MobileTestObject) testObject).getPlatform());
        var enabledPrioritizedLocatorStrategies = prioritizedLocatorStrategies.stream()
                .filter(entry -> entry.getRight() && VALID_LOCATOR_STRATEGIES_FOR_AI.contains(entry.getLeft()))
                .map(entry -> entry.getLeft())
                .collect(Collectors.toList());
        customPrompts.outputRules = MessageFormat.format("""
                - Prefer the locator follow this order: {0}
                """,
                StringUtils.join(enabledPrioritizedLocatorStrategies, ", "));
        customPrompts.locatorTypes = "\"" + StringUtils.join(VALID_LOCATOR_STRATEGIES_FOR_AI, "\" | \"") + "\"";
        selfHealingParams.customPrompts = customPrompts;

        var context = selfHealingPipeline.execute(selfHealingParams);
        return context.results.foundElements;
    }

    private static List<WebElement> findElementsWithAlternativeLocators(AppiumDriver driver, TestObject testObject, int timeout)
            throws Exception {
        SelfHealingController.setLogger(logger);

        try {
            boolean isMobilePlatform = KeywordExecutionContext.isRunningMobile();
            String runningKeyword = KeywordExecutionContext.getRunningKeyword();
            List<String> excludeKeywords = RunConfiguration.getExcludedMobileKeywordsFromSelfHealing();
            if (!isMobilePlatform || excludeKeywords.contains(runningKeyword)) {
                return Collections.emptyList();
            }

            MobileTestObject mto = (MobileTestObject) testObject;
            SelfHealingController.logWarning(MessageFormat.format(
                    StringConstants.KW_LOG_INFO_DEFAULT_LOCATOR_FAILED_TRY_SELF_HEALING, testObject.getObjectId(),
                    mto.getLocatorStrategy(), mto.getLocatorCollection().get(mto.getLocatorStrategy())));

            FindElementsResult substituteFindElementsResult = null;
            List<TestObject> healingCandidates = SelfHealingController.findHealedTestObjects(testObject);
            for (TestObject healingCandidate : healingCandidates) {
                if (healingCandidate == null) {
                    continue;
                }

                MobileTestObject mobileTestObject = (MobileTestObject) healingCandidate;
                LocatorStrategy locatorStrategy = mobileTestObject.getLocatorStrategy();
                String locator = mobileTestObject.getLocatorCollection().get(locatorStrategy);
                if (StringUtils.isEmpty(locator)) {
                    SelfHealingController.logWarning(
                            MessageFormat.format(StringConstants.KW_LOG_INFO_EMPTY_LOCATOR, locatorStrategy.getName()));
                    continue;
                }

                SelfHealingController.logInfo(MessageFormat.format(
                        StringConstants.KW_LOG_INFO_FINDING_ELEMENT_WITH_LOCATOR, locatorStrategy.getName(), locator));

                List<WebElement> foundElements = new ArrayList<WebElement>();
                FindElementsResult findResult = findElementsByDefaultLocator(driver, mobileTestObject, 0);
                if (findResult != null) {
                    foundElements = findResult.getElements();
                }

                if (foundElements != null && !foundElements.isEmpty()) {
                    SelfHealingController.logInfo(MessageFormat.format(StringConstants.KW_LOG_INFO_FOUND_ELEMENT,
                            foundElements.size(), locatorStrategy.getName(), locator));
                    if (foundElements.size() > 1) {
                        if (substituteFindElementsResult == null) {
                            substituteFindElementsResult = findResult;
                        }

                        SelfHealingController.logWarning(MessageFormat.format(
                                StringConstants.KW_LOG_INFO_FOUND_MULTI_ELEMENTS, locatorStrategy.getName(), locator));
                        continue;
                    }

                    registerHealingObject(findResult, testObject);
                    return foundElements;
                }
            }

            if (substituteFindElementsResult != null) {
                registerHealingObject(substituteFindElementsResult, testObject);
                return substituteFindElementsResult.getElements();
            }

            List<Pair<LocatorStrategy, Boolean>> prioritizedLocatorStrategies = RunConfiguration
                    .getPrioritizedLocatorStrategies(((MobileTestObject) testObject).getPlatform());
            LocatorStrategy locatorStrategy = ((MobileTestObject) testObject).getLocatorStrategy();
            for (Pair<LocatorStrategy, Boolean> prioritizedLocatorStrategy : prioritizedLocatorStrategies) {
                boolean isEnabled = prioritizedLocatorStrategy.getRight();
                if (!isEnabled) {
                    continue;
                }

                LocatorStrategy currentLocatorStrategy = prioritizedLocatorStrategy.getLeft();
                if (currentLocatorStrategy != locatorStrategy) {
                    try {
                        String proposedLocator = ((MobileTestObject) testObject).getLocatorCollection()
                                .get(currentLocatorStrategy);
                        if (StringUtils.isEmpty(proposedLocator)) {
                            SelfHealingController.logWarning(MessageFormat.format(
                                    StringConstants.KW_LOG_INFO_EMPTY_LOCATOR, currentLocatorStrategy.getName()));
                            continue;
                        }

                        SelfHealingController
                                .logInfo(MessageFormat.format(StringConstants.KW_LOG_INFO_FINDING_ELEMENT_WITH_LOCATOR,
                                        currentLocatorStrategy.getName(), proposedLocator));

                        List<WebElement> foundElements = new ArrayList<WebElement>();
                        FindElementsResult findResult = findElementsByLocatorStrategy(driver, testObject, 0,
                                currentLocatorStrategy);
                        if (findResult != null) {
                            foundElements = findResult.getElements();
                        }

                        if (findResult != null && !foundElements.isEmpty()) {
                            SelfHealingController
                                    .logInfo(MessageFormat.format(StringConstants.KW_LOG_INFO_FOUND_ELEMENT,
                                            foundElements.size(), currentLocatorStrategy.getName(), proposedLocator));
                            if (foundElements.size() > 1) {
                                if (substituteFindElementsResult == null) {
                                    substituteFindElementsResult = findResult;
                                }

                                SelfHealingController.logWarning(
                                        MessageFormat.format(StringConstants.KW_LOG_INFO_FOUND_MULTI_ELEMENTS,
                                                currentLocatorStrategy.getName(), proposedLocator));
                                continue;
                            }

                            registerHealingObject(findResult, testObject);
                            return foundElements;
                        }
                    } catch (Exception exception) {
                        SelfHealingController.logWarning(exception.getMessage(), exception);
                    }
                }
            }

            if (substituteFindElementsResult != null) {
                registerHealingObject(substituteFindElementsResult, testObject);
                return substituteFindElementsResult.getElements();
            }
        } catch (Exception exception) {
            SelfHealingController.logWarning(exception.getMessage(), exception);
        }

        return Collections.emptyList();
    }

    public static long getAndroidPackageSize(String deviceId, String appId, String adbFolder) {
        try {
            String command = MessageFormat.format(
                    "adb -s {0} shell \"stat -c %s $(pm path {1} | grep base.apk | cut -d : -f 2)\"", deviceId, appId);
            List<String> packageSize = ConsoleCommandBuilder.create(command).path(adbFolder).execSync();
            return Long.valueOf(packageSize.get(0));
        } catch (Exception error) {
            return -1;
        }
    }

    public static String getAttributeLocatorValue(TestObject testObject) {
        if (testObject == null || testObject.getProperties().isEmpty()) {
            return null;
        }

        MobileXPathBuilder xpathBuilder = new MobileXPathBuilder(testObject.getActiveProperties());
        return xpathBuilder.build();
    }

    public static String getAttributeValue(WebElement element, String attributeName) {
        switch (attributeName.toString()) {
            case GUIObject.HEIGHT:
                return String.valueOf(element.getSize().height);
            case GUIObject.WIDTH:
                return String.valueOf(element.getSize().width);
            case GUIObject.X:
                return String.valueOf(element.getLocation().x);
            case GUIObject.Y:
                return String.valueOf(element.getLocation().y);
            case AndroidProperties.ANDROID_RESOURCE_ID: {
                if (MobileDriverFactory.getDriver() instanceof AndroidDriver) {
                    return element.getAttribute(ATTRIBUTE_NAME_FOR_ANDROID_RESOURCE_ID);
                }

                return null;
            }
            case AndroidProperties.ANDROID_CONTENT_DESC: {
                if (MobileDriverFactory.getDriver() instanceof AndroidDriver) {
                    return element.getAttribute(ATTRIBUTE_NAME_FOR_ANDROID_CONTENT_DESC);
                }

                return null;
            }
            default:
                try {
                    return element.getAttribute(attributeName);
                } catch (NoSuchElementException e) {
                    // attribute not found, return null
                    return null;
                }
        }
    }

    public static String getBundleId(String packagePath, String aaptFolder) {
        return grepBundleInfo(packagePath, aaptFolder, "package: name='(.*?)'");
    }

    public static String getBundleVersion(String packagePath, String aaptFolder) {
        return grepBundleInfo(packagePath, aaptFolder, "versionName='(.*?)'");
    }

    public static String getInstalledAppVersion(String deviceId, String appId, String adbFolder) {
        return grepInstalledAppInfo(deviceId, appId, adbFolder, "versionName=(.*?)(\\s|$)");
    }

    public static int getMajorVersion(String version) {
        return Integer.parseInt(version.split("\\.")[0]);
    }

    public static float getScaleFactor(AppiumDriver driver) {
        float scaleFactor = 1;
        if (driver instanceof IOSDriver) {
            scaleFactor = IOSHelper.getScaleFactor(driver);
        }

        return scaleFactor;
    }

    public static int getStatusBarHeight(AppiumDriver driver) {
        int statusBar = 0;
        if (driver instanceof AndroidDriver) {
            statusBar = AndroidHelper.getStatusBarHeight(driver);
        }

        if (driver instanceof IOSDriver) {
            statusBar = IOSHelper.getStatusBarHeight(driver);
        }

        return statusBar;
    }

    public static String grepBundleInfo(String packagePath, String aaptFolder, String infoPattern) {
        try {
            String command = MessageFormat.format("aapt dump badging \"{0}\"", packagePath);
            List<String> infoLines = ConsoleCommandBuilder.create(command).path(aaptFolder).execSync();
            Pattern idPattern = Pattern.compile(infoPattern);
            for (String lineI : infoLines) {
                Matcher idMatcher = idPattern.matcher(lineI);
                if (idMatcher.find()) {
                    return idMatcher.group(1);
                }
            }
            return null;
        } catch (IOException | InterruptedException error) {
            return null;
        }
    }

    public static String grepInstalledAppInfo(String deviceId, String appId, String adbFolder, String infoPattern) {
        try {
            String command = MessageFormat.format("adb -s {0} shell dumpsys package \"{1}\"", deviceId, appId);
            List<String> infoLines = ConsoleCommandBuilder.create(command).path(adbFolder).execSync();
            Pattern idPattern = Pattern.compile(infoPattern);
            for (String lineI : infoLines) {
                Matcher idMatcher = idPattern.matcher(lineI);
                if (idMatcher.find()) {
                    return idMatcher.group(1);
                }
            }
            return null;
        } catch (IOException | InterruptedException error) {
            return null;
        }
    }

    public static void holdAndSwipe(AppiumDriver driver, Point start, Point end, Duration duration) {
        var swipe = holdAndSwipeSequence("finger", start, end, duration);
        driver.perform(List.of(swipe));
    }

    public static Sequence holdAndSwipeSequence(String fingerId, Point start, Point end, Duration duration) {
        var finger = new PointerInput(PointerInput.Kind.TOUCH, fingerId);
        var swipe = new Sequence(finger, 1);

        swipe.addAction(finger.createPointerMove(Duration.ofMillis(0), PointerInput.Origin.viewport(), start));
        swipe.addAction(finger.createPointerDown(PointerInput.MouseButton.LEFT.asArg()));
        swipe.addAction(new Pause(finger, Duration.ofMillis(DEFAULT_LONG_TAP_DURATION)));
        swipe.addAction(finger.createPointerMove(duration, PointerInput.Origin.viewport(), end));
        swipe.addAction(finger.createPointerUp(PointerInput.MouseButton.LEFT.asArg()));
        return swipe;
    }

    public static boolean inAndroidAppInstalled(String deviceId, String appId, String adbFolder)
            throws IOException, InterruptedException {
        List<String> result = ConsoleCommandBuilder
                .create(MessageFormat.format("adb -s \"{0}\" shell pm path \"{1}\"", deviceId, appId))
                .path(adbFolder)
                .execSync();
        return ProcessUtil.includes(result, appId);
    }

    public static void injectImage(AppiumDriver driver, String imageFilePath) throws StepFailedException {
        File file = new File(imageFilePath);
        if (!file.exists()) {
            throw new StepFailedException(MessageFormat.format(StringConstants.KW_MSG_FILE_NOT_FOUND, imageFilePath));
        }

        long fileSizeInByte = file.length();
        if (fileSizeInByte > MAX_FILE_SIZE_MB * (1024 * 1024)) {
            throw new StepFailedException(
                    MessageFormat.format(StringConstants.KW_MSG_FILE_SIZE_IS_EXCEEDING, MAX_FILE_SIZE_MB));
        }

        if (!isSupportedFileType(file)) {
            throw new StepFailedException(StringConstants.KW_MSG_FILE_TYPE_IS_NOT_SUPPORTED);
        }

        try {
            byte[] byteArray = Files.readAllBytes(file.toPath());
            String encodedImage = Base64.getEncoder().encodeToString(byteArray);
            driver.executeScript("image-injection=" + encodedImage);
        } catch (Exception e) {
            logger.logWarning(e.getMessage());
            throw new StepFailedException(StringConstants.KW_LOG_FAILED_INJECT_IMAGE);
        }
    }

    public static boolean isIPad(String deviceModel) {
        return deviceModel.contains("iPad");
    }

    public static boolean isIPhoneXOrLater(String deviceModel) {
        if (!deviceModel.contains("iPhone")) {
            return false;
        }

        String[] versionNumbers = deviceModel.replace("iPhone", "").split(",");
        int majorVersion = Integer.parseInt(versionNumbers[0]);
        int minorVersion = Integer.parseInt(versionNumbers[1]);

        // https://www.theiphonewiki.com/wiki/Models
        return (majorVersion >= 11 || (majorVersion == 10 && (minorVersion == 3 || minorVersion == 6)));
    }

    public static boolean isSameApp(String packagePath, String deviceId, String appId, String adbFolder,
            String aaptFolder) {
        return isSameVersion(packagePath, deviceId, appId, adbFolder, aaptFolder)
                && isSameSize(packagePath, deviceId, appId, adbFolder);
    }

    public static boolean isSameSize(String packagePath, String deviceId, String appId, String adbFolder) {
        long installedPackageSize = getAndroidPackageSize(deviceId, appId, adbFolder);
        try {
            return Files.size(new File(packagePath).toPath()) == installedPackageSize;
        } catch (IOException error) {
            return false;
        }
    }

    public static boolean isSameVersion(String packagePath, String deviceId, String appId, String adbFolder,
            String aaptFolder) {
        String bundleVersion = getBundleVersion(packagePath, aaptFolder);
        String installedAppVersion = getInstalledAppVersion(deviceId, appId, adbFolder);
        return StringUtils.equals(bundleVersion, installedAppVersion);
    }

    private static boolean isSupportedFileType(File file) {
        String fileName = file.getName();
        String fileExtension = fileName.substring(fileName.lastIndexOf('.') + 1).toLowerCase();
        return Arrays.asList(SUPPORTED_IMAGE_TYPES).contains(fileExtension);
    }

    public static void pinch(AppiumDriver driver, Point start1, Point start2, Point end1, Point end2) {
        pinch(driver, start1, start2, end1, end2, Duration.ofMillis(DEFAULT_PINCH_DURATION));
    }

    public static void pinch(AppiumDriver driver, Point start1, Point start2, Point end1, Point end2,
            Duration duration) {
        var swipe1 = swipeSequence("finger1", start1, end1, duration);
        var swipe2 = swipeSequence("finger2", start2, end2, duration);

        driver.perform(List.of(swipe1, swipe2));
    }

    private static void registerHealingObject(FindElementsResult findResult, TestObject testObject) {
        if (testObject == null || StringUtils.isEmpty(testObject.getObjectId())) {
            return;
        }

        BrokenTestObject brokenTestObject = SelfHealingController.registerBrokenTestObject(findResult, testObject);
        LocatorStrategy brokenLocatorStrategy = LocatorStrategy.valueOf(brokenTestObject.getBrokenLocatorMethod());
        LocatorStrategy proposedLocatorStrategy = LocatorStrategy.valueOf(brokenTestObject.getProposedLocatorMethod());

        SelfHealingController.logWarning(MessageFormat.format(StringConstants.KW_LOG_INFO_PROPOSE_ALTERNATE_LOCATOR,
                testObject.getObjectId(), brokenLocatorStrategy.getName(), brokenTestObject.getBrokenLocator(),
                proposedLocatorStrategy.getName(), brokenTestObject.getProposedLocator()));
        KeywordExecutionContext.setHasHealedSomeObjects(true);
    }

    public static void setCommonAppiumSessionProperties(AppiumDriver driver) {
        AppiumDriverSession session = AppiumSessionCollector.getSession(driver);
        session.getProperties().put(PROPERTY_NAME_STATUS_BAR_HEIGHT, getStatusBarHeight(driver));
        session.getProperties().put(PROPERTY_NAME_SCALE_FACTOR, getScaleFactor(driver));
    }

    public static void swipe(AppiumDriver driver, Point start, Point end) {
        swipe(driver, start, end, Duration.ofMillis(DEFAULT_SWIPE_DURATION));
    }

    public static void swipe(AppiumDriver driver, Point start, Point end, Duration duration) {
        var swipe = swipeSequence("finger", start, end, duration);
        driver.perform(List.of(swipe));
    }

    public static Sequence swipeSequence(String fingerId, Point start, Point end, Duration duration) {
        var finger = new PointerInput(PointerInput.Kind.TOUCH, fingerId);
        var swipe = new Sequence(finger, 1);

        swipe.addAction(finger.createPointerMove(Duration.ofMillis(0), PointerInput.Origin.viewport(), start));
        swipe.addAction(finger.createPointerDown(PointerInput.MouseButton.LEFT.asArg()));
        swipe.addAction(new Pause(finger, Duration.ofMillis(DEFAULT_TAP_DURATION)));
        swipe.addAction(finger.createPointerMove(duration, PointerInput.Origin.viewport(), end));
        swipe.addAction(finger.createPointerUp(PointerInput.MouseButton.LEFT.asArg()));
        return swipe;
    }

    public static void tap(AppiumDriver driver, Point point) {
        if (driver instanceof AndroidDriver) {
            var finger = new PointerInput(PointerInput.Kind.TOUCH, "finger");
            var tap = new Sequence(finger, 0);

            tap.addAction(finger.createPointerMove(Duration.ofMillis(0), PointerInput.Origin.viewport(), point));
            tap.addAction(finger.createPointerDown(PointerInput.MouseButton.LEFT.asArg()));
            tap.addAction(finger.createPointerUp(PointerInput.MouseButton.LEFT.asArg()));

            driver.perform(List.of(tap));
        } else if (driver instanceof IOSDriver) {
            Map<String, Object> args = new HashMap<>();
            args.put("x", point.x);
            args.put("y", point.y);
            driver.executeScript("mobile: tap", args);
        }
    }

    public static void touchAndHold(AppiumDriver driver, Point point, Duration duration) {
        if (driver instanceof AndroidDriver) {
            var finger = new PointerInput(PointerInput.Kind.TOUCH, "finger");
            var tap = new Sequence(finger, 0);

            tap.addAction(finger.createPointerMove(Duration.ofMillis(0), PointerInput.Origin.viewport(), point));
            tap.addAction(finger.createPointerDown(PointerInput.MouseButton.LEFT.asArg()));
            tap.addAction(new Pause(finger, duration));
            tap.addAction(finger.createPointerUp(PointerInput.MouseButton.LEFT.asArg()));

            driver.perform(List.of(tap));
        } else if (driver instanceof IOSDriver) {
            Map<String, Object> args = new HashMap<>();
            args.put("duration", duration.getSeconds());
            args.put("x", point.x);
            args.put("y", point.y);
            driver.executeScript("mobile: touchAndHold", args);
        }
    }

    public static void uninstallAndroidApp(String deviceId, String appId, String adbFolder)
            throws IOException, InterruptedException {
        ConsoleCommandBuilder.create(MessageFormat.format("adb -s \"{0}\" uninstall \"{1}\"", deviceId, appId))
                .path(adbFolder)
                .execSync();
    }
}
