package com.kms.katalon.core.webui.authentication.basic;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.HasAuthentication;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kms.katalon.core.webui.driver.DriverFactory;
import com.kms.katalon.core.webui.driver.bidi.BiDiDriverUtil;
import com.kms.katalon.core.webui.util.WebDriverUtil;

public class BasicAuthenticationStrategyFactory {
    private static final Logger logger = LoggerFactory.getLogger(BasicAuthenticationStrategyFactory.class);

    private final String CHROME_BROWSER_NAME = "chrome";
    private final String FIREFOX_BROWSER_NAME = "firefox";
    private final String EDGE_BROWSER_NAME = "edge";
    

    public BasicAuthenticationStrategy getBasicAuthenticationStrategy() {
        WebDriver driver = DriverFactory.getWebDriver();
        String browserName = WebDriverUtil.getBrowserName(driver);
        if (!StringUtils.isEmpty(browserName)) {
            if (StringUtils.containsIgnoreCase(browserName, FIREFOX_BROWSER_NAME)) {
                if (useBiDi()) {
                    logger.info("Using BiDi to handle Basic authentication.");
                    return new BiDiBasicAuth();
                }
                logger.info("Using embedded credentials for Basic authentication.");
                return new EmbeddedCredentialsUrlBasicAuth();
            }

            if (
                StringUtils.containsIgnoreCase(browserName, CHROME_BROWSER_NAME) ||
                StringUtils.containsIgnoreCase(browserName, EDGE_BROWSER_NAME)
            ) {
                // Bypass basic auth using BiDi currently does not work on Chromium browser,
                // so use the Chrome Devtools Protocol instead.
                if (isCdpSupportedForAuthenticationDriver(driver)) {
                    logger.info("Using CDP to handle Basic authentication.");
                    return new CdpBasicAuth();
                }
                logger.info("Using embedded credentials for Basic authentication.");
                return new EmbeddedCredentialsUrlBasicAuth();
            }
        }
        
        return new NoOpBasicAuth();
    }
    
    private boolean useBiDi() {
        if (DriverFactory.getWebDriver() instanceof RemoteWebDriver remoteDriver) {
            return BiDiDriverUtil.getBiDiCapability(remoteDriver.getCapabilities());
        }
        
        return false;
    }

    private boolean isCdpSupportedForAuthenticationDriver(WebDriver driver) {
        return driver instanceof HasAuthentication;
    }
}
