package com.kms.katalon.core.webui.common.controller;

import java.io.IOException;
import java.util.LinkedHashMap;

import org.openqa.selenium.JavascriptException;
import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.WebElement;

import com.kms.katalon.core.webui.constants.ElementWaitingScript;
import com.kms.katalon.core.webui.driver.bidi.BiDiDriverUtil;
import com.kms.katalon.core.webui.model.ElementWaitingInteractableState;

public class ElementWaitingPhase {

    private final String INTERACTABLE_WAIT_UTIL_SCRIPT_NAME = "interactable-wait-util.js";

    protected JavascriptExecutor executor;

    public ElementWaitingPhase() {
        super();
    }

    public ElementWaitingPhase(JavascriptExecutor executor) {
        this.executor = executor;
    }

    protected String getInteractableWaitUtilScript() throws IOException {
        return BiDiDriverUtil.getScriptContent(INTERACTABLE_WAIT_UTIL_SCRIPT_NAME);
    }

    /**
     * Already assign element variable in the ADD_DOM_OBSERVER_SCRIPT
     * 
     * @return String
     */
    protected String getWaitingScript() {
        return ElementWaitingScript.ADD_DOM_OBSERVER_SCRIPT + """
                const notifyWaitingResult = arguments[arguments.length - 1];
                const element = arguments[0];
                const timeout = arguments[1];

                const ElementWaitingInteractableState = {
                    DONE: 'DONE',
                    ELEMENT_DETACHED: 'ELEMENT_DETACHED',
                    TIMED_OUT: 'TIMED_OUT',
                    ERROR: 'ERROR',
                };

                function notifyWaitingError(error) {
                    notifyWaitingResult({ result: ElementWaitingInteractableState.ERROR, error: error.message });
                }


                // Wait for detached
                waitWithDOMMutation(() => {
                    if (!element || !element.isConnected) {
                        return true;
                    }
                    return false;
                }, timeout).then((isDetached) => {
                    notifyWaitingResult({ result: isDetached
                        ? ElementWaitingInteractableState.ELEMENT_DETACHED
                        : ElementWaitingInteractableState.TIMED_OUT });
                }).catch(notifyWaitingError);
                """;
    }

    /**
     * @param element
     * @param timeoutInMillis
     * @throws IllegalStateException if the element is not found in the DOM or the element was removed from the DOM
     * before the waiting script completed
     */
    public boolean wait(WebElement element, long timeoutInMillis) {
        if (executor == null) {
            throw new IllegalStateException("JavascriptExecutor is not found.");
        }
        String runningScript = getWaitingScript();
        try {
            Object result = ((JavascriptExecutor) this.executor).executeAsyncScript(runningScript, element,
                    timeoutInMillis);
            if (result instanceof LinkedHashMap resultMap && resultMap.containsKey("result")) {
                switch (ElementWaitingInteractableState.valueOf(resultMap.get("result").toString())) {
                    case DONE:
                        return true;
                    case ELEMENT_DETACHED:
                        throw new IllegalStateException("Element is detached from the DOM while waiting interactable");
                    case TIMED_OUT:
                        throw new IllegalStateException("Waiting for element timed out");
                    case ERROR:
                        throw new IllegalStateException("Error during waiting interactable: " + resultMap.get("error"));
                    default:
                        return false;
                }
            }
            return false;
        } catch (JavascriptException e) {
            throw new IllegalStateException("Error executing waiting script: " + e.getMessage(), e);
        }
    }
}
