package com.kms.katalon.core.webui.driver;

import java.io.IOException;
import java.io.PrintStream;
import java.net.Socket;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.openqa.selenium.Alert;
import org.openqa.selenium.BuildInfo;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.NoAlertPresentException;
import org.openqa.selenium.NoSuchWindowException;
import org.openqa.selenium.UnsupportedCommandException;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebDriver.Timeouts;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.bidi.HasBiDi;
import org.openqa.selenium.edge.EdgeDriver;
import org.openqa.selenium.firefox.FirefoxOptions;
import org.openqa.selenium.firefox.FirefoxProfile;
import org.openqa.selenium.remote.CommandExecutor;
import org.openqa.selenium.remote.HttpCommandExecutor;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.openqa.selenium.remote.SessionId;
import org.openqa.selenium.remote.UnreachableBrowserException;
import org.openqa.selenium.safari.SafariDriver;
import org.openqa.selenium.support.decorators.Decorated;

import com.google.gson.JsonIOException;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.driver.IDriverType;
import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.logging.LogLevel;
import com.kms.katalon.core.model.RunningMode;
import com.kms.katalon.core.network.ProxyInformation;
import com.kms.katalon.core.util.TestCloudPropertyUtil;
import com.kms.katalon.core.util.internal.ExceptionsUtil;
import com.kms.katalon.core.util.internal.JsonUtil;
import com.kms.katalon.core.webui.common.WebUiCommonHelper;
import com.kms.katalon.core.webui.constants.CoreWebuiMessageConstants;
import com.kms.katalon.core.webui.constants.StringConstants;
import com.kms.katalon.core.webui.driver.chrome.ChromeDriverBuilder;
import com.kms.katalon.core.webui.driver.edge.EdgeChromiumDriverBuilder;
import com.kms.katalon.core.webui.driver.firefox.FirefoxDriverBuilder;
import com.kms.katalon.core.webui.driver.remote.GenericRemoteDriverBuilder;
import com.kms.katalon.core.webui.driver.safari.SafariDriverBuilder;
import com.kms.katalon.core.webui.exception.BrowserNotOpenedException;
import com.kms.katalon.core.webui.util.URLUtils;
import com.kms.katalon.core.webui.util.WebDriverPropertyUtil;
import com.kms.katalon.logging.LogUtil;
import com.kms.katalon.selenium.constant.SeleniumW3CCapabilityConstant;
import com.kms.katalon.testcloud.core.constants.TestCloudStringConstants;
import com.kms.katalon.util.CryptoUtil;

public class DriverFactory {

    private static final KeywordLogger logger = KeywordLogger.getInstance(DriverFactory.class);

    public static final String WEB_UI_DRIVER_PROPERTY = StringConstants.CONF_PROPERTY_WEBUI_DRIVER;

    public static final String MOBILE_DRIVER_PROPERTY = StringConstants.CONF_PROPERTY_MOBILE_DRIVER;

    public static final String EXISTING_DRIVER_PROPERTY = StringConstants.CONF_PROPERTY_EXISTING_DRIVER;

    public static final String CHROME_DRIVER_PATH_PROPERTY_KEY = "webdriver.chrome.driver";

    public static final String FIREFOX_DRIVER_PATH_PROPERTY_KEY = "webdriver.gecko.driver";

    // Temp error constant message for issues
    // https://code.google.com/p/selenium/issues/detail?id=7977
    private static final String JAVA_SCRIPT_ERROR_H_IS_NULL_MESSAGE = "[JavaScript Error: \"h is null\"";

    public static final String EDGE_CHROMIUM_DRIVER_PATH_PROPERTY = StringConstants.CONF_PROPERTY_EDGE_CHROMIUM_DRIVER_PATH;

    public static final String CHROME_DRIVER_PATH_PROPERTY = StringConstants.CONF_PROPERTY_CHROME_DRIVER_PATH;

    public static final String ENABLE_PAGE_LOAD_TIMEOUT = StringConstants.CONF_PROPERTY_ENABLE_PAGE_LOAD_TIMEOUT;

    public static final String DEFAULT_PAGE_LOAD_TIMEOUT = StringConstants.CONF_PROPERTY_DEFAULT_PAGE_LOAD_TIMEOUT;

    public static final String ACTION_DELAY = StringConstants.CONF_PROPERTY_ACTION_DELAY;

    public static final String USE_ACTION_DELAY_IN_SECOND = StringConstants.CONF_PROPERTY_USE_ACTION_DELAY_IN_SECOND;

    public static final String IGNORE_PAGE_LOAD_TIMEOUT_EXCEPTION = StringConstants.CONF_PROPERTY_IGNORE_PAGE_LOAD_TIMEOUT_EXCEPTION;

    public static final String EXECUTED_BROWSER_PROPERTY = StringConstants.CONF_PROPERTY_EXECUTED_BROWSER;

    public static final String REMOTE_WEB_DRIVER_URL = StringConstants.CONF_PROPERTY_REMOTE_WEB_DRIVER_URL;

    public static final String REMOTE_WEB_DRIVER_TYPE = StringConstants.CONF_PROPERTY_REMOTE_WEB_DRIVER_TYPE;

    public static final String IS_REMOTE_WEB_DRIVER_URL_ENCRYPTED = "isEncrypted";

    public static final String REMOTE_MOBILE_DRIVER = "remoteMobileDriver";

    public static final String DEBUG_PORT = "debugPort";

    public static final String DEBUG_HOST = "debugHost";

    public static final String DEFAULT_DEBUG_HOST = "localhost";

    private static final ThreadLocal<WebDriver> localWebServerStorage = new ThreadLocal<WebDriver>() {
        @Override
        protected WebDriver initialValue() {
            return null;
        }
    };

    private static IDriverConfigurationProvider driverConfigProvider = new DriverConfigurationProvider(logger);

    public static void setDriverConfigurationProvider(IDriverConfigurationProvider provider) {
        driverConfigProvider = provider;
    }

    public static IDriverConfigurationProvider getDriverConfigurationProvider() {
        return driverConfigProvider;
    }

    /**
     * Open a new web driver based on the execution configuration.
     * 
     * @return An instance of {@link WebDriver}
     * @throws Exception
     */
    public static WebDriver openWebDriver() throws Exception {
        try {
            WebDriver webDriver;
            if (isUsingExistingDriver()) {
                webDriver = startExistingBrowser();
            } else {
                String remoteWebDriverUrl = getRemoteWebDriverServerUrl();
                if (StringUtils.isNotEmpty(remoteWebDriverUrl)) {
                    webDriver = startRemoteBrowser();
                } else {
                    webDriver = startNewBrowser(getExecutedBrowser());
                }

                saveWebDriverSessionData(webDriver);
                changeWebDriver(webDriver);
            }

            return webDriver;
        } catch (Error e) {
            logger.logMessage(LogLevel.WARNING, e.getMessage(), e);
            throw new StepFailedException(e);
        }
    }

    /**
     * Change the web driver that Katalon uses to execute your test to the given web
     * driver instance.
     * 
     * @param webDriver
     */
    public static void changeWebDriver(WebDriver webDriver) {
        logBrowserRunData(webDriver);
        changeWebDriverWithoutLog(webDriver);
    }

    private static void changeWebDriverWithoutLog(WebDriver webDriver) {
        localWebServerStorage.set(webDriver);
        RunConfiguration.storeDriver(webDriver);
        setTimeout();
    }

    private static boolean isUsingExistingDriver() {
        return RunConfiguration.getDriverSystemProperties(EXISTING_DRIVER_PROPERTY) != null;
    }

    private static WebDriver startRemoteBrowser() throws Exception {
        if (null != localWebServerStorage.get() && null != getRemoteSessionId(localWebServerStorage.get())) {
            logger.logWarning(StringConstants.DRI_LOG_WARNING_BROWSER_ALREADY_OPENED);
            closeWebDriver();
        }

        WebUIDriverType driver = WebUIDriverType.REMOTE_WEB_DRIVER;
        String remoteServerUrl = getRemoteWebDriverServerUrl();
        if (StringUtils.isEmpty(remoteServerUrl)) {
            return null;
        }

        Map<String, Object> driverPreferenceProps = RunConfiguration
                .getDriverPreferencesProperties(RunConfiguration.REMOTE_DRIVER_PROPERTY);
        MutableCapabilities mutableCapabilities = new MutableCapabilities();

        if (driverPreferenceProps != null) {
            mutableCapabilities = WebDriverPropertyUtil.toDesireCapabilities(driverPreferenceProps, driver);
        }

        var webDriver = new GenericRemoteDriverBuilder().capabilities(mutableCapabilities)
                .withSmartLocator(WebUiCommonHelper.smartLocatorEnabled())
                .withAccessibilityTree(RunConfiguration.getWebAccessibilityTreeEnabled())
                .withSmartWait(globalSmartWaitEnabled())
                .withFlutterAppTesting(RunConfiguration.isFlutterAppTestingEnabled())
                .withCanvasTextExtraction(RunConfiguration.isCanvasTextExtractionEnabled())
                .withClosedShadowDOMTesting(RunConfiguration.isClosedShadowDOMTestingEnabled())
                .build();

        return webDriver;
    }

    private static WebDriver startNewBrowser(IDriverType executedBrowser) throws Exception {
        WebUIDriverType driverType = (WebUIDriverType) executedBrowser;
        if (driverType == null) {
            throw new StepFailedException(StringConstants.DRI_ERROR_MSG_NO_BROWSER_SET);
        }

        if (null != localWebServerStorage.get() && null != getRemoteSessionId(localWebServerStorage.get())) {
            logger.logWarning(StringConstants.DRI_LOG_WARNING_BROWSER_ALREADY_OPENED);
            closeWebDriver();
        }

        logger.logInfo(MessageFormat.format(StringConstants.XML_LOG_STARTING_DRIVER_X, driverType.toString()));

        Map<String, Object> driverPreferenceProps = RunConfiguration
                .getDriverPreferencesProperties(WEB_UI_DRIVER_PROPERTY);
        MutableCapabilities capabilities = null;
        if (driverPreferenceProps != null) {
            capabilities = WebDriverPropertyUtil.toDesireCapabilities(driverPreferenceProps, driverType);
        }

        if (capabilities == null) {
            capabilities = new MutableCapabilities();
        }

        var driver = DriverBuilderFactory.getDriverBuilder(driverType)
                .capabilities(capabilities)
                .withSmartLocator(WebUiCommonHelper.smartLocatorEnabled())
                .withAccessibilityTree(RunConfiguration.getWebAccessibilityTreeEnabled())
                .withSmartWait(globalSmartWaitEnabled())
                .withFlutterAppTesting(RunConfiguration.isFlutterAppTestingEnabled())
                .withCanvasTextExtraction(RunConfiguration.isCanvasTextExtractionEnabled())
                .withClosedShadowDOMTesting(RunConfiguration.isClosedShadowDOMTestingEnabled())
                .build();

        checkTimeCapsuleAvailable(driverType);

        return driver;
    }

    /**
     * Check whether the Time Capsule is enabled or not, if yes, continue to check
     * DriverType is Chrome or not. Because Katalon support Time Capsule for Chrome
     * Browser only.
     * 
     * @param driver
     * @return true if Time Capsule is enabled and DriverType is Chrome. Otherwise,
     * return false.
     */
    private static boolean checkTimeCapsuleAvailable(WebUIDriverType driver) {
        if (!RunConfiguration.shouldApplyTimeCapsule()) {
            return false;
        }

        switch (driver) {
            case CHROME_DRIVER:
            case HEADLESS_DRIVER:
            case EDGE_CHROMIUM_DRIVER:
                return true;
            default:
                KeywordLogger.getInstance(KatalonSmartEventListener.class)
                        .logWarning(CoreWebuiMessageConstants.MSG_TIME_CAPSULE_IS_NOT_SUPPORTED_ON_CURRENT_BROWSER);
                return false;
        }
    }

    private static boolean globalSmartWaitEnabled() {
        // Default to false if previously an error occurs in writing the value in
        // RunConfiguration
        boolean globalSmartWaitEnabled = (boolean) Optional
                .ofNullable(RunConfiguration.getExecutionProperties().get(RunConfiguration.GLOBAL_SMART_WAIT_MODE))
                .orElse(false);
        return globalSmartWaitEnabled;
    }

    private static void saveWebDriverSessionData(WebDriver webDriver) {
        if (!(webDriver instanceof RemoteWebDriver) || webDriver instanceof ExistingRemoteWebDriver
                || webDriver instanceof SafariDriver) {
            return;
        }

        // KRE will be hangs if session exceed 50. So that we do not store session with
        // Linux
        // KRE does not need get session to display
        if (RunConfiguration.getRunningMode() == RunningMode.CONSOLE) {
            return;
        }

        RemoteWebDriver remoteWebDriver = (RemoteWebDriver) webDriver;
        Socket myClient = null;
        PrintStream output = null;
        try {
            myClient = new Socket(RunConfiguration.getSessionServerHost(), RunConfiguration.getSessionServerPort());
            output = new PrintStream(myClient.getOutputStream());
            output.println(remoteWebDriver.getSessionId());
            output.println(getWebDriverServerUrl(remoteWebDriver));
            IDriverType remoteDriverType = getExecutedBrowser();
            output.println(remoteDriverType);
            output.println(RunConfiguration.getLogFolderPath());
            if (remoteDriverType == WebUIDriverType.ANDROID_DRIVER) {
                output.println(WebMobileDriverFactory.getDeviceManufacturer() + " "
                        + WebMobileDriverFactory.getDeviceModel() + " " + WebMobileDriverFactory.getDeviceOSVersion());
            } else if (remoteDriverType == WebUIDriverType.IOS_DRIVER) {
                output.println(
                        WebMobileDriverFactory.getDeviceName() + " " + WebMobileDriverFactory.getDeviceOSVersion());
            }

            if (webDriver instanceof HasBiDi webDriverBidi && webDriverBidi.maybeGetBiDi().isPresent()) {
                output.println(remoteWebDriver.getCapabilities()
                        .getCapability(SeleniumW3CCapabilityConstant.WEB_SOCKET_URL_CAP));
            } else {
                output.println();
            }

            output.flush();
        } catch (Exception e) {
            // Ignore for this exception
        } finally {
            if (myClient != null) {
                try {
                    myClient.close();
                } catch (IOException e) {
                    // Ignore for this exception
                }
            }
            if (output != null) {
                output.close();
            }
        }
    }

    public static String getWebDriverServerUrl(RemoteWebDriver remoteWebDriver) {
        CommandExecutor commandExecutor = remoteWebDriver.getCommandExecutor();
        if (commandExecutor instanceof HttpCommandExecutor) {
            return ((HttpCommandExecutor) commandExecutor).getAddressOfRemoteServer().toString();
        }
        return StringUtils.EMPTY;
    }

    protected static WebDriver startExistingBrowser() throws Exception {
        String remoteDriverType = RunConfiguration.getExistingSessionDriverType();
        String sessionId = RunConfiguration.getExistingSessionSessionId();
        String remoteServerUrl = RunConfiguration.getExistingSessionServerUrl();
        String webSocketUrl = RunConfiguration.getExistingSessionWebSocketUrl();
        WebDriver webDriver;

        if (WebUIDriverType.ANDROID_DRIVER.toString().equals(remoteDriverType)
                || WebUIDriverType.IOS_DRIVER.toString().equals(remoteDriverType)) {
            webDriver = WebMobileDriverFactory.startExistingMobileDriver(
                    WebUIDriverType.fromStringValue(remoteDriverType), sessionId, remoteServerUrl);
        } else {
            webDriver = new ExistingSessionDriverBuilder().remoteServerUrl(remoteServerUrl)
                    .sessionId(sessionId)
                    .webSocketUrl(webSocketUrl)
                    .withSmartLocator(WebUiCommonHelper.smartLocatorEnabled())
                    .withAccessibilityTree(RunConfiguration.getWebAccessibilityTreeEnabled())
                    .withSmartWait(globalSmartWaitEnabled())
                    .withFlutterAppTesting(RunConfiguration.isFlutterAppTestingEnabled())
                    .withCanvasTextExtraction(RunConfiguration.isCanvasTextExtractionEnabled())
                    .withClosedShadowDOMTesting(RunConfiguration.isClosedShadowDOMTestingEnabled())
                    .build();
        }

        return webDriver;
    }

    private static void logBrowserRunData(WebDriver webDriver) {
        if (webDriver == null) {
            return;
        }

        String serverUrl = getRemoteWebDriverServerUrl();
        if (StringUtils.isNotEmpty(serverUrl) && webDriver instanceof RemoteWebDriver) {
            String testCloudRemoteOS = RunConfiguration.getDriverSystemProperty(RunConfiguration.REMOTE_DRIVER_PROPERTY,
                    "testCloudRemoteOS");

            RemoteWebDriver remoteDriver = (RemoteWebDriver) webDriver;
            Capabilities driverCapabilities = remoteDriver.getCapabilities();
            Map<String, Object> driverCapabilitiesMapAsDisplay = new HashMap<String, Object>(
                    driverCapabilities.asMap());
            Map<String, Object> driverPreferenceProps = RunConfiguration
                    .getDriverPreferencesProperties(RunConfiguration.REMOTE_DRIVER_PROPERTY);
            MutableCapabilities inputDesireCapibilities = null;
            if (driverPreferenceProps != null) {
                inputDesireCapibilities = new MutableCapabilities(driverPreferenceProps);

                if (testCloudRemoteOS != null) {
                    driverCapabilitiesMapAsDisplay.put("deviceName", driverPreferenceProps.get("deviceName"));

                    String browserVersion = WebUiCommonHelper.getBrowserVersion(webDriver);

                    if (browserVersion != null && !browserVersion.isEmpty()) {
                        driverCapabilitiesMapAsDisplay.put(SeleniumW3CCapabilityConstant.BROWSER_VERSION_CAP,
                                browserVersion);
                    }
                }

                // Extract browserVersionType from driverPreferenceProps
                Object katalonOptionsObj = driverPreferenceProps
                        .get(TestCloudStringConstants.CAPABILITY_KATALON_OPTIONS);
                if (katalonOptionsObj instanceof Map<?, ?> katalonOptions) {
                    Object browserVersionTypeObj = (String) katalonOptions
                            .get(TestCloudStringConstants.CAPABILITY_BROWSER_VERSION_TYPE);
                    logger.logRunData(TestCloudStringConstants.CAPABILITY_BROWSER_VERSION_TYPE,
                            browserVersionTypeObj != null ? browserVersionTypeObj.toString() : StringUtils.EMPTY);
                }
            }

            if (testCloudRemoteOS != null) {
                logger.logRunData("remoteDriverUrl", URLUtils.removeCrendentialsFromURL(serverUrl));
                logger.logRunData("remoteOS", testCloudRemoteOS);
                TestCloudPropertyUtil.getInstance().setRunFromTestCloud(driverCapabilitiesMapAsDisplay);
            } else {
                if (!TestCloudPropertyUtil.getInstance().isRunFromTestCloud(inputDesireCapibilities)) {
                    logger.logRunData("remoteDriverUrl", serverUrl);
                }
            }

            if (TestCloudPropertyUtil.getInstance().isRunFromTestCloud()) {
                String log = JsonUtil.toJson(TestCloudPropertyUtil.getInstance()
                        .hideTestCloudSensitiveDesiredCapabilities(driverCapabilitiesMapAsDisplay));
                log = TestCloudPropertyUtil.getInstance().hideSauceLabsDocumentLink(log);
                logger.logRunData("desiredCapabilities", log);
                Object katalonCaps = driverCapabilitiesMapAsDisplay.get(TestCloudPropertyUtil.KATALON_CAPS);
                if (katalonCaps != null && katalonCaps instanceof Map<?, ?>) {
                    String katalonCapsJson = JsonUtil.toJson(katalonCaps);
                    logger.logRunData(TestCloudPropertyUtil.KATALON_CAPS, katalonCapsJson);
                    Map<?, ?> katalonCapsMap = (Map<?, ?>) katalonCaps;
                    if (katalonCapsMap.containsKey(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY)) {
                        String deviceName = (String) katalonCapsMap.get(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY);
                        logger.logRunData(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY, deviceName);
                    }
                }
            } else {
                try {
                    logger.logRunData("desiredCapabilities", JsonUtil.toJson(driverCapabilitiesMapAsDisplay));
                } catch (JsonIOException jse) {
                    try {
                        logger.logRunData("desiredCapabilities",
                                JsonUtil.toJson(driverCapabilitiesMapAsDisplay, true, true));
                    } catch (JsonIOException jsex) {
                        logger.logInfo("Cannot parse driverCapabilities to json - " + jsex.getMessage());
                    }
                }
            }
        }

        logger.logRunData("sessionId", getRemoteSessionId(webDriver).toString());
        logger.logRunData(StringConstants.BROWSER_CAPABILITY_NAME, getBrowserVersion(webDriver));
        logger.logRunData("platform",
                webDriver.getClass() == RemoteWebDriver.class
                        ? ((RemoteWebDriver) webDriver).getCapabilities().getPlatformName().toString()
                        : System.getProperty("os.name"));
        logger.logRunData(StringConstants.XML_LOG_SELENIUM_VERSION, new BuildInfo().getReleaseLabel());

        ProxyInformation proxyInfo = RunConfiguration.getProxyInformation();
        proxyInfo.setPassword("******");
        logger.logRunData("proxyInformation", proxyInfo.toString());
    }

    public static String getBrowserVersion(WebDriver webDriver) {
        try {
            return WebUiCommonHelper.getBrowserAndVersion(unwrapWebDriver(webDriver));
        } catch (Exception e) {
            e = ExceptionsUtil.customException(e);
            e.printStackTrace();
            return "";
        }
    }

    public static WebDriver openWebDriver(IDriverType driver, Object options) throws Exception {
        try {
            if (!(driver instanceof WebUIDriverType webUIDriver)) {
                return null;
            }

            closeWebDriver();
            WebDriver webDriver = null;
            switch (webUIDriver) {
                case FIREFOX_DRIVER:
                    if (options instanceof FirefoxOptions firefoxOptions) {
                        webDriver = new FirefoxDriverBuilder().options(firefoxOptions).build();
                    } else if (options instanceof FirefoxProfile firefoxProfile) {
                        webDriver = new FirefoxDriverBuilder().profile(firefoxProfile).build();
                    } else if (options instanceof MutableCapabilities capabilities) {
                        webDriver = new FirefoxDriverBuilder().capabilities(capabilities).build();
                    } else {
                        webDriver = new FirefoxDriverBuilder().build();
                    }
                    break;
                case SAFARI_DRIVER:
                    if (options instanceof MutableCapabilities capabilities) {
                        webDriver = new SafariDriverBuilder().capabilities(capabilities).build();
                    }
                    break;
                case CHROME_DRIVER:
                    if (options instanceof MutableCapabilities capabilities) {
                        webDriver = new ChromeDriverBuilder().capabilities(capabilities).build();
                    }
                    break;
                case EDGE_CHROMIUM_DRIVER:
                    if (options instanceof MutableCapabilities capabilities) {
                        webDriver = new EdgeChromiumDriverBuilder().capabilities(capabilities).build();
                    }
                    break;
                default:
                    throw new StepFailedException(
                            MessageFormat.format(StringConstants.DRI_ERROR_DRIVER_X_NOT_IMPLEMENTED, driver.getName()));
            }
            localWebServerStorage.set(webDriver);
            setTimeout();
            return webDriver;
        } catch (Error e) {
            logger.logMessage(LogLevel.WARNING, e.getMessage(), e);
            throw new StepFailedException(e);
        }
    }

    private static void setTimeout() {
        WebDriver webDriver = localWebServerStorage.get();
        if (webDriver instanceof EdgeDriver) {
            return;
        }

        Timeouts timeouts = webDriver.manage().timeouts();
        timeouts.implicitlyWait(Duration.ofSeconds(0));
        if (isEnablePageLoadTimeout()) {
            timeouts.pageLoadTimeout(Duration.ofSeconds(getDefaultPageLoadTimeout()));
        }
    }

    /**
     * Get the current active web driver
     * 
     * @return the current active WebDriver
     * @throws StepFailedException
     * @throws WebDriverException
     */
    public static WebDriver getWebDriver() throws StepFailedException, WebDriverException {
        try {
            verifyWebDriver();
        } catch (BrowserNotOpenedException e) {
            for (Object driverObject : RunConfiguration.getStoredDrivers()) {
                if (!(driverObject instanceof WebDriver webDriver)) {
                    continue;
                }

                var unwrapped = unwrapWebDriver(webDriver);
                if (unwrapped instanceof RemoteWebDriver) {
                    // return original driver
                    return webDriver;
                }
            }

            throw e;
        }

        return localWebServerStorage.get();
    }

    public static WebDriver getUnwrappedWebDriver() throws StepFailedException, WebDriverException {
        return unwrapWebDriver(getWebDriver());
    }

    private static void verifyWebDriver() throws StepFailedException, WebDriverException {
        startExistingBrowserIfPossible();
        verifyWebDriverIsOpen();
        try {
            if (null == getRemoteSessionId(localWebServerStorage.get())) {
                switchToAvailableWindow();
            }
        } catch (WebDriverException e) {
            if (!(e instanceof NoSuchWindowException) && e.getMessage() != null
                    && !e.getMessage().startsWith(JAVA_SCRIPT_ERROR_H_IS_NULL_MESSAGE)) {
                throw e;
            }
        }
    }

    private static void verifyWebDriverIsOpen() throws WebDriverException {
        if (localWebServerStorage.get() == null) {
            throw new BrowserNotOpenedException();
        }
    }

    /**
     * Get the current alert if there is one popped up
     * 
     * @return the current alert if there is one popped up, or null it there is none
     * @throws WebDriverException
     */
    public static Alert getAlert() throws WebDriverException {
        verifyWebDriverIsOpen();
        Alert alert = null;
        try {
            try {
                alert = localWebServerStorage.get().switchTo().alert();
            } catch (Exception e) {
                if (!(e instanceof NoSuchWindowException) && e.getMessage() != null
                        && !e.getMessage().startsWith(JAVA_SCRIPT_ERROR_H_IS_NULL_MESSAGE)) {
                    throw e;
                }
                switchToAvailableWindow();
                alert = localWebServerStorage.get().switchTo().alert();
            }

            return alert;
        } catch (NoAlertPresentException ex) {
            return null;
        }
    }

    /**
     * Get the current alert if there is one popped up
     * 
     * @return the current alert if there is one popped up, or null it there is none
     * @throws WebDriverException
     */
    public static Alert getAlert(WebDriver driver) throws WebDriverException {
        if (driver == null) {
            throw new BrowserNotOpenedException();
        }

        Alert alert = null;
        try {
            try {
                alert = driver.switchTo().alert();
            } catch (Exception e) {
                if (!(e instanceof NoSuchWindowException) && e.getMessage() != null
                        && !e.getMessage().startsWith(JAVA_SCRIPT_ERROR_H_IS_NULL_MESSAGE)) {
                    throw e;
                }
                switchToAvailableWindow(driver);
                alert = driver.switchTo().alert();
            }

            return alert;
        } catch (NoAlertPresentException ex) {
            return null;
        }
    }

    /**
     * Wait for an alert to pop up for a specific time
     * 
     * @param timeOut the timeout to wait for the alert (in milliseconds)
     * @return
     */
    public static boolean waitForAlert(int timeOut) {
        verifyWebDriverIsOpen();
        float count = 0;
        long miliseconds = System.currentTimeMillis();
        while (count < timeOut) {
            Alert alert = getAlert();
            if (alert != null) {
                return true;
            }
            count += ((System.currentTimeMillis() - miliseconds) / 1000);
            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                // Thread is interrupted, do nothing
            }
            count += 0.5;
            miliseconds = System.currentTimeMillis();
        }

        return false;
    }

    /**
     * Switch the active web driver to any available window
     */
    public static void switchToAvailableWindow() {
        verifyWebDriverIsOpen();
        try {
            localWebServerStorage.get().switchTo().window("");
        } catch (WebDriverException e) {
            if (!(e instanceof NoSuchWindowException) && e.getMessage() != null
                    && !e.getMessage().startsWith(JAVA_SCRIPT_ERROR_H_IS_NULL_MESSAGE)) {
                throw e;
            }
            // default window is closed, try to switch to available window
            Set<String> availableWindows = localWebServerStorage.get().getWindowHandles();
            for (String windowId : availableWindows) {
                try {
                    localWebServerStorage.get().switchTo().window(windowId);
                    return;
                } catch (WebDriverException exception) {
                    if (!(exception instanceof NoSuchWindowException) && e.getMessage() != null
                            && !exception.getMessage().startsWith(JAVA_SCRIPT_ERROR_H_IS_NULL_MESSAGE)) {
                        throw exception;
                    }
                    continue;
                }
            }
        }
    }

    /**
     * Switch the active web driver to any available window
     */
    public static void switchToAvailableWindow(WebDriver driver) {
        try {
            driver.switchTo().window("");
        } catch (WebDriverException e) {
            if (!(e instanceof NoSuchWindowException) && e.getMessage() != null
                    && !e.getMessage().startsWith(JAVA_SCRIPT_ERROR_H_IS_NULL_MESSAGE)) {
                throw e;
            }

            // default window is closed, try to switch to available window
            Set<String> availableWindows = driver.getWindowHandles();
            for (String windowId : availableWindows) {
                try {
                    driver.switchTo().window(windowId);
                    return;
                } catch (WebDriverException exception) {
                    if (!(exception instanceof NoSuchWindowException) && e.getMessage() != null
                            && !exception.getMessage().startsWith(JAVA_SCRIPT_ERROR_H_IS_NULL_MESSAGE)) {
                        throw exception;
                    }
                    continue;
                }
            }
        }
    }

    /**
     * Get the index of the window the web driver is on
     * 
     * @return the index of the window the web driver is on
     */
    public static int getCurrentWindowIndex() {
        verifyWebDriverIsOpen();
        String currentWindowHandle = localWebServerStorage.get().getWindowHandle();
        Set<String> availableWindowHandles = localWebServerStorage.get().getWindowHandles();
        int count = 0;
        for (String windowHandle : availableWindowHandles) {
            if (windowHandle.equals(currentWindowHandle)) {
                return count;
            }
            count++;
        }

        throw new StepFailedException(StringConstants.XML_LOG_ERROR_CANNOT_FOUND_WINDOW_HANDLE);
    }

    private static void startExistingBrowserIfPossible() {
        if (isUsingExistingDriver() && localWebServerStorage.get() == null) {
            try {
                WebDriver webDriver = startExistingBrowser();
                changeWebDriver(webDriver);
            } catch (Exception exception) {
                logger.logError("Exception while trying to start existing browser");
            }
        }
    }

    /**
     * Check if page load timeout is enabled
     * 
     * @return true if page load timeout is enabled; otherwise false
     */
    public static boolean isEnablePageLoadTimeout() {
        return RunConfiguration.getBooleanProperty(ENABLE_PAGE_LOAD_TIMEOUT,
                RunConfiguration.getExecutionGeneralProperties());
    }

    /**
     * Get the default page load timeout
     * 
     * @return the default page load timeout
     */
    public static int getDefaultPageLoadTimeout() {
        return RunConfiguration.getIntProperty(DEFAULT_PAGE_LOAD_TIMEOUT,
                RunConfiguration.getExecutionGeneralProperties());
    }

    /**
     * Get the action delay (in seconds)
     * 
     * @return the action delay
     */
    public static int getActionDelay() {
        return driverConfigProvider.getActionDelayInMilisecond();
    }

    /**
     * Check if ignoring the page load timeout exception
     * 
     * @return true if ignoring the page load timeout exception; otherwise false
     */
    public static boolean isIgnorePageLoadTimeoutException() {
        return RunConfiguration.getBooleanProperty(IGNORE_PAGE_LOAD_TIMEOUT_EXCEPTION,
                RunConfiguration.getExecutionGeneralProperties());
    }

    /**
     * Get the current executed browser type
     * 
     * @see WebUIDriverType
     * @return the current executed browser type as a {@link IDriverType} object
     */
    public static IDriverType getExecutedBrowser() {
        IDriverType webDriverType = null;
        if (isUsingExistingDriver()) {
            webDriverType = WebUIDriverType.fromStringValue(RunConfiguration.getExistingSessionDriverType());
        }

        if (webDriverType != null) {
            return webDriverType;
        }

        String remoteWebDriverUrl = getRemoteWebDriverServerUrl();
        String driverConnectorProperty = StringUtils.isNotBlank(remoteWebDriverUrl)
                ? RunConfiguration.REMOTE_DRIVER_PROPERTY : WEB_UI_DRIVER_PROPERTY;
        String driverTypeString = RunConfiguration.getDriverSystemProperty(driverConnectorProperty,
                EXECUTED_BROWSER_PROPERTY);
        if (driverTypeString != null) {
            webDriverType = WebUIDriverType.valueOf(driverTypeString);
        }

        if (webDriverType == null && RunConfiguration.getDriverSystemProperty(MOBILE_DRIVER_PROPERTY,
                WebMobileDriverFactory.EXECUTED_MOBILE_PLATFORM) != null) {
            webDriverType = WebUIDriverType.valueOf(RunConfiguration.getDriverSystemProperty(MOBILE_DRIVER_PROPERTY,
                    WebMobileDriverFactory.EXECUTED_MOBILE_PLATFORM));
        }

        return webDriverType;
    }

    /**
     * Get the url of the remove web driver is the current web driver type is remote
     * 
     * @return the url of the remove web driver is the current web driver type is
     * remote, or null if it is not
     */
    public static String getRemoteWebDriverServerUrl() {
        String remoteServerUrl = RunConfiguration.getDriverSystemProperty(RunConfiguration.REMOTE_DRIVER_PROPERTY,
                REMOTE_WEB_DRIVER_URL);
        Map<String, Object> jsonObjProperties = RunConfiguration
                .getDriverSystemProperties(RunConfiguration.REMOTE_DRIVER_PROPERTY);
        if (jsonObjProperties != null) {
            boolean isEncrypted = RunConfiguration.getBooleanProperty(IS_REMOTE_WEB_DRIVER_URL_ENCRYPTED,
                    jsonObjProperties);
            if (isEncrypted) {
                try {
                    remoteServerUrl = CryptoUtil.decode(CryptoUtil.getDefault(remoteServerUrl));
                } catch (GeneralSecurityException | IOException exception) {
                    LogUtil.logError(exception);
                }
            }
        }

        return remoteServerUrl;
    }

    /**
     * Get the type of the remove web driver is the current web driver type is
     * remote
     * <p>
     * Possible values: "Selenium", "Appium"
     * 
     * @return the type of the remove web driver is the current web driver type is
     * remote, or null if it is not
     */
    public static String getRemoteWebDriverServerType() {
        return RunConfiguration.getDriverSystemProperty(RunConfiguration.REMOTE_DRIVER_PROPERTY,
                REMOTE_WEB_DRIVER_TYPE);
    }

    /**
     * Close the active web driver
     */
    public static void closeWebDriver() {
        IDriverType driverType = getExecutedBrowser();
        WebDriver webDriver = localWebServerStorage.get();
        if (webDriver != null) {
            try {
                webDriver.quit();
                if (driverType instanceof WebUIDriverType) {
                    switch ((WebUIDriverType) driverType) {
                        case ANDROID_DRIVER:
                        case IOS_DRIVER:
                            WebMobileDriverFactory.closeDriver();
                            break;
                        default:
                            break;

                    }
                }
            } catch (UnreachableBrowserException e) {
                logger.logWarning(StringConstants.DRI_LOG_WARNING_BROWSER_NOT_REACHABLE, null, e);
            } catch (UnsupportedCommandException e) {
                // STUDIO-1684 Browser has been closed
                String message = e.getMessage();
                if (StringUtils.isNotEmpty(message)) {
                    logger.logDebug(message);
                }
            }
        }

        localWebServerStorage.set(null);
        RunConfiguration.removeDriver(webDriver);
    }

    private static SessionId getRemoteSessionId(WebDriver webDriver) {
        try {
            WebDriver unwrappedDriver = unwrapWebDriver(webDriver);
            if (unwrappedDriver instanceof RemoteWebDriver remoteWebDriver) {
                return remoteWebDriver.getSessionId();
            }
        } catch (Exception e) {
            logger.logInfo(ExceptionUtils.getStackTrace(e));
        }

        return null;
    }

    private static WebDriver unwrapWebDriver(WebDriver webDriver) {
        WebDriver unwrappedDriver = webDriver;
        if (unwrappedDriver instanceof Decorated decoratedDriver) {
            unwrappedDriver = (WebDriver) decoratedDriver.getOriginal();
        }

        return unwrappedDriver;
    }
    
    public static String getWebSocketUrl(RemoteWebDriver remoteWebDriver) {
        if (remoteWebDriver == null) {
            return StringUtils.EMPTY;
        }
        
        String webSocketUrl = (String) remoteWebDriver.getCapabilities()
                .getCapability(SeleniumW3CCapabilityConstant.WEB_SOCKET_URL_CAP);

        return webSocketUrl != null ? webSocketUrl : StringUtils.EMPTY;
    }
}
