package com.kms.katalon.core.webui.keyword.internal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.ScriptTimeoutException;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;

import com.katalon.selfhealing.execution.healers.LLMTestObjectHealer.LLMCustomPrompts;
import com.katalon.selfhealing.execution.healers.SelfHealingPipeline;
import com.katalon.selfhealing.execution.healers.SelfHealingPipeline.SelfHealingParams;
import com.katalon.selfhealing.execution.healers.WebStandardTestObjectMapper;
import com.kms.katalon.core.aut.WebAUT;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.CoreConstants;
import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.keyword.internal.AbstractKeyword;
import com.kms.katalon.core.keyword.internal.KeywordExecutionContext;
import com.kms.katalon.core.keyword.internal.KeywordExecutor;
import com.kms.katalon.core.keyword.internal.SupportLevel;
import com.kms.katalon.core.testobject.SelectorMethod;
import com.kms.katalon.core.testobject.TestObject;
import com.kms.katalon.core.webui.common.FindElementsResult;
import com.kms.katalon.core.webui.common.WebUiCommonHelper;
import com.kms.katalon.core.webui.common.controller.ElementWaitingPhase;
import com.kms.katalon.core.webui.common.controller.WaitForEnabledPhase;
import com.kms.katalon.core.webui.common.controller.WaitForStablePhase;
import com.kms.katalon.core.webui.common.controller.WaitForTopmostPhase;
import com.kms.katalon.core.webui.common.controller.WaitForVisiblePhase;
import com.kms.katalon.core.webui.constants.CoreWebuiMessageConstants;
import com.kms.katalon.core.webui.driver.DriverFactory;
import com.kms.katalon.core.webui.model.FindElementParams;
import com.kms.katalon.core.webui.model.FindElementWithTimeoutParams;
import com.kms.katalon.core.webui.model.SeleniumActionRetryController;
import com.kms.katalon.core.webui.model.SeleniumActionRetryController.RetryContext;
import com.kms.katalon.core.webui.model.WebUiElementFinder;
import com.kms.katalon.core.webui.model.WebUiElementFinderWithSelfHealing;
import com.kms.katalon.core.webui.model.WebUiElementFinderWithTimeout;

public abstract class WebUIAbstractKeyword extends AbstractKeyword {

    @Override
    public SupportLevel getSupportLevel(Object... params) {
        return SupportLevel.NOT_SUPPORT;
    }

    protected List<ElementWaitingPhase> getElementWaitingPhases(JavascriptExecutor jsExecutor) {
        List<ElementWaitingPhase> result = new ArrayList<>();
        result.add(new WaitForVisiblePhase(jsExecutor));
        result.add(new WaitForStablePhase(jsExecutor));
        result.add(new WaitForEnabledPhase(jsExecutor));
        result.add(new WaitForTopmostPhase(jsExecutor));
        return result;
    }

    public static WebElement findWebElement(TestObject to) throws Exception {
        return findWebElement(to, RunConfiguration.getElementTimeoutForWeb());
    }

    public static WebElement findWebElement(TestObject to, int timeOut) throws Exception {
        return WebUiCommonHelper.findWebElement(to, timeOut);
    }

    public static List<WebElement> findWebElements(TestObject to, int timeOut) throws Exception {
        return WebUiCommonHelper.findWebElements(to, timeOut);
    }

    public static List<WebElement> findWebElements(WebDriver webDriver, TestObject testObject, long timeoutInMillis)
            throws Exception {
        AtomicReference<Exception> defaultException = new AtomicReference<>(null);
        var params = new FindElementWithTimeoutParams(webDriver, testObject, timeoutInMillis);

        boolean shouldApplySelfHealing = RunConfiguration.shouldApplySelfHealing();
        if (!shouldApplySelfHealing) {
            return new WebUiElementFinderWithTimeout().findElements(params);
        }

        var ctrl = new SeleniumActionRetryController();
        var retryTimes = shouldApplySelfHealing ? 1 : 0;
        return ctrl.retry((retryContext) -> {
            if (retryContext.retryCount == 0) {
                try {
                    return new WebUiElementFinderWithTimeout().findElements(params);
                } catch (Exception error) {
                    defaultException.set(error);
                    throw error;
                }
            }

            var foundElements = findElemnentsWithSelfHealing(params);
            if (foundElements == null || foundElements.isEmpty()) {
                throw defaultException.get();
            }

            return foundElements;
        }, retryTimes);
    }

    public static WebElement findWebElement(WebDriver driver, TestObject testObject, long timeoutInMillis)
            throws Exception {
        AtomicReference<Exception> defaultException = new AtomicReference<>(null);
        var params = new FindElementWithTimeoutParams(driver, testObject, timeoutInMillis);

        boolean shouldApplySelfHealing = RunConfiguration.shouldApplySelfHealing();
        if (!shouldApplySelfHealing) {
            return new WebUiElementFinderWithTimeout().findElement(params);
        }

        var ctrl = new SeleniumActionRetryController();
        return ctrl.retry((retryContext) -> {
            if (retryContext.retryCount == 0) {
                try {
                    return new WebUiElementFinderWithTimeout().findElement(params);
                } catch (Exception error) {
                    defaultException.set(error);
                    throw error;
                }
            }

            var foundElements = findElemnentsWithSelfHealing(params);
            var firstElement = foundElements != null && !foundElements.isEmpty() ? foundElements.get(0) : null;
            if (firstElement == null) {
                throw defaultException.get();
            }

            return firstElement;
        }, 1);
    }

    private static List<WebElement> findElemnentsWithSelfHealing(FindElementWithTimeoutParams findElementParams)
            throws Exception {
        var driver = findElementParams.getWebDriver();
        var testObject = findElementParams.getTestObject();
        var selfHealingParams = new SelfHealingParams();

        boolean isWebPlatform = KeywordExecutionContext.isRunningWebUI();
        String runningKeyword = KeywordExecutionContext.getRunningKeyword();
        List<String> excludedKeywords = RunConfiguration.getExcludedWebUIKeywordsFromSelfHealing();

        var isSelfHealingEnabled = isWebPlatform && !excludedKeywords.contains(runningKeyword)
                && RunConfiguration.shouldApplySelfHealing();
        if (!isSelfHealingEnabled) {
            return null;
        }

        selfHealingParams.useBasicSelfHealing = true;
        selfHealingParams.useAISelfHealing = RunConfiguration.getWebAISelfHealingEnabled();
        selfHealingParams.aiSelfHealingInputSources = RunConfiguration.getWebAISelfHealingInputSources();
        selfHealingParams.currentAction = runningKeyword;

        selfHealingParams.aut = new WebAUT(driver);

        selfHealingParams.testObject = testObject;
        selfHealingParams.standardTestObjectMapper = (originalTestObject) -> {
            return new WebStandardTestObjectMapper().map(originalTestObject);
        };

        selfHealingParams.basicSelfHealer = (tempTestObject) -> {
            var foundElement = new WebUiElementFinderWithSelfHealing().findElement(findElementParams);
            return foundElement != null ? Arrays.asList(foundElement) : null;
        };

        selfHealingParams.elementsFinder = (foundLocator) -> {
            var tempTestObject = new TestObject("self-healing-temp-test-object-id");
            var locatorType = SelectorMethod.valueOf(foundLocator.type);
            tempTestObject.setSelectorMethod(locatorType);
            tempTestObject.setSelectorValue(locatorType, foundLocator.value);
            var result = new WebUiElementFinder().findElementsBySelectedMethod(driver, tempTestObject);
            return result.getElements();
        };

        var selfHealingPipeline = new SelfHealingPipeline();
        selfHealingPipeline.registerPhase((context) -> {
            var foundElements = context.data.results.foundElements;
            if (foundElements == null || foundElements.isEmpty()) {
                return;
            }

            var firstElement = foundElements.get(0);

            var healedXPath = WebUiElementFinderWithSelfHealing.generateNewXPath(firstElement, driver);

            var healingResult = FindElementsResult.from(firstElement, healedXPath, SelectorMethod.XPATH, null);
            WebUiElementFinderWithSelfHealing.registerBrokenTestObject(healingResult, testObject, driver);
        });

        var customPrompts = new LLMCustomPrompts();
        customPrompts.outputRules = """
                - Prefer **CSS** when stable; **XPath** only when necessary.
                - Use attributes in this order: `id` → `data-*` → `name` → `aria-*` → short class chains → tag+attr.
                """;
        customPrompts.locatorTypes = "\""
                + StringUtils.join(Arrays.asList(SelectorMethod.CSS, SelectorMethod.XPATH), "\" | \"") + "\"";
        selfHealingParams.customPrompts = customPrompts;

        var context = selfHealingPipeline.execute(selfHealingParams);
        return context.results.foundElements;
    }

    public static WebElement findWebElementWithoutRetry(WebDriver webDriver, TestObject testObject,
            long timeoutInMillis) throws Exception {
        getKeywordExecutor().setCurrentKeywordTimeout(timeoutInMillis);
        var params = new FindElementParams(webDriver, testObject);
        return new WebUiElementFinder().findElement(params);
    }

    /**
     * Checks if the specified web element is interactable within a given timeout.
     *
     * @param scriptExecutor the JavaScript executor to run scripts
     * @param element the web element to wait for
     * @param timeoutInMillis the maximum time to wait in milliseconds
     * @param retryContext the context for retrying actions
     * @return true if the element is interactable, false otherwise
     */
    public boolean waitElementInteractable(JavascriptExecutor scriptExecutor, WebElement element, long timeoutInMillis,
            RetryContext retryContext) {
        if (!RunConfiguration.getEnhancedWaitingEnabled()) {
            return true;
        }

        return waitElementInteractableByWaitingPhases(scriptExecutor, element, timeoutInMillis, retryContext);
    }

    boolean waitElementInteractableByWaitingPhases(JavascriptExecutor scriptExecutor, WebElement element,
            long timeoutInMillis, RetryContext retryContext) {
        List<ElementWaitingPhase> waitingPhases = getElementWaitingPhases(scriptExecutor);
        for (ElementWaitingPhase waitingPhase : waitingPhases) {
            try {
                boolean waitingResult = waitingPhase.wait(element, timeoutInMillis - retryContext.getElapsedTime());
                if (waitingResult == false) {
                    return false;
                }
            } catch (IllegalArgumentException | IllegalStateException | ScriptTimeoutException e) {
                logger.logWarning(e.getMessage());
                return false;
            }
        }
        return true;
    }

    protected WebDriver getWebDriver() {
        WebDriver webDriver = DriverFactory.getWebDriver();
        if (webDriver == null) {
            throw new StepFailedException(CoreWebuiMessageConstants.EXC_BROWSER_IS_NOT_OPENED);
        }
        return webDriver;
    }

    private static KeywordExecutor getKeywordExecutor() {
        return KeywordExecutor.getInstance(CoreConstants.PLATFORM_WEB);
    }
}
