package com.kms.katalon.core.webui.model;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.By;
import org.openqa.selenium.DetachedShadowRootException;
import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;

import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.testobject.SelectorMethod;
import com.kms.katalon.core.testobject.TestObject;
import com.kms.katalon.core.testobject.TestObjectXpath;
import com.kms.katalon.core.webui.common.CssLocatorBuilder;
import com.kms.katalon.core.webui.common.FindElementsResult;
import com.kms.katalon.core.webui.common.WebUiCommonHelper;
import com.kms.katalon.core.webui.common.internal.ImageLocatorController;
import com.kms.katalon.core.webui.common.internal.SelfHealingController;
import com.kms.katalon.core.webui.constants.CoreWebuiMessageConstants;
import com.kms.katalon.core.webui.constants.StringConstants;
import com.kms.katalon.core.webui.exception.WebElementNotFoundException;

public class WebUiElementFinder implements SeleniumElementFinder<FindElementParams> {

    private final KeywordLogger logger = KeywordLogger.getInstance(WebUiElementFinder.class);

    @Override
    public WebElement findElement(FindElementParams params) throws Exception {
        WebDriver webDriver = params.getWebDriver();
        TestObject testObject = params.getTestObject();
        WebElement cachedWebElement = testObject.getCachedWebElement();
        if (cachedWebElement != null) {
            return cachedWebElement;
        }

        List<WebElement> elements = findElementsBySelectedMethod(webDriver, testObject).getElements();
        if (elements != null && elements.size() > 0) {
            WebElement foundElement = elements.get(0);
            return foundElement;
        } else {
            String locator = StringUtils.EMPTY;
            if (testObject.getSelectorMethod() == SelectorMethod.SMART_LOCATOR
                    && !WebUiCommonHelper.ableExecuteWithSmartLocator(webDriver)) {
                locator = WebUiCommonHelper.getSelectorValue(testObject, SelectorMethod.XPATH);
            } else {
                locator = WebUiCommonHelper.getSelectorValue(testObject);
            }
            throw new WebElementNotFoundException(testObject.getObjectId(), locator);
        }
    }

    @Override
    public List<WebElement> findElements(FindElementParams params) throws Exception {
        WebDriver webDriver = params.getWebDriver();
        TestObject testObject = params.getTestObject();
        List<WebElement> elements = findElementsBySelectedMethod(webDriver, testObject).getElements();
        return elements;
    }

    public FindElementsResult findElementsBySelectedMethod(WebDriver webDriver, TestObject testObject)
            throws Exception {
        return findElementsBySelectedMethod(webDriver, testObject, testObject.getSelectorMethod());
    }

    public FindElementsResult findElementsBySelectedMethod(WebDriver webDriver, TestObject testObject,
            SelectorMethod method) throws Exception {
        return findElementsBySelectedMethod(webDriver, testObject, method, false);
    }

    public FindElementsResult findElementsBySelectedMethod(WebDriver webDriver, TestObject testObject,
            SelectorMethod method, boolean useSmartXPath) throws Exception {
        if (isElementInsideShadowDOM(webDriver, testObject) && method != SelectorMethod.SMART_LOCATOR) {
            testObject.setSelectorMethod(SelectorMethod.CSS);
            return findElementsInsideShadowDOM(webDriver, testObject);
        }

        switch (method) {
            case BASIC:
                return findElementByNormalMethods(webDriver, testObject, method);
            case CSS:
                return findElementByNormalMethods(webDriver, testObject, method);
            case XPATH:
                return useSmartXPath ? findWebElementsWithSmartXPath(webDriver, testObject)
                        : findElementByNormalMethods(webDriver, testObject, method);
            case IMAGE:
                return findElementsByImage(webDriver, testObject);
            case SMART_LOCATOR:
                if (WebUiCommonHelper.ableExecuteWithSmartLocator(webDriver)) {
                    return findElementBySmartLocator(webDriver, testObject);
                }
                return findElementByNormalMethods(webDriver, testObject, SelectorMethod.XPATH);
            default:
                return findElementByNormalMethods(webDriver, testObject, SelectorMethod.XPATH);
        }
    }

    private boolean isElementInsideShadowDOM(WebDriver webDriver, TestObject testObject) {
        return testObject.getParentObject() != null && testObject.isParentObjectShadowRoot();
    }

    private FindElementsResult findElementsInsideShadowDOM(WebDriver webDriver, TestObject testObject) throws Exception {
        String cssLocator = CssLocatorBuilder.buildCssSelectorLocator(testObject);
        List<WebElement> foundElements = Collections.emptyList();

        if (!isElementInsideShadowDOM(webDriver, testObject)) {
            return FindElementsResult.from(cssLocator, SelectorMethod.CSS);
        }

        boolean isSwitchToParentFrame = false;

        final TestObject parentObject = testObject.getParentObject();
        WebElement shadowRootElement = null;
        if (cssLocator == null) {
            throw new StepFailedException(MessageFormat.format(
                    StringConstants.KW_EXC_WEB_ELEMENT_W_ID_DOES_NOT_HAVE_SATISFY_PROP, testObject.getObjectId()));
        }

        logger.logDebug(MessageFormat.format(CoreWebuiMessageConstants.MSG_INFO_WEB_ELEMENT_HAVE_PARENT_SHADOW_ROOT,
                testObject.getObjectId(), testObject.getParentObject().getObjectId()));
        try {
            isSwitchToParentFrame = WebUiCommonHelper.switchToParentFrame(webDriver, parentObject);
            FindElementParams findParentParams = new FindElementParams(webDriver, parentObject);
            shadowRootElement = findElement(findParentParams);
        } catch (WebElementNotFoundException e) {
            return FindElementsResult.from(cssLocator, SelectorMethod.CSS);
        }

        if (shadowRootElement == null) {
            return FindElementsResult.from(cssLocator, SelectorMethod.CSS);
        }

        try {
            foundElements = doFindElementsInsideShadowDom(testObject, webDriver, cssLocator, shadowRootElement);
            if (foundElements == null || foundElements.isEmpty()) {
                throw new DetachedShadowRootException(MessageFormat
                        .format(StringConstants.KW_LOG_INFO_CANNOT_FIND_WEB_ELEMENT_BY_LOCATOR, cssLocator));
            } else {
                if (isSwitchToParentFrame) {
                    WebUiCommonHelper.switchToDefaultContent();
                }
            }
        } catch (WebElementNotFoundException exception) {
            // Element not found, do nothing
        } finally {
            if (isSwitchToParentFrame) {
                WebUiCommonHelper.switchToDefaultContent();
            }
        }
        return FindElementsResult.from(foundElements, cssLocator, SelectorMethod.CSS, StringUtils.EMPTY);
    }

    @SuppressWarnings("unchecked")
    private List<WebElement> doFindElementsInsideShadowDom(TestObject testObject, WebDriver webDriver,
            final String cssLocator, WebElement shadowRootElement) throws WebElementNotFoundException {
        String filteredCssSelector = StringUtils.defaultString(cssLocator).replace("'", "\\\'");
        List<WebElement> webElements = (List<WebElement>) ((JavascriptExecutor) webDriver).executeScript(
                "return arguments[0].shadowRoot.querySelectorAll('" + filteredCssSelector + "');", shadowRootElement);
        if (webElements != null && webElements.size() > 0) {
            logger.logDebug(MessageFormat.format("Found {0} web elements with id: ''{1}'' located by ''{2}''",
                    webElements.size(), testObject.getObjectId(), cssLocator));
            return webElements;
        }
        throw new WebElementNotFoundException(testObject.getObjectId(), cssLocator);
    }

    /**
     * Find Web Elements by using Attributes, XPath or CSS locators
     */
    private FindElementsResult findElementByNormalMethods(WebDriver webDriver, TestObject testObject,
            SelectorMethod locatorMethod) throws Exception {
        By defaultLocator = WebUiCommonHelper.buildLocator(testObject, locatorMethod);
        if (defaultLocator == null) {
            throw new StepFailedException(MessageFormat.format(
                    StringConstants.KW_EXC_WEB_ELEMENT_W_ID_DOES_NOT_HAVE_SATISFY_PROP, testObject.getObjectId()));
        }

        List<WebElement> foundElements = Collections.emptyList();
        foundElements = webDriver.findElements(defaultLocator);
        if (foundElements != null && !foundElements.isEmpty()) {
            logger.logDebug(MessageFormat.format("Found {0} web elements with id: ''{1}'' located by ''{2}''",
                    foundElements.size(), testObject.getObjectId(), defaultLocator.toString()));

        }

        if (foundElements == null || foundElements.isEmpty()) {
            throw new WebElementNotFoundException(testObject.getObjectId(), defaultLocator);
        }
        return FindElementsResult.from(foundElements, WebUiCommonHelper.getSelectorValue(testObject, locatorMethod),
                locatorMethod);
    }

    private FindElementsResult findWebElementsWithSmartXPath(WebDriver webDriver, TestObject testObject) {
        if (isElementInsideShadowDOM(webDriver, testObject)) {
            return FindElementsResult.from(SelectorMethod.XPATH);
        }

        SelfHealingController.setLogger(logger);

        List<TestObjectXpath> allXPaths = testObject.getXpaths();
        TestObjectXpath selectedXPath = null;
        List<WebElement> foundElements = null;
        String screenshotPath = StringUtils.EMPTY;

        for (int i = 0; i < allXPaths.size(); i++) {
            selectedXPath = allXPaths.get(i);
            String xpathValue = selectedXPath.getValue();
            if (StringUtils.isBlank(xpathValue)) {
                continue;
            }

            By bySelectedXPath = By.xpath(selectedXPath.getValue());
            try {
                foundElements = webDriver.findElements(bySelectedXPath);
            } catch (NoSuchElementException e) {
                // do nothing
            }

            if (foundElements != null && !foundElements.isEmpty()) {
                SelfHealingController.logInfo(MessageFormat.format(
                        StringConstants.KW_LOG_INFO_FOUND_WEB_ELEMENT_WITH_THIS_SMART_XPATH, selectedXPath.getValue()));
                String screenshotName = selectedXPath.getName().split(":")[1];

                screenshotPath = SelfHealingController.takeScreenShot(webDriver, foundElements.get(0), testObject,
                        screenshotName);
                break;
            } else {
                SelfHealingController.logInfo(MessageFormat.format(
                        StringConstants.KW_LOG_INFO_COULD_NOT_FIND_WEB_ELEMENT_WITH_THIS_SMART_XPATH, xpathValue));
            }
        }
        if (selectedXPath == null) {
            SelfHealingController.logInfo(StringConstants.KW_LOG_INFO_COULD_NOT_FIND_ANY_WEB_ELEMENT_WITH_SMART_XPATHS);
            return FindElementsResult.from(SelectorMethod.XPATH);
        }
        return FindElementsResult.from(foundElements, selectedXPath.getValue(), SelectorMethod.XPATH, screenshotPath);
    }

    private FindElementsResult findElementsByImage(WebDriver webDriver, TestObject testObject) {
        String screenshot = testObject.getSelectorCollection().get(SelectorMethod.IMAGE);

        // For backward compatible
        if (StringUtils.isBlank(screenshot)) {
            screenshot = testObject.getProperties()
                    .stream()
                    .filter(prop -> prop.getName().equals("screenshot"))
                    .findAny()
                    .map(screenshotProp -> screenshotProp.getValue())
                    .orElse(StringUtils.EMPTY);
        }

        if (StringUtils.isBlank(screenshot)) {
            return FindElementsResult.from(SelectorMethod.IMAGE);
        }

        List<WebElement> foundElements = ImageLocatorController.findElementByScreenShot(webDriver, screenshot, 0);

        return FindElementsResult.from(foundElements, screenshot, SelectorMethod.IMAGE);
    }

    private FindElementsResult findElementBySmartLocator(WebDriver webDriver, TestObject testObject) {
        List<WebElement> foundElements = Collections.emptyList();
        String smartLocator = "";
        smartLocator = testObject.getSmartSelectorCollection().get(SelectorMethod.SMART_LOCATOR);
        // Early return if locator is blank
        if (StringUtils.isBlank(smartLocator)) {
            return FindElementsResult.from(SelectorMethod.SMART_LOCATOR);
        }
        JavascriptExecutor js = (JavascriptExecutor) webDriver;
        foundElements = new ArrayList<>();
        String jsLocator = String.format("return katalonSmartLocator.find_element_by_smart_locator(`%s`)",
                smartLocator);
        WebElement element = (WebElement) js.executeScript(jsLocator);
        if (element != null) {
            foundElements.add(element);
        }
        if (foundElements != null && !foundElements.isEmpty()) {
            logger.logDebug(MessageFormat.format("Found {0} web elements with id: ''{1}'' located by ''{2}''",
                    foundElements.size(), testObject.getObjectId(), smartLocator));
            return FindElementsResult.from(foundElements, smartLocator, SelectorMethod.SMART_LOCATOR);
        }
        FindElementsResult result = FindElementsResult.from(foundElements, smartLocator, SelectorMethod.SMART_LOCATOR);
        return result;
    }
}
