package com.kms.katalon.core.webui.trace;

import com.kms.katalon.core.trace.TraceDebug;
import com.kms.katalon.core.trace.TraceUtils;
import com.kms.katalon.core.webui.driver.DriverFactory;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.devtools.DevTools;
import org.openqa.selenium.devtools.HasDevTools;
import org.openqa.selenium.devtools.v143.network.Network;
import org.openqa.selenium.devtools.v143.network.model.LoadingFailed;
import org.openqa.selenium.devtools.v143.network.model.LoadingFinished;
import org.openqa.selenium.devtools.v143.network.model.MonotonicTime;
import org.openqa.selenium.devtools.v143.network.model.RequestId;
import org.openqa.selenium.devtools.v143.network.model.RequestWillBeSent;
import org.openqa.selenium.devtools.v143.network.model.RequestWillBeSentExtraInfo;
import org.openqa.selenium.devtools.v143.network.model.ResourceTiming;
import org.openqa.selenium.devtools.v143.network.model.ResponseReceived;
import org.openqa.selenium.devtools.v143.network.model.ResponseReceivedExtraInfo;
import org.openqa.selenium.devtools.v143.network.model.TimeSinceEpoch;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

public class HarTracer {
    private static final class EntryState {
        Map<String, Object> request = emptyRequest();
        Map<String, Object> response = emptyResponse();
        long startMonotonic = TraceUtils.monotonicMs();
        double requestTimestampSeconds = -1d;
        String startedDateIso = Instant.now().toString();
        String frameId;
        String pageRef;
        String loaderId;
        String resourceType;
        ResourceTiming resourceTiming;
        boolean servedFromCache;
        boolean fromDiskCache;
        boolean fromServiceWorker;
        boolean loadingFailed;
        String failureText;
        long encodedDataLength;
    }

    private static final class State {
        DevTools devTools;
        TraceSession session;
        Map<String, EntryState> entries = new ConcurrentHashMap<>();
        boolean enabled;
    }

    private static volatile State state;

    private HarTracer() {
    }

    private static List<Map<String, Object>> toHeaderList(Map<?, ?> headers) {
        if (headers == null || headers.isEmpty()) {
            return new ArrayList<>();
        }
        List<Map<String, Object>> list = new ArrayList<>();
        for (Map.Entry<?, ?> e : headers.entrySet()) {
            Map<String, Object> h = new LinkedHashMap<>();
            h.put("name", String.valueOf(e.getKey()));
            h.put("value", String.valueOf(e.getValue()));
            list.add(h);
        }
        return list;
    }

    private static int headerBytes(List<Map<String, Object>> headers) {
        if (headers == null || headers.isEmpty()) {
            return -1;
        }
        int total = 0;
        for (Map<String, Object> h : headers) {
            String name = h.get("name") != null ? String.valueOf(h.get("name")) : "";
            String value = h.get("value") != null ? String.valueOf(h.get("value")) : "";
            total += name.length() + value.length() + 4;
        }
        return total;
    }

    private static List<Map<String, Object>> parseQueryString(String url) {
        if (url == null) {
            return new ArrayList<>();
        }
        try {
            URI uri = new URI(url);
            String query = uri.getRawQuery();
            if (query == null || query.isEmpty()) {
                return new ArrayList<>();
            }
            String[] pairs = query.split("&");
            List<Map<String, Object>> list = new ArrayList<>();
            for (String pair : pairs) {
                String[] nv = pair.split("=", 2);
                String name = URLDecoder.decode(nv[0], StandardCharsets.UTF_8.name());
                String value = nv.length > 1 ? URLDecoder.decode(nv[1], StandardCharsets.UTF_8.name()) : "";
                Map<String, Object> entry = new LinkedHashMap<>();
                entry.put("name", name);
                entry.put("value", value);
                list.add(entry);
            }
            return list;
        } catch (Throwable ignored) {
            return new ArrayList<>();
        }
    }

    private static List<Map<String, Object>> parseRequestCookies(Map<?, ?> headers) {
        String header = valueFromHeaderMap(headers, "cookie");
        if (header == null || header.isEmpty()) {
            return new ArrayList<>();
        }
        String[] tokens = header.split(";");
        List<Map<String, Object>> cookies = new ArrayList<>();
        for (String token : tokens) {
            String[] nv = token.trim().split("=", 2);
            Map<String, Object> c = new LinkedHashMap<>();
            c.put("name", nv[0]);
            c.put("value", nv.length > 1 ? nv[1] : "");
            cookies.add(c);
        }
        return cookies;
    }

    private static String valueFromHeaderMap(Map<?, ?> headers, String name) {
        if (headers == null) {
            return null;
        }
        for (Map.Entry<?, ?> e : headers.entrySet()) {
            if (name.equalsIgnoreCase(String.valueOf(e.getKey()))) {
                Object v = e.getValue();
                return v != null ? v.toString() : null;
            }
        }
        return null;
    }

    private static List<Map<String, Object>> parseResponseCookies(List<Map<String, Object>> headers) {
        if (headers == null || headers.isEmpty()) {
            return new ArrayList<>();
        }
        List<Map<String, Object>> cookies = new ArrayList<>();
        for (Map<String, Object> h : headers) {
            String name = h.get("name") != null ? String.valueOf(h.get("name")) : "";
            if ("set-cookie".equalsIgnoreCase(name)) {
                String header = h.get("value") != null ? String.valueOf(h.get("value")) : "";
                Map<String, Object> cookie = parseSetCookie(header);
                cookies.add(cookie);
            }
        }
        return cookies;
    }

    private static Map<String, Object> parseSetCookie(String header) {
        Map<String, Object> cookie = new LinkedHashMap<>();
        cookie.put("name", "");
        cookie.put("value", "");
        if (header == null || header.isEmpty()) {
            return cookie;
        }
        String[] parts = header.split(";");
        if (parts.length > 0) {
            String[] first = parts[0].trim().split("=", 2);
            cookie.put("name", first[0]);
            cookie.put("value", first.length > 1 ? first[1] : "");
        }
        for (int i = 1; i < parts.length; i++) {
            String attr = parts[i].trim();
            if (attr.isEmpty()) continue;
            String[] kv = attr.split("=", 2);
            String key = kv[0] != null ? kv[0].toLowerCase() : "";
            String value = kv.length > 1 ? kv[1] : "";
            switch (key) {
                case "path":
                    cookie.put("path", value);
                    break;
                case "domain":
                    cookie.put("domain", value);
                    break;
                case "expires":
                    cookie.put("expires", value);
                    break;
                case "max-age":
                    cookie.put("maxAge", value);
                    break;
                case "secure":
                    cookie.put("secure", true);
                    break;
                case "httponly":
                    cookie.put("httpOnly", true);
                    break;
                case "samesite":
                    cookie.put("sameSite", value);
                    break;
                default:
                    break;
            }
        }
        return cookie;
    }

    private static String isoFrom(TimeSinceEpoch wallTime) {
        if (wallTime == null) {
            return Instant.now().toString();
        }
        double seconds = wallTime.toJson().doubleValue();
        long millis = (long) Math.floor(seconds * 1000);
        return Instant.ofEpochMilli(millis).toString();
    }

    private static double secondsFrom(MonotonicTime ts) {
        if (ts == null) {
            return -1d;
        }
        return ts.toJson().doubleValue();
    }

    @SuppressWarnings("unchecked")
    private static Map<String, Object> timingsFor(EntryState entry, LoadingFinished event) {
        Map<String, Object> timings = new LinkedHashMap<>();
        timings.put("dns", -1);
        timings.put("connect", -1);
        timings.put("ssl", -1);
        timings.put("send", -1);
        timings.put("wait", -1);
        timings.put("receive", -1);

        ResourceTiming rt = entry.resourceTiming;
        if (rt != null) {
            timings.put("dns", duration(rt.getDnsStart(), rt.getDnsEnd()));
            timings.put("connect", duration(rt.getConnectStart(), rt.getConnectEnd()));
            timings.put("ssl", duration(rt.getSslStart(), rt.getSslEnd()));
            timings.put("send", duration(rt.getSendStart(), rt.getSendEnd()));
            timings.put("wait", duration(rt.getSendEnd(), rt.getReceiveHeadersStart()));
            timings.put("receive", duration(rt.getReceiveHeadersStart(), rt.getReceiveHeadersEnd()));
        }
        long total = totalDuration(entry, event);

        long consumed = 0;
        for (String key : new String[]{"dns", "connect", "ssl", "send", "wait"}) {
            Object v = timings.get(key);
            if (v instanceof Number) {
                int iv = ((Number) v).intValue();
                if (iv >= 0) {
                    consumed += iv;
                }
            }
        }

        Object receiveObj = timings.get("receive");
        int receive = receiveObj instanceof Number ? ((Number) receiveObj).intValue() : -1;
        if (receive < 0) {
            receive = (int) Math.max(0, total - consumed);
        } else if (total > 0 && consumed + receive < total) {
            receive = (int) Math.max(0, total - consumed);
        }
        timings.put("receive", receive);

        Map<String, Object> pack = new LinkedHashMap<>();
        pack.put("detail", timings);
        pack.put("total", total);
        return pack;
    }

    private static int duration(Number start, Number end) {
        if (start == null || end == null) {
            return -1;
        }
        double s = start.doubleValue();
        double e = end.doubleValue();
        if (s < 0 || e < 0 || e < s) {
            return -1;
        }
        return (int) Math.round(e - s);
    }

    private static long totalDuration(EntryState entry, LoadingFinished event) {
        if (event == null) {
            return TraceUtils.monotonicMs() - entry.startMonotonic;
        }
        Double finish = event.getTimestamp() != null ? event.getTimestamp().toJson().doubleValue() : null;
        if (finish == null) {
            return TraceUtils.monotonicMs() - entry.startMonotonic;
        }
        double start = entry.requestTimestampSeconds >= 0 ? entry.requestTimestampSeconds : 0;
        return (long) Math.max(0, (finish - start) * 1000);
    }

    private static String headerValue(List<Map<String, Object>> headers, String name) {
        if (headers == null || headers.isEmpty()) {
            return null;
        }
        for (Map<String, Object> h : headers) {
            String headerName = h.get("name") != null ? String.valueOf(h.get("name")) : "";
            if (name.equalsIgnoreCase(headerName)) {
                Object v = h.get("value");
                return v != null ? v.toString() : null;
            }
        }
        return null;
    }

    private static Map<String, Object> emptyRequest() {
        Map<String, Object> req = new LinkedHashMap<>();
        req.put("method", "GET");
        req.put("url", "");
        req.put("httpVersion", "HTTP/1.1");
        req.put("cookies", new ArrayList<>());
        req.put("headers", new ArrayList<>());
        req.put("queryString", new ArrayList<>());
        req.put("headersSize", -1);
        req.put("bodySize", -1);
        return req;
    }

    private static Map<String, Object> emptyResponse() {
        Map<String, Object> resp = new LinkedHashMap<>();
        resp.put("status", 0);
        resp.put("statusText", "");
        resp.put("httpVersion", "HTTP/1.1");
        resp.put("cookies", new ArrayList<>());
        resp.put("headers", new ArrayList<>());
        Map<String, Object> content = new LinkedHashMap<>();
        content.put("mimeType", "application/octet-stream");
        content.put("size", -1);
        resp.put("content", content);
        resp.put("redirectURL", "");
        resp.put("headersSize", -1);
        resp.put("bodySize", -1);
        return resp;
    }

    private static void addListeners(State st) {
        DevTools dt = st.devTools;
        dt.addListener(Network.requestWillBeSent(), (RequestWillBeSent event) -> {
            try {
                recordRequest(st, event);
            } catch (Throwable t) {
                TraceDebug.writeLine("har: requestWillBeSent error " + t.getMessage());
            }
        });
        dt.addListener(Network.requestWillBeSentExtraInfo(), (RequestWillBeSentExtraInfo extra) -> {
            try {
                applyRequestExtra(st, extra);
            } catch (Throwable t) {
                TraceDebug.writeLine("har: requestExtra error " + t.getMessage());
            }
        });
        dt.addListener(Network.responseReceived(), (ResponseReceived event) -> {
            try {
                recordResponse(st, event);
            } catch (Throwable t) {
                TraceDebug.writeLine("har: responseReceived error " + t.getMessage());
            }
        });
        dt.addListener(Network.responseReceivedExtraInfo(), (ResponseReceivedExtraInfo extra) -> {
            try {
                applyResponseExtra(st, extra);
            } catch (Throwable t) {
                TraceDebug.writeLine("har: responseExtra error " + t.getMessage());
            }
        });
        dt.addListener(Network.loadingFinished(), (LoadingFinished event) -> {
            try {
                handleFinished(st, event);
            } catch (Throwable t) {
                TraceDebug.writeLine("har: loadingFinished error " + t.getMessage());
            }
        });
        dt.addListener(Network.loadingFailed(), (LoadingFailed event) -> {
            try {
                handleFailure(st, event);
            } catch (Throwable t) {
                TraceDebug.writeLine("har: loadingFailed error " + t.getMessage());
            }
        });
    }

    private static void recordRequest(State st, RequestWillBeSent event) {
        String id = event.getRequestId() != null ? event.getRequestId().toString() : null;
        if (id == null || id.isEmpty()) {
            return;
        }
        EntryState entry = st.entries.computeIfAbsent(id, k -> new EntryState());
        Map<String, Object> entryReq = entry.request != null ? entry.request : emptyRequest();

        var req = event.getRequest();
        String url = req.getUrl();
        String currentUrl = entryReq.get("url") != null ? String.valueOf(entryReq.get("url")) : "";
        if (currentUrl.isEmpty()) {
            entryReq.put("url", url);
        }
        String method = req.getMethod();
        if (method != null) {
            entryReq.put("method", method);
        }
        // DevTools v138 Request does not expose protocol; keep existing/default HTTP version.
        String httpVersion = (String) entryReq.getOrDefault("httpVersion", "HTTP/1.1");
        entryReq.put("httpVersion", httpVersion);

        Map<?, ?> headersMap = req.getHeaders() != null ? req.getHeaders().toJson() : Collections.emptyMap();
        List<Map<String, Object>> headerList = toHeaderList(headersMap);
        entryReq.put("headers", headerList);
        entryReq.put("headersSize", headerBytes(headerList));
        entryReq.put("queryString", parseQueryString((String) entryReq.get("url")));
        entryReq.put("cookies", parseRequestCookies(headersMap));

        String postData = req.getPostData().orElse(null);
        if (postData != null) {
            Map<String, Object> post = new LinkedHashMap<>();
            String mime = valueFromHeaderMap(headersMap, "content-type");
            if (mime == null) {
                mime = "application/octet-stream";
            }
            post.put("mimeType", mime);
            post.put("text", postData);
            entryReq.put("postData", post);
            entryReq.put("bodySize", postData.getBytes(StandardCharsets.UTF_8).length);
        } else {
            entryReq.put("bodySize", 0);
        }

        entry.request = entryReq;
        entry.startMonotonic = TraceUtils.monotonicMs();
        entry.requestTimestampSeconds = secondsFrom(event.getTimestamp());
        entry.startedDateIso = isoFrom(event.getWallTime());
        entry.frameId = event.getFrameId().map(idObj -> idObj.toJson()).orElse(null);
        entry.pageRef = entry.frameId != null ? entry.frameId : st.session.getContextId();
        entry.loaderId = event.getLoaderId() != null ? event.getLoaderId().toString() : null;
        entry.resourceType = event.getType().map(Object::toString).orElse(null);
    }

    private static void applyRequestExtra(State st, RequestWillBeSentExtraInfo extra) {
        String id = extra.getRequestId() != null ? extra.getRequestId().toString() : null;
        if (id == null || id.isEmpty()) {
            return;
        }
        EntryState entry = st.entries.computeIfAbsent(id, k -> new EntryState());
        Map<String, Object> req = entry.request != null ? entry.request : emptyRequest();
        Map<?, ?> headersMap = extra.getHeaders() != null ? extra.getHeaders().toJson() : Collections.emptyMap();
        List<Map<String, Object>> headerList = toHeaderList(headersMap);
        if (!headerList.isEmpty()) {
            req.put("headers", headerList);
            req.put("headersSize", headerBytes(headerList));
            req.put("cookies", parseRequestCookies(headersMap));
        }
        entry.request = req;
    }

    private static void recordResponse(State st, ResponseReceived event) {
        String id = event.getRequestId() != null ? event.getRequestId().toString() : null;
        if (id == null || id.isEmpty()) {
            return;
        }
        EntryState entry = st.entries.computeIfAbsent(id, k -> new EntryState());
        var resp = event.getResponse();
        Map<String, Object> responseMap = entry.response != null ? entry.response : emptyResponse();
        Object contentObj = responseMap.get("content");
        if (!(contentObj instanceof Map)) {
            contentObj = new LinkedHashMap<String, Object>();
            ((Map<String, Object>) contentObj).put("mimeType", "application/octet-stream");
            ((Map<String, Object>) contentObj).put("size", -1);
            responseMap.put("content", contentObj);
        }
        Map<String, Object> content = (Map<String, Object>) contentObj;

        Integer statusVal = resp.getStatus();
        if (statusVal != null && statusVal > 0) {
            responseMap.put("status", statusVal);
        }
        String statusText = resp.getStatusText();
        if (statusText != null && !statusText.isEmpty()) {
            responseMap.put("statusText", statusText);
        }

        Map<?, ?> headersMap = resp.getHeaders() != null ? resp.getHeaders().toJson() : Collections.emptyMap();
        List<Map<String, Object>> headers = toHeaderList(headersMap);
        if (!headers.isEmpty()) {
            responseMap.put("headers", headers);
            responseMap.put("headersSize", headerBytes(headers));
            responseMap.put("cookies", parseResponseCookies(headers));
            String redirect = headerValue(headers, "location");
            if (redirect != null) {
                responseMap.put("redirectURL", redirect);
            }
            String ct = headerValue(headers, "content-type");
            if (ct != null && !ct.isEmpty()) {
                content.put("mimeType", ct);
            }
        }
        String httpVersion = resp.getProtocol().orElse((String) responseMap.getOrDefault("httpVersion", "HTTP/1.1"));
        responseMap.put("httpVersion", httpVersion);

        Long transfer = resp.getEncodedDataLength() != null ? resp.getEncodedDataLength().longValue() : null;
        if (transfer != null && transfer >= 0) {
            responseMap.put("_transferSize", transfer);
        }
        responseMap.put("serverIPAddress", resp.getRemoteIPAddress().orElse((String) responseMap.get("serverIPAddress")));
        responseMap.put("_remotePort", resp.getRemotePort().orElse((Integer) responseMap.get("_remotePort")));
        boolean servedFromCache = resp.getFromDiskCache().orElse(false) || resp.getFromServiceWorker().orElse(false);
        responseMap.put("_servedFromCache", servedFromCache);

        resp.getSecurityDetails().ifPresent(details -> {
            Map<String, Object> sec = new LinkedHashMap<>();
            sec.put("protocol", details.getProtocol());
            sec.put("subjectName", details.getSubjectName());
            sec.put("issuer", details.getIssuer());
            sec.put("validFrom", details.getValidFrom());
            sec.put("validTo", details.getValidTo());
            responseMap.put("_securityDetails", sec);
        });

        entry.response = responseMap;
        Map<String, Object> requestMap = entry.request != null ? entry.request : emptyRequest();
        String reqUrl = (String) requestMap.get("url");
        if (reqUrl == null || reqUrl.isEmpty()) {
            String respUrl = resp.getUrl();
            if (respUrl != null) {
                requestMap.put("url", respUrl);
            }
        }
        requestMap.put("httpVersion", responseMap.get("httpVersion"));
        entry.request = requestMap;

        entry.resourceTiming = resp.getTiming().orElse(null);
        entry.fromDiskCache = resp.getFromDiskCache().orElse(false);
        entry.fromServiceWorker = resp.getFromServiceWorker().orElse(false);
        if (entry.frameId == null) {
            entry.frameId = event.getFrameId().map(idObj -> idObj.toJson()).orElse(null);
        }
        if (entry.pageRef == null) {
            entry.pageRef = entry.frameId != null ? entry.frameId : st.session.getContextId();
        }
    }

    private static void applyResponseExtra(State st, ResponseReceivedExtraInfo extra) {
        String id = extra.getRequestId() != null ? extra.getRequestId().toString() : null;
        if (id == null || id.isEmpty()) {
            return;
        }
        EntryState entry = st.entries.computeIfAbsent(id, k -> new EntryState());
        Map<String, Object> responseMap = entry.response != null ? entry.response : emptyResponse();
        String protocol = extra.getHeadersText().orElse((String) responseMap.get("httpVersion"));
        if (protocol != null && !protocol.isEmpty()) {
            responseMap.put("httpVersion", protocol);
        }
        Map<?, ?> headersMap = extra.getHeaders() != null ? extra.getHeaders().toJson() : Collections.emptyMap();
        List<Map<String, Object>> headers = toHeaderList(headersMap);
        if (!headers.isEmpty()) {
            responseMap.put("headers", headers);
            responseMap.put("headersSize", headerBytes(headers));
            responseMap.put("cookies", parseResponseCookies(headers));
        }
        // Preserve any existing headersText, but prefer the value from the extra info if present.
        responseMap.put("headersText", extra.getHeadersText().orElse((String) responseMap.get("headersText")));
        // IPAddressSpace is a concrete value in this DevTools version, not Optional.
        responseMap.put("resourceIPAddressSpace", String.valueOf(extra.getResourceIPAddressSpace()));
        Integer statusCode = extra.getStatusCode();
        if (statusCode != null) {
            responseMap.put("status", statusCode);
        }
        // ResponseReceivedExtraInfo in v138 does not expose statusText; keep any value from the main response.
        entry.response = responseMap;
    }

    @SuppressWarnings("unchecked")
    private static void handleFailure(State st, LoadingFailed event) {
        String id = event.getRequestId() != null ? event.getRequestId().toString() : null;
        if (id == null || id.isEmpty()) {
            return;
        }
        EntryState entry = st.entries.remove(id);
        if (entry == null) {
            return;
        }
        entry.loadingFailed = true;
        entry.failureText = event.getErrorText();

        Map<String, Object> req = entry.request != null ? entry.request : emptyRequest();
        Map<String, Object> resp = entry.response != null ? entry.response : emptyResponse();
        req.put("_failed", true);
        resp.put("_failed", true);
        resp.put("_errorText", entry.failureText);

        Map<String, Object> timingsPack = timingsFor(entry, null);
        Map<String, Object> harEntry = new LinkedHashMap<>();
        harEntry.put("startedDateTime", entry.startedDateIso);
        harEntry.put("time", TraceUtils.monotonicMs() - entry.startMonotonic);
        harEntry.put("request", req);
        harEntry.put("response", resp);
        harEntry.put("cache", new LinkedHashMap<>());
        harEntry.put("timings", timingsPack.get("detail"));
        harEntry.put("pageref", entry.pageRef);

        long nowMono = TraceUtils.monotonicMs();
        Map<String, Object> snapshot = new LinkedHashMap<>();
        snapshot.put("_requestId", id);
        snapshot.put("_loaderId", entry.loaderId);
        snapshot.put("_frameref", entry.frameId != null ? entry.frameId : entry.pageRef);
        snapshot.put("_resourceType", entry.resourceType);
        snapshot.put("_servedFromCache", entry.servedFromCache);
        snapshot.put("_monotonicTime", nowMono);
        snapshot.put("startedDateTime", entry.startedDateIso);
        snapshot.put("time", harEntry.get("time"));
        snapshot.put("pageref", entry.pageRef != null ? entry.pageRef : "page-1");
        snapshot.put("request", req);
        snapshot.put("response", resp);
        snapshot.put("cache", new LinkedHashMap<>());
        snapshot.put("timings", harEntry.get("timings"));
        snapshot.put("serverIPAddress", resp.get("serverIPAddress"));

        Map<String, Object> snapshotEnvelope = new LinkedHashMap<>();
        snapshotEnvelope.put("type", "resource-snapshot");
        snapshotEnvelope.put("snapshot", snapshot);
        TraceUtils.appendJsonLine(st.session.getNetworkFile(), snapshotEnvelope);

        Map<String, Object> resourceEnvelope = new LinkedHashMap<>();
        resourceEnvelope.put("type", "resource");
        resourceEnvelope.put("contextId", st.session.getContextId());
        resourceEnvelope.put("pageId", entry.pageRef);
        resourceEnvelope.put("resourceId", id);
        resourceEnvelope.put("ordinal", nowMono);
        resourceEnvelope.put("ended", true);
        resourceEnvelope.put("resource", harEntry);
        TraceUtils.appendJsonLine(st.session.getNetworkFile(), resourceEnvelope);
    }

    @SuppressWarnings("unchecked")
    private static void handleFinished(State st, LoadingFinished event) {
        String id = event.getRequestId() != null ? event.getRequestId().toString() : null;
        if (id == null || id.isEmpty()) {
            return;
        }
        EntryState entry = st.entries.remove(id);
        if (entry == null) {
            return;
        }
        entry.encodedDataLength = event.getEncodedDataLength() != null ? event.getEncodedDataLength().longValue() : 0;
        Map<String, Object> request = entry.request != null ? entry.request : emptyRequest();
        Map<String, Object> response = entry.response != null ? entry.response : emptyResponse();

        Map<String, Object> timingsPack = timingsFor(entry, event);
        if (!response.containsKey("bodySize") || response.get("bodySize") == null) {
            response.put("bodySize", -1);
        }
        Object contentObj = response.get("content");
        if (!(contentObj instanceof Map)) {
            contentObj = new LinkedHashMap<String, Object>();
            ((Map<String, Object>) contentObj).put("mimeType", "application/octet-stream");
            ((Map<String, Object>) contentObj).put("size", -1);
            response.put("content", contentObj);
        }
        Map<String, Object> content = (Map<String, Object>) contentObj;

        String mime = content.get("mimeType") != null ? String.valueOf(content.get("mimeType")) : "application/octet-stream";
        String reqUrl = request.get("url") != null ? String.valueOf(request.get("url")) : "";

        boolean captureBody = shouldCaptureBody(mime) || shouldCaptureUrl(reqUrl);
        byte[] bodyBytes = null;
        Map<String, Object> fallbackMeta = null;
        String shaName = null;

        if (captureBody) {
            bodyBytes = readResponseBody(st, new RequestId(id), entry);
            if (bodyBytes == null || bodyBytes.length == 0) {
                fallbackMeta = httpFetchBody(entry);
                if (fallbackMeta != null) {
                    Object bytesObj = fallbackMeta.get("bytes");
                    if (bytesObj instanceof byte[]) {
                        bodyBytes = (byte[]) bytesObj;
                    }
                }
            }
            if (bodyBytes != null && bodyBytes.length > 0) {
                String ext = extensionForMime(mime);
                if (ext == null && fallbackMeta != null && fallbackMeta.get("contentType") != null) {
                    ext = extensionForMime(String.valueOf(fallbackMeta.get("contentType")));
                }
                if (ext == null) {
                    ext = extensionFromUrl(reqUrl);
                }
                String sha = TraceUtils.sha1Hex(bodyBytes);
                shaName = ext != null ? (sha + "." + ext) : sha;
                File out = new File(st.session.getResourcesDir(), shaName);
                if (st.session.getResourceKeys().add(shaName)) {
                    TraceUtils.writeBytes(out, bodyBytes);
                }
                content.put("_sha1", shaName);
                content.put("size", bodyBytes.length);
                response.put("bodySize", bodyBytes.length);
                if (fallbackMeta != null && fallbackMeta.get("contentType") != null) {
                    content.put("mimeType", fallbackMeta.get("contentType"));
                }
            }
        }

        if (shaName == null) {
            content.remove("_sha1");
            if (((Number) content.getOrDefault("size", -1)).longValue() == -1) {
                long size = entry.encodedDataLength >= 0 ? entry.encodedDataLength : -1;
                content.put("size", size);
            }
            Object bodySizeObj = response.get("bodySize");
            long bodySize = bodySizeObj instanceof Number ? ((Number) bodySizeObj).longValue() : -1;
            if (bodySize == -1) {
                Object sizeObj = content.get("size");
                long s = sizeObj instanceof Number ? ((Number) sizeObj).longValue() : -1;
                response.put("bodySize", s);
            }
        }

        if (fallbackMeta != null) {
            Object statusObj = response.get("status");
            int status = statusObj instanceof Number ? ((Number) statusObj).intValue() : 0;
            if (status == 0) {
                Object fbStatus = fallbackMeta.get("status");
                response.put("status", fbStatus instanceof Number ? ((Number) fbStatus).intValue() : 200);
            }
            Object statusTextObj = response.get("statusText");
            String stText = statusTextObj != null ? String.valueOf(statusTextObj) : "";
            if (stText.isEmpty()) {
                Object fbStatusText = fallbackMeta.get("statusText");
                response.put("statusText", fbStatusText != null ? String.valueOf(fbStatusText) : "");
            }
            Object headersObj = fallbackMeta.get("headers");
            List<Map<String, Object>> metaHeaders = headersObj instanceof List ? (List<Map<String, Object>>) headersObj : null;
            if (metaHeaders != null && !metaHeaders.isEmpty()) {
                response.put("headers", metaHeaders);
                response.put("headersSize", headerBytes(metaHeaders));
                response.put("cookies", parseResponseCookies(metaHeaders));
                String redirect = headerValue(metaHeaders, "location");
                if (redirect != null) {
                    response.put("redirectURL", redirect);
                }
            }
            Object fbCt = fallbackMeta.get("contentType");
            if (fbCt != null && content != null) {
                String currentMime = content.get("mimeType") != null ? String.valueOf(content.get("mimeType")) : "";
                if (currentMime.isEmpty() || "application/octet-stream".equals(currentMime)) {
                    content.put("mimeType", fbCt);
                }
            }
            Object transferObj = fallbackMeta.get("transferSize");
            long transfer = transferObj instanceof Number ? ((Number) transferObj).longValue() : -1;
            if (transfer >= 0) {
                response.put("_transferSize", transfer);
            }
        }

        Object statusObj2 = response.get("status");
        int status2 = statusObj2 instanceof Number ? ((Number) statusObj2).intValue() : 0;
        if (status2 == 0) {
            response.put("status", 200);
            Object stText = response.get("statusText");
            if (stText == null || String.valueOf(stText).isEmpty()) {
                response.put("statusText", "OK");
            }
        }
        long transferFinal = entry.encodedDataLength >= 0 ? entry.encodedDataLength
                : (response.get("_transferSize") instanceof Number ? ((Number) response.get("_transferSize")).longValue() : -1);
        response.put("_transferSize", transferFinal);

        boolean servedFromCache = Boolean.TRUE.equals(response.get("_servedFromCache"))
                || entry.servedFromCache || entry.fromDiskCache || entry.fromServiceWorker;
        response.put("_servedFromCache", servedFromCache);

        Map<String, Object> snapshot = buildSnapshot(id, entry, request, response, timingsPack);
        TraceUtils.appendJsonLine(st.session.getNetworkFile(), mapOf("type", "resource-snapshot", "snapshot", snapshot));
        TraceDebug.writeLine("har: wrote resource-snapshot " + request.get("url")
                + " bytes=" + (response.get("bodySize") != null ? response.get("bodySize") : -1)
                + " sha=" + (content.get("_sha1") != null ? content.get("_sha1") : "none"));

        Map<String, Object> harEntry = new LinkedHashMap<>();
        harEntry.put("startedDateTime", entry.startedDateIso);
        harEntry.put("time", timingsPack.get("total"));
        harEntry.put("request", request);
        harEntry.put("response", response);
        harEntry.put("cache", new LinkedHashMap<>());
        harEntry.put("timings", timingsPack.get("detail"));
        harEntry.put("pageref", entry.pageRef);
        harEntry.put("_resourceType", entry.resourceType);
        harEntry.put("_fromDiskCache", entry.fromDiskCache);
        harEntry.put("_fromServiceWorker", entry.fromServiceWorker);
        harEntry.put("_encodedDataLength", entry.encodedDataLength);
        harEntry.put("_loaderId", entry.loaderId);
        harEntry.put("_frameId", entry.frameId);
        harEntry.put("_requestTimestamp", entry.requestTimestampSeconds);
        harEntry.put("_servedFromCache", entry.servedFromCache);

        Map<String, Object> resourceEnvelope = new LinkedHashMap<>();
        resourceEnvelope.put("type", "resource");
        resourceEnvelope.put("contextId", st.session.getContextId());
        resourceEnvelope.put("pageId", entry.pageRef);
        resourceEnvelope.put("resourceId", id);
        resourceEnvelope.put("ordinal", TraceUtils.monotonicMs());
        resourceEnvelope.put("ended", true);
        resourceEnvelope.put("resource", harEntry);

        TraceUtils.appendJsonLine(st.session.getNetworkFile(), resourceEnvelope);
    }

    private static Map<String, Object> buildSnapshot(String requestId, EntryState entry, Map<String, Object> request,
                                                     Map<String, Object> response, Map<String, Object> timingsPack) {
        long nowMono = TraceUtils.monotonicMs();
        Map<String, Object> snapshot = new LinkedHashMap<>();
        snapshot.put("_requestId", requestId);
        snapshot.put("_loaderId", entry.loaderId);
        snapshot.put("_frameref", entry.frameId != null ? entry.frameId : entry.pageRef);
        snapshot.put("_resourceType", entry.resourceType);
        snapshot.put("_servedFromCache", entry.servedFromCache);
        snapshot.put("_monotonicTime", nowMono);
        snapshot.put("startedDateTime", entry.startedDateIso);
        snapshot.put("time", timingsPack.get("total"));
        snapshot.put("pageref", entry.pageRef != null ? entry.pageRef : "page-1");
        snapshot.put("request", request);
        snapshot.put("response", response);
        snapshot.put("cache", new LinkedHashMap<>());
        snapshot.put("timings", timingsPack.get("detail"));
        Object serverIp = response.get("serverIPAddress");
        if (serverIp == null && entry.response != null) {
            serverIp = entry.response.get("serverIPAddress");
        }
        snapshot.put("serverIPAddress", serverIp);
        return snapshot;
    }

    private static byte[] readResponseBody(State st, RequestId requestId, EntryState entry) {
        try {
            var bodyObj = st.devTools.send(Network.getResponseBody(requestId));
            String bodyText = bodyObj.getBody();
            boolean b64 = bodyObj.getBase64Encoded();
            byte[] bytes = b64 ? java.util.Base64.getDecoder().decode(bodyText) : bodyText.getBytes(StandardCharsets.UTF_8);
            if (bytes != null && bytes.length > 0) {
                Object url = entry.request != null ? entry.request.get("url") : "";
                TraceDebug.writeLine("har: CDP body bytes=" + bytes.length + " url=" + (url != null ? url : ""));
            }
            return bytes;
        } catch (Throwable t) {
            TraceDebug.writeLine("har: getResponseBody error " + t.getMessage());
            return null;
        }
    }

    @SuppressWarnings("unchecked")
    private static Map<String, Object> httpFetchBody(EntryState entry) {
        HttpURLConnection conn = null;
        java.io.InputStream stream = null;
        try {
            String url = entry.request != null && entry.request.get("url") != null
                    ? String.valueOf(entry.request.get("url"))
                    : (entry.response != null ? String.valueOf(entry.response.get("url")) : null);
            if (url == null || url.isEmpty()) {
                return null;
            }
            conn = (HttpURLConnection) new URL(url).openConnection();
            conn.setInstanceFollowRedirects(true);
            conn.setRequestProperty("Accept-Encoding", "identity");

            String method = entry.request != null && entry.request.get("method") != null
                    ? String.valueOf(entry.request.get("method")).toUpperCase()
                    : "GET";
            conn.setRequestMethod(method);

            List<Map<String, Object>> headers = entry.request != null
                    ? (List<Map<String, Object>>) entry.request.getOrDefault("headers", new ArrayList<>())
                    : new ArrayList<>();
            applyPassthroughHeader(conn, headers, "user-agent", "User-Agent");
            applyPassthroughHeader(conn, headers, "accept", "Accept");
            applyPassthroughHeader(conn, headers, "accept-language", "Accept-Language");
            applyPassthroughHeader(conn, headers, "referer", "Referer");
            applyPassthroughHeader(conn, headers, "origin", "Origin");
            String cookieHeader = cookiesHeader(entry.request != null
                    ? (List<Map<String, Object>>) entry.request.get("cookies")
                    : null);
            if (cookieHeader != null && !cookieHeader.isEmpty()) {
                conn.setRequestProperty("Cookie", cookieHeader);
            }

            Object postDataObj = entry.request != null ? entry.request.get("postData") : null;
            if (("POST".equals(method) || "PUT".equals(method) || "PATCH".equals(method))
                    && postDataObj instanceof Map && ((Map<?, ?>) postDataObj).get("text") != null) {
                Map<?, ?> postData = (Map<?, ?>) postDataObj;
                byte[] payload = String.valueOf(postData.get("text")).getBytes(StandardCharsets.UTF_8);
                conn.setDoOutput(true);
                Object mime = postData.get("mimeType");
                if (mime != null) {
                    conn.setRequestProperty("Content-Type", String.valueOf(mime));
                }
                OutputStream os = conn.getOutputStream();
                try {
                    os.write(payload);
                } finally {
                    try {
                        os.close();
                    } catch (Throwable ignored) {
                    }
                }
            }

            int code = conn.getResponseCode();
            stream = (code >= 400 ? conn.getErrorStream() : conn.getInputStream());
            byte[] bytes = stream != null ? stream.readAllBytes() : null;

            Map<String, Object> meta = new LinkedHashMap<>();
            meta.put("status", code);
            meta.put("statusText", conn.getResponseMessage());
            Map<String, List<String>> headerFields = conn.getHeaderFields();
            List<Map<String, Object>> respHeaders = new ArrayList<>();
            if (headerFields != null) {
                for (Map.Entry<String, List<String>> e : headerFields.entrySet()) {
                    if (e.getKey() == null) continue;
                    Map<String, Object> h = new LinkedHashMap<>();
                    h.put("name", e.getKey());
                    List<String> vals = e.getValue();
                    String joined = vals != null ? String.join(", ", vals) : "";
                    h.put("value", joined);
                    respHeaders.add(h);
                }
            }
            meta.put("headers", respHeaders);
            long len = conn.getContentLengthLong();
            meta.put("transferSize", len >= 0 ? len : -1);
            meta.put("contentType", conn.getContentType());
            if (bytes != null) {
                meta.put("bytes", bytes);
            }
            return meta;
        } catch (Throwable t) {
            TraceDebug.writeLine("har: httpFetchBody error " + t.getMessage());
            return null;
        } finally {
            if (stream != null) {
                try {
                    stream.close();
                } catch (Throwable ignored) {
                }
            }
            if (conn != null) {
                conn.disconnect();
            }
        }
    }

    private static void applyPassthroughHeader(HttpURLConnection conn, List<Map<String, Object>> headers, String inName, String outName) {
        if (headers == null) {
            return;
        }
        for (Map<String, Object> h : headers) {
            String name = h.get("name") != null ? String.valueOf(h.get("name")) : "";
            if (inName.equalsIgnoreCase(name)) {
                Object v = h.get("value");
                if (v != null) {
                    conn.setRequestProperty(outName, String.valueOf(v));
                }
                break;
            }
        }
    }

    private static String cookiesHeader(List<Map<String, Object>> cookies) {
        if (cookies == null || cookies.isEmpty()) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        for (Map<String, Object> c : cookies) {
            String name = c.get("name") != null ? String.valueOf(c.get("name")) : "";
            String value = c.get("value") != null ? String.valueOf(c.get("value")) : "";
            if (!first) {
                sb.append("; ");
            }
            sb.append(name).append("=").append(value);
            first = false;
        }
        return sb.toString();
    }

    private static boolean shouldCaptureBody(String mime) {
        if (mime == null || mime.isEmpty()) {
            return false;
        }
        String lower = mime.toLowerCase();
        return lower.contains("text") || lower.contains("javascript") || lower.contains("json")
                || lower.contains("xml") || lower.contains("svg") || lower.contains("font");
    }

    private static boolean shouldCaptureUrl(String url) {
        if (url == null || url.isEmpty()) {
            return false;
        }
        String lower = url.toLowerCase();
        if (lower.contains("fonts.googleapis.com")) {
            return true;
        }
        int q = lower.indexOf("?");
        String base = q >= 0 ? lower.substring(0, q) : lower;
        String[] exts = {".css", ".js", ".mjs", ".json", ".html", ".htm", ".xml", ".svg", ".woff", ".woff2", ".ttf", ".otf", ".eot", ".png", ".jpg", ".jpeg", ".gif", ".ico"};
        for (String ext : exts) {
            if (base.endsWith(ext)) {
                return true;
            }
        }
        return false;
    }

    private static String extensionFromUrl(String url) {
        if (url == null || url.isEmpty()) {
            return null;
        }
        int q = url.indexOf("?");
        String base = q >= 0 ? url.substring(0, q) : url;
        int dot = base.lastIndexOf('.');
        if (dot < 0 || dot == base.length() - 1) {
            return null;
        }
        String ext = base.substring(dot + 1).toLowerCase();
        switch (ext) {
            case "css":
            case "js":
            case "mjs":
            case "json":
            case "html":
            case "htm":
            case "xml":
            case "svg":
            case "woff":
            case "woff2":
            case "ttf":
            case "otf":
            case "eot":
            case "png":
            case "jpg":
            case "jpeg":
            case "gif":
            case "ico":
                return ext;
            default:
                return null;
        }
    }

    private static String extensionForMime(String mime) {
        if (mime == null || mime.isEmpty()) {
            return null;
        }
        String lower = mime.toLowerCase();
        if (lower.contains("text/css")) return "css";
        if (lower.contains("javascript")) return "js";
        if (lower.contains("json")) return "json";
        if (lower.contains("html")) return "html";
        if (lower.contains("xml")) return "xml";
        if (lower.contains("svg")) return "svg";
        if (lower.contains("woff2")) return "woff2";
        if (lower.contains("woff")) return "woff";
        if (lower.contains("ttf")) return "ttf";
        if (lower.contains("otf")) return "otf";
        if (lower.contains("png")) return "png";
        if (lower.contains("jpeg") || lower.contains("jpg")) return "jpeg";
        if (lower.contains("gif")) return "gif";
        if (lower.contains("ico")) return "ico";
        return null;
    }

    private static Map<String, Object> mapOf(Object k1, Object v1, Object k2, Object v2) {
        Map<String, Object> m = new LinkedHashMap<>();
        m.put(String.valueOf(k1), v1);
        m.put(String.valueOf(k2), v2);
        return m;
    }

    private static Map<String, Object> mapOf(Object k1, Object v1, Object k2, Object v2, Object k3, Object v3,
                                             Object k4, Object v4, Object k5, Object v5, Object k6, Object v6,
                                             Object k7, Object v7, Object k8, Object v8, Object k9, Object v9,
                                             Object k10, Object v10) {
        Map<String, Object> m = new LinkedHashMap<>();
        m.put(String.valueOf(k1), v1);
        m.put(String.valueOf(k2), v2);
        m.put(String.valueOf(k3), v3);
        m.put(String.valueOf(k4), v4);
        m.put(String.valueOf(k5), v5);
        m.put(String.valueOf(k6), v6);
        m.put(String.valueOf(k7), v7);
        m.put(String.valueOf(k8), v8);
        m.put(String.valueOf(k9), v9);
        m.put(String.valueOf(k10), v10);
        return m;
    }

    public static boolean attachIfNeeded(TraceSession session) {
        try {
            State current = state;
            if (current != null && current.enabled) {
                current.session = session;
                current.entries = new ConcurrentHashMap<>();
                return true;
            }
            WebDriver driver = DriverFactory.getWebDriver();
            if (!(driver instanceof HasDevTools)) {
                TraceDebug.writeLine("har: driver lacks DevTools");
                return false;
            }
            DevTools dt = ((HasDevTools) driver).getDevTools();
            if (dt == null) {
                TraceDebug.writeLine("har: getDevTools null");
                return false;
            }
            dt.createSession();
            // Selenium CDP binding for Network.enable now exposes `enableDurableMessages`.
            // Enabling durable messages requires `maxTotalBufferSize` to be provided; we don't need it for trace.
            dt.send(Network.enable(Optional.empty(), Optional.empty(), Optional.empty(), Optional.empty(), Optional.of(false)));
            dt.send(Network.setCacheDisabled(true));
            State st = new State();
            st.devTools = dt;
            st.session = session;
            st.enabled = true;
            state = st;
            addListeners(st);
            TraceDebug.writeLine("har: attached to DevTools Network domain");
            return true;
        } catch (Throwable t) {
            TraceDebug.writeLine("har: attach failed " + t.getMessage());
            return false;
        }
    }

    public static void ensureCacheDisabled() {
        try {
            State st = state;
            if (st != null && st.enabled) {
                st.devTools.send(Network.setCacheDisabled(true));
            }
        } catch (Throwable t) {
            TraceDebug.writeLine("har: ensureCacheDisabled error " + t.getMessage());
        }
    }

    public static void awaitIdle(long timeoutMs) {
        State st = state;
        if (st == null) {
            return;
        }
        long deadline = System.currentTimeMillis() + timeoutMs;
        while (!st.entries.isEmpty() && System.currentTimeMillis() < deadline) {
            try {
                Thread.sleep(50);
            } catch (InterruptedException ignored) {
                Thread.currentThread().interrupt();
                break;
            }
        }
        if (!st.entries.isEmpty()) {
            TraceDebug.writeLine("har: awaitIdle timed out with " + st.entries.size() + " inflight requests");
        }
    }

    public static void stopAndFlush(long timeoutMs) {
        awaitIdle(timeoutMs);
        State st = state;
        if (st != null && st.enabled) {
            TraceDebug.writeLine("har: stop (remaining=" + st.entries.size() + ")");
        }
        state = null;
    }
}
