package com.kms.katalon.core.webui.trace

import java.lang.reflect.Array
import java.lang.reflect.InvocationTargetException
import java.lang.reflect.Method
import java.lang.reflect.Modifier

import com.kms.katalon.core.testobject.TestObject
import com.kms.katalon.core.trace.SourceLocator
import com.kms.katalon.core.trace.TraceDebug
import com.kms.katalon.core.trace.TraceHolder
import com.kms.katalon.core.webui.keyword.WebUiBuiltInKeywords as WebUI

import groovy.json.JsonOutput

class KeywordInterceptor {
    private static boolean installed = false
    private static final ThreadLocal<Boolean> IN = ThreadLocal.withInitial { false }
    private static final int ARG_SOURCE_LIMIT = 8192
    static void reset() {
        installed = false; IN.set(false)
    }

    static void install() {
        if (installed) return
            installed = true
        try {
            TraceDebug.writeLine('KeywordInterceptor: installed for WebUI.*')
        } catch (Throwable ignored) {}

        def mc = WebUI.metaClass
        mc.'static'.invokeMethod = { String name, Object args ->
            if (IN.get()) {
                return invokeReflective(name, args)
            }
            IN.set(true)
            def session = TraceHolder.session
            String title = "WebUI.${name}"
            String callId = null
            Map meta = [:]
            if (session != null && session.recording) {
                try {
                    meta = buildMeta(name)
                    String argDetails = describeArgsForSource(title, args)
                    Map extras = argDetails ? [argsSourceText: argDetails] : null
                    callId = session.beginStep(title, buildParams(args), meta, extras)
                } catch (Throwable t) {
                    TraceDebug.writeLine('KeywordInterceptor: beginStep error ' + t.message)
                }
            }
            try {
                def result = invokeReflective(name, args)
                try {
                    def s = TraceHolder.session
                    if (s != null) {
                        if (name == 'openBrowser' || name == 'navigateToUrl') {
                            if (HarTracer.attachIfNeeded(s)) HarTracer.ensureCacheDisabled()
                        }
                    }
                } catch (Throwable ignored) {}
                if (callId != null) session.endStep(callId, 'ok', null, meta)
                return result
            } catch (Throwable t) {
                if (callId != null) session.endStep(callId, null, t, meta)
                try {
                    TraceDebug.writeLine('KeywordInterceptor: call ' + name + ' threw ' + t.class.simpleName + ': ' + t.message)
                } catch (Throwable ignored) {}
                throw t
            } finally {
                IN.set(false)
            }
        }
    }

    private static Object invokeReflective(String name, Object args) {
        try {
            Object[] arr = toArray(args)
            Method m = findStatic(WebUI.class, name, arr)
            if (m == null) {
                def mm = WebUI.metaClass.getStaticMetaMethod(name, arr.collect { it?.getClass() ?: Object } as Class[])
                if (mm != null) return mm.invoke(WebUI, arr)
                m = WebUI.class.methods.find { it.name == name && Modifier.isStatic(it.modifiers) && it.parameterCount == 0 }
                if (m != null) return m.invoke(null)
                throw new NoSuchMethodException("WebUI.${name} with ${arr.length} args")
            }
            return m.invoke(null, arr)
        } catch (InvocationTargetException e) {
            throw e.getCause() ?: e
        }
    }

    private static Method findStatic(Class cls, String name, Object[] args) {
        def candidates = cls.methods.findAll { it.name == name && Modifier.isStatic(it.modifiers) }
        def byCount = candidates.findAll { it.parameterCount == args.length }
        Method match = byCount.find { compatible(it.parameterTypes, args) }
        if (match != null) return match
        match = candidates.find { it.isVarArgs() }
        return match
    }

    private static boolean compatible(Class<?>[] types, Object[] args) {
        for (int i = 0; i < types.length; i++) {
            Class t = types[i]
            Object a = args[i]
            if (a == null) continue
                if (t.isAssignableFrom(a.getClass())) continue
                if (t == String.class && (a instanceof GString)) continue
                if (t.isPrimitive()) {
                    if (t == int.class && a instanceof Number) continue
                        if (t == long.class && a instanceof Number) continue
                        if (t == boolean.class && a instanceof Boolean) continue
                        if (t == double.class && a instanceof Number) continue
                        if (t == float.class && a instanceof Number) continue
                }
            return false
        }
        return true
    }

    private static Object[] toArray(Object args) {
        if (args == null) return new Object[0]
        if (args instanceof Object[]) return (Object[]) args
        if (args instanceof List) return ((List) args).toArray()
        return [args] as Object[]
    }

    private static Map buildParams(Object args) {
        Map params = [:]
        String preview = previewArgs(args)
        if (preview) params.argsPreview = preview
        List<String> testObjectIds = extractTestObjectIds(args)
        if (!testObjectIds.isEmpty()) {
            if (testObjectIds.size() == 1) params.testObjectId = testObjectIds.first()
            else params.testObjectIds = testObjectIds
        }
        return params
    }

    private static String previewArgs(Object args) {
        try {
            def arr = toArray(args)
            return arr.collect { formatArgPreview(it) }.join(', ')
        } catch (Throwable ignored) {
            return ''
        }
    }

    private static String formatArgPreview(Object arg) {
        if (arg == null) return 'null'
        if (arg instanceof TestObject) {
            String id = safeTestObjectId((TestObject) arg)
            return id ? "TestObject(${id})" : 'TestObject'
        }
        return String.valueOf(arg)
    }

    private static List<String> extractTestObjectIds(Object args) {
        List<String> ids = []
        try {
            def arr = toArray(args)
            for (Object arg : arr) {
                if (arg instanceof TestObject) {
                    String id = safeTestObjectId((TestObject) arg)
                    if (id) ids.add(id)
                }
            }
        } catch (Throwable ignored) {}
        return ids
    }

    private static String safeTestObjectId(TestObject obj) {
        if (obj == null) return null
        try {
            String id = obj.getObjectId()
            if (id?.trim()) return id
        } catch (Throwable ignored) {}
        return null
    }

    private static Map buildMeta(String keywordName) {
        Map location = SourceLocator.capture()
        Map meta = [apiName: "WebUI.${keywordName}", category: 'WebUI']
        if (location && !location.isEmpty()) meta.location = location
        return meta
    }

    private static String describeArgsForSource(String keywordTitle, Object args) {
        try {
            def arr = toArray(args)
            if (!arr || arr.length == 0) return null
            StringBuilder sb = new StringBuilder()
            sb.append("${keywordTitle} arguments (${arr.length})")
            for (int i = 0; i < arr.length; i++) {
                Object arg = arr[i]
                sb.append("\n\n#${i + 1}")
                sb.append("\nkind: ${argKind(arg)}")
                if (arg instanceof TestObject) {
                    String id = safeTestObjectId((TestObject) arg) ?: 'unknown'
                    sb.append("\nobjectId: ${id}")
                }
                sb.append("\nvalue: ${stringifyArgValue(arg)}")
            }
            String text = sb.toString()
            if (text.length() > ARG_SOURCE_LIMIT) return text.substring(0, ARG_SOURCE_LIMIT) + "\n... truncated ..."
            return text
        } catch (Throwable ignored) {
            return null
        }
    }

    private static String argKind(Object arg) {
        if (arg == null) return 'null'
        if (arg instanceof TestObject) return 'TestObject'
        return arg.getClass().getName()
    }

    private static String stringifyArgValue(Object arg) {
        if (arg == null) return 'null'
        if (arg instanceof CharSequence) return "\"${arg}\""
        if (arg instanceof Number || arg instanceof Boolean) return String.valueOf(arg)
        if (arg instanceof TestObject) {
            String id = safeTestObjectId((TestObject) arg) ?: 'unknown'
            return "TestObject(${id})"
        }
        if (arg instanceof Map || arg instanceof Collection) return safeJson(arg)
        if (arg.getClass().isArray()) {
            int len = Array.getLength(arg)
            List<Object> values = []
            for (int i = 0; i < len; i++) values << Array.get(arg, i)
            return safeJson(values)
        }
        return String.valueOf(arg)
    }

    private static String safeJson(Object value) {
        try {
            return JsonOutput.prettyPrint(JsonOutput.toJson(value))
        } catch (Throwable ignored) {
            return String.valueOf(value)
        }
    }
}
