package com.kms.katalon.core.webui.trace;

import com.kms.katalon.core.trace.TraceDebug;
import com.kms.katalon.core.trace.TraceHolder;
import com.kms.katalon.core.trace.TraceUtils;
import com.kms.katalon.core.webui.driver.DriverFactory;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.TakesScreenshot;
import org.openqa.selenium.WebDriver;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

public class TraceSession {
    public static final int VERSION = 8;

    private final File tracesDir;
    private final File resourcesDir;
    private final File traceFile;
    private final File networkFile;
    private final String contextId = UUID.randomUUID().toString();
    private final Set<String> resourceKeys = ConcurrentHashMap.newKeySet();
    private volatile boolean recording = false;
    private String snapshotStreamerName;

    public TraceSession(File tracesDir) {
        this.tracesDir = tracesDir;
        this.resourcesDir = new File(tracesDir, "resources");
        this.traceFile = new File(tracesDir, "trace.trace");
        this.networkFile = new File(tracesDir, "trace.network");
        TraceUtils.ensureDir(resourcesDir);
        traceFile.delete();
        networkFile.delete();
        File parent = networkFile.getParentFile();
        if (parent != null) {
            parent.mkdirs();
        }
        try {
            networkFile.createNewFile();
        } catch (Throwable ignored) {
        }
    }

    public void start(String title) {
        WebDriver driver = null;
        try {
            driver = DriverFactory.getWebDriver();
        } catch (Throwable ignored) {
        }
        String browserName = safeBrowserName(driver);
        Map<String, Object> options = new LinkedHashMap<>();
        Map<String, Object> viewport = new LinkedHashMap<>();
        viewport.put("width", 1280);
        viewport.put("height", 800);
        options.put("viewport", viewport);
        options.put("deviceScaleFactor", 1);
        options.put("isMobile", false);

        Map<String, Object> ctx = new LinkedHashMap<>();
        ctx.put("version", VERSION);
        ctx.put("type", "context-options");
        ctx.put("origin", "testRunner");
        ctx.put("browserName", browserName);
        ctx.put("platform", System.getProperty("os.name"));
        ctx.put("wallTime", Instant.now().toEpochMilli());
        ctx.put("monotonicTime", TraceUtils.monotonicMs());
        ctx.put("sdkLanguage", "java");
        ctx.put("title", title);
        ctx.put("contextId", contextId);
        ctx.put("options", options);

        TraceUtils.appendJsonLine(traceFile, ctx);
        TraceDebug.writeLine("TraceSession.start: context-options written, driverPresent=" + (driver != null));
        try {
            if (driver != null) {
                snapshotStreamerName = SnapshotAgent.installAndGetName();
                TraceDebug.writeLine("TraceSession.start: snapshot streamer= " + snapshotStreamerName);
            }
        } catch (Throwable t) {
            TraceDebug.writeLine("TraceSession.start: snapshot inject error: " + t.getMessage());
        }
        recording = true;
    }

    public String beginStep(String title, Map<String, Object> params) {
        return beginStep(title, params, new LinkedHashMap<>(), null);
    }

    @SuppressWarnings("unchecked")
    public String beginStep(String title, Map<String, Object> params, Map<String, Object> meta, Map<String, Object> extras) {
        String callId = UUID.randomUUID().toString();
        Map<String, Object> effectiveParams = params != null ? new LinkedHashMap<>(params) : new LinkedHashMap<>();

        Map<String, Object> ev = new LinkedHashMap<>();
        ev.put("type", "before");
        ev.put("callId", callId);
        ev.put("startTime", TraceUtils.monotonicMs());
        ev.put("wallTime", System.currentTimeMillis());
        ev.put("title", title);
        ev.put("class", "Test");
        ev.put("method", "step");
        ev.put("params", effectiveParams);
        ev.put("stepId", callId);
        ev.put("pageId", currentPageId());

        if (meta != null) {
            Object apiName = meta.get("apiName");
            if (apiName != null) {
                ev.put("apiName", apiName);
            }
            Object category = meta.get("category");
            if (category != null) {
                ev.put("category", category);
            }
        }

        Map<String, Object> locBefore = sanitizeLocation(meta != null ? meta.get("location") : null);
        if (locBefore != null) {
            ev.put("location", locBefore);
        }

        List<Map<String, Object>> stackFrames = new ArrayList<>();
        String argsSourceText = extras != null ? (String) extras.get("argsSourceText") : null;
        Map<String, Object> argsSource = maybeCreateArgsSource(callId, title, argsSourceText);
        if (argsSource != null) {
            Object file = argsSource.get("file");
            Object sha = argsSource.get("sha");
            Object frame = argsSource.get("frame");
            if (file != null && !effectiveParams.containsKey("argsSourceFile")) {
                effectiveParams.put("argsSourceFile", file);
            }
            if (sha != null && !effectiveParams.containsKey("argsSourceSha")) {
                effectiveParams.put("argsSourceSha", sha);
            }
            if (frame instanceof Map) {
                stackFrames.add((Map<String, Object>) frame);
            }
        }
        if (!stackFrames.isEmpty()) {
            ev.put("stack", stackFrames);
        }
        String beforeSnapshot = "before@" + callId;
        ev.put("beforeSnapshot", beforeSnapshot);

        TraceUtils.appendJsonLine(traceFile, ev);
        try {
            captureFrameSnapshot(callId, beforeSnapshot);
            TraceDebug.writeLine("beginStep: captured before snapshot for " + callId);
        } catch (Throwable t) {
            TraceDebug.writeLine("beginStep snapshot err " + t.getMessage());
        }
        return callId;
    }

    public void endStep(String callId, Object result, Map<String, Object> meta) {
        endStep(callId, result, null, meta);
    }

    public void endStep(String callId, Object result, Throwable err, Map<String, Object> meta) {
        Map<String, Object> after = new LinkedHashMap<>();
        after.put("type", "after");
        after.put("callId", callId);
        after.put("endTime", TraceUtils.monotonicMs());
        after.put("wallTime", System.currentTimeMillis());
        after.put("result", result);

        if (meta != null) {
            Object apiName = meta.get("apiName");
            if (apiName != null) {
                after.put("apiName", apiName);
            }
            Object category = meta.get("category");
            if (category != null) {
                after.put("category", category);
            }
        }

        Map<String, Object> locAfter = sanitizeLocation(meta != null ? meta.get("location") : null);
        if (locAfter != null) {
            after.put("location", locAfter);
        }
        if (err != null) {
            Map<String, Object> error = new LinkedHashMap<>();
            error.put("name", err.getClass().getSimpleName());
            error.put("message", err.getMessage());
            after.put("error", error);
        }
        String afterSnapshot = "after@" + callId;
        after.put("afterSnapshot", afterSnapshot);

        TraceUtils.appendJsonLine(traceFile, after);
        try {
            captureFrameSnapshot(callId, afterSnapshot);
        } catch (Throwable t) {
            TraceDebug.writeLine("endStep snapshot err " + t.getMessage());
        }
        try {
            captureScreencastFrame();
        } catch (Throwable ignored) {
        }
    }

    public void captureScreencastFrame() {
        WebDriver driver;
        try {
            driver = DriverFactory.getWebDriver();
        } catch (Throwable ignored) {
            return;
        }
        if (!(driver instanceof TakesScreenshot)) {
            return;
        }
        byte[] png = ((TakesScreenshot) driver).getScreenshotAs(OutputType.BYTES);
        byte[] jpeg;
        try {
            jpeg = TraceUtils.pngToJpeg(png);
        } catch (Throwable t) {
            jpeg = png;
        }
        long ts = TraceUtils.monotonicMs();
        String pageId = currentPageId();
        String key = pageId + "-" + ts + ".jpeg";
        if (!resourceKeys.add(key)) {
            return;
        }
        File out = new File(resourcesDir, key);
        TraceUtils.writeBytes(out, jpeg);
        TraceDebug.writeLine("screencast: wrote " + out.getName() + " size=" + jpeg.length);

        Map<String, Object> ev = new LinkedHashMap<>();
        ev.put("type", "screencast-frame");
        ev.put("pageId", pageId);
        ev.put("sha1", key);
        ev.put("width", 1280);
        ev.put("height", 800);
        ev.put("timestamp", ts);
        TraceUtils.appendJsonLine(traceFile, ev);
    }

    @SuppressWarnings("unchecked")
    public void captureFrameSnapshot(String callId, String snapshotName) {
        if (snapshotStreamerName == null) {
            try {
                snapshotStreamerName = SnapshotAgent.installAndGetName();
                TraceDebug.writeLine("captureFrameSnapshot: installed streamer=" + snapshotStreamerName);
            } catch (Throwable t) {
                TraceDebug.writeLine("captureFrameSnapshot: install error " + t.getMessage());
            }
            if (snapshotStreamerName == null) {
                return;
            }
        }
        Object data = SnapshotAgent.captureSnapshot(snapshotStreamerName);
        if (!(data instanceof Map)) {
            try {
                snapshotStreamerName = SnapshotAgent.installAndGetName();
            } catch (Throwable ignored) {
            }
            if (snapshotStreamerName == null) {
                return;
            }
            data = SnapshotAgent.captureSnapshot(snapshotStreamerName);
        }
        if (!(data instanceof Map)) {
            return;
        }
        Map<String, Object> snap = (Map<String, Object>) data;
        List<Map<String, Object>> overrides = new ArrayList<>();
        Object roObj = snap.get("resourceOverrides");
        List<?> ro = roObj instanceof List ? (List<?>) roObj : new ArrayList<>();
        for (Object item : ro) {
            if (!(item instanceof Map)) {
                continue;
            }
            Map<?, ?> m = (Map<?, ?>) item;
            String url = (String) m.get("url");
            Object content = m.get("content");
            String contentType = (String) m.get("contentType");
            if (content instanceof CharSequence) {
                byte[] bytes = content.toString().getBytes(StandardCharsets.UTF_8);
                String sha = TraceUtils.sha1Hex(bytes);
                String ext = extFromContentType(contentType);
                String name = sha + "." + ext;
                File out = new File(resourcesDir, name);
                if (resourceKeys.add(name) || !out.exists()) {
                    TraceUtils.writeBytes(out, bytes);
                    TraceDebug.writeLine("snapshot override: wrote " + name + " bytes=" + bytes.length);
                }
                Map<String, Object> override = new LinkedHashMap<>();
                override.put("url", url);
                override.put("sha1", name);
                if (contentType != null) {
                    override.put("contentType", contentType);
                }
                overrides.add(override);
            } else if (content instanceof Number) {
                Map<String, Object> override = new LinkedHashMap<>();
                override.put("url", url);
                override.put("ref", ((Number) content).intValue());
                overrides.add(override);
            }
        }

        Map<String, Object> snapshot = new LinkedHashMap<>();
        snapshot.put("snapshotName", snapshotName);
        snapshot.put("callId", callId);
        snapshot.put("pageId", currentPageId());
        snapshot.put("frameId", "main");
        Object url = snap.get("url");
        snapshot.put("frameUrl", url != null ? url : "");
        snapshot.put("timestamp", TraceUtils.monotonicMs());
        Object wallTime = snap.get("wallTime");
        snapshot.put("wallTime", wallTime != null ? wallTime : System.currentTimeMillis());
        Object collectionTime = snap.get("collectionTime");
        snapshot.put("collectionTime", collectionTime != null ? collectionTime : 0);
        snapshot.put("doctype", snap.get("doctype"));
        snapshot.put("html", snap.get("html"));
        snapshot.put("resourceOverrides", overrides);
        Object viewport = snap.get("viewport");
        if (viewport == null) {
            Map<String, Object> vp = new LinkedHashMap<>();
            vp.put("width", 1280);
            vp.put("height", 800);
            viewport = vp;
        }
        snapshot.put("viewport", viewport);
        snapshot.put("isMainFrame", true);

        Map<String, Object> frameSnapshot = new LinkedHashMap<>();
        frameSnapshot.put("type", "frame-snapshot");
        frameSnapshot.put("snapshot", snapshot);
        TraceUtils.appendJsonLine(traceFile, frameSnapshot);
    }

    public void stopChunk(File zipFile) {
        try {
            HarTracer.awaitIdle(500);
        } catch (Throwable ignored) {
        }
        List<Map<String, Object>> entries = new ArrayList<>();

        Map<String, Object> e1 = new LinkedHashMap<>();
        e1.put("name", "trace.trace");
        e1.put("file", traceFile);
        entries.add(e1);

        Map<String, Object> e2 = new LinkedHashMap<>();
        e2.put("name", "trace.network");
        e2.put("file", networkFile);
        entries.add(e2);

        File[] files = resourcesDir.listFiles();
        if (files != null) {
            for (File f : files) {
                Map<String, Object> ent = new LinkedHashMap<>();
                ent.put("name", "resources/" + f.getName());
                ent.put("file", f);
                entries.add(ent);
            }
        }
        TraceDebug.writeLine("stopChunk: entries=" + entries.stream().map(m -> String.valueOf(m.get("name"))).reduce((a, b) -> a + ", " + b).orElse(""));
        TraceUtils.zipEntries(zipFile, entries);
        recording = false;
    }

    private Map<String, Object> maybeCreateArgsSource(String callId, String title, String content) {
        if (content == null || content.trim().isEmpty()) {
            return null;
        }
        try {
            String safeSegment = sanitizeArgsSegment(title != null ? title : "keyword");
            String pseudoFile = "/trace-args/" + safeSegment + "/" + callId + ".txt";
            byte[] bytes = content.getBytes(StandardCharsets.UTF_8);
            String shaForPath = TraceUtils.sha1Hex(pseudoFile.getBytes(StandardCharsets.UTF_8));
            File out = new File(resourcesDir, "src@" + shaForPath + ".txt");
            if (!out.exists()) {
                TraceUtils.writeBytes(out, bytes);
            }
            Map<String, Object> frame = new LinkedHashMap<>();
            frame.put("file", pseudoFile);
            frame.put("line", 1);
            frame.put("column", 1);

            Map<String, Object> result = new LinkedHashMap<>();
            result.put("file", pseudoFile);
            result.put("frame", frame);
            result.put("sha", shaForPath);
            return result;
        } catch (Throwable t) {
            TraceDebug.writeLine("args source error: " + t.getMessage());
            return null;
        }
    }

    private static String sanitizeArgsSegment(String title) {
        String base = title != null ? title : "keyword";
        base = base.replaceAll("[^A-Za-z0-9._-]+", "_");
        base = base.replaceAll("_{2,}", "_");
        base = base.replaceAll("^_", "").replaceAll("_$", "");
        return base.isEmpty() ? "keyword" : base;
    }

    private static String extFromContentType(String ct) {
        if (ct == null) {
            return "dat";
        }
        String lower = ct.toLowerCase();
        if (lower.startsWith("text/css")) return "css";
        if (lower.startsWith("text/html")) return "html";
        if (lower.startsWith("application/json")) return "json";
        return "dat";
    }

    @SuppressWarnings("unchecked")
    private static Map<String, Object> sanitizeLocation(Object raw) {
        if (!(raw instanceof Map)) {
            return null;
        }
        Map<String, Object> out = new LinkedHashMap<>();
        Map<?, ?> src = (Map<?, ?>) raw;
        for (Map.Entry<?, ?> e : src.entrySet()) {
            Object k = e.getKey();
            Object v = e.getValue();
            if (k != null && v != null) {
                out.put(String.valueOf(k), v);
            }
        }
        return out.isEmpty() ? null : out;
    }

    private static String currentPageId() {
        try {
            WebDriver driver = DriverFactory.getWebDriver();
            if (driver == null) {
                return "page-1";
            }
            String handle = driver.getWindowHandle();
            return handle != null ? handle : "page-1";
        } catch (Throwable t) {
            return "page-1";
        }
    }

    private static String safeBrowserName(WebDriver driver) {
        try {
            return driver != null && driver.getClass() != null && driver.getClass().getSimpleName() != null
                    ? driver.getClass().getSimpleName()
                    : "browser";
        } catch (Throwable t) {
            return "browser";
        }
    }

    public boolean isRecording() {
        return recording;
    }

    public void setRecording(boolean recording) {
        this.recording = recording;
    }

    public File getTracesDir() {
        return tracesDir;
    }

    public File getResourcesDir() {
        return resourcesDir;
    }

    public File getTraceFile() {
        return traceFile;
    }

    public File getNetworkFile() {
        return networkFile;
    }

    public String getContextId() {
        return contextId;
    }

    public Set<String> getResourceKeys() {
        return resourceKeys;
    }
}
