package com.kms.katalon.core.webui.util;

import static java.util.Objects.requireNonNull;

import java.io.IOException;
import java.lang.reflect.Field;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.remote.Command;
import org.openqa.selenium.remote.CommandExecutor;
import org.openqa.selenium.remote.CommandInfo;
import org.openqa.selenium.remote.DriverCommand;
import org.openqa.selenium.remote.HttpCommandExecutor;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.openqa.selenium.remote.Response;
import org.openqa.selenium.remote.codec.w3c.W3CHttpCommandCodec;
import org.openqa.selenium.remote.codec.w3c.W3CHttpResponseCodec;
import org.openqa.selenium.remote.http.ClientConfig;
import org.openqa.selenium.remote.http.HttpClient.Factory;

import com.kms.katalon.core.appium.util.AppiumDriverUtil;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.network.ProxyInformation;
import com.kms.katalon.core.selenium.remote.http.ConfiguredHttpClientFactory;
import com.kms.katalon.logging.LogUtil;
import com.kms.katalon.selenium.constant.SeleniumConstants;
import com.kms.katalon.selenium.constant.SeleniumW3CCapabilityConstant;
import com.kms.katalon.selenium.driver.CRemoteWebDriver;
import com.kms.katalon.selenium.filter.SeleniumAcceptEncodingFilter;
import com.kms.katalon.selenium.filter.SeleniumBasicAuthorizationFilter;
import com.kms.katalon.selenium.filter.SeleniumCreateSessionPayloadFilter;
import com.kms.katalon.selenium.filter.SeleniumCreateSessionResponseFilter;
import com.kms.katalon.selenium.filter.SeleniumProxyBasicAuthorizationFilter;
import com.kms.katalon.selenium.filter.SeleniumResponseContentEncodingDecompressionFilter;

public class WebDriverUtil {
    public static CommandExecutor createCommandExecutorWithRunningSession(final String remoteServerUrl,
            final String sessionId, final String webSocketUrl) throws MalformedURLException {
        return createCommandExecutorWithRunningSession(new URL(remoteServerUrl), sessionId, webSocketUrl);
    }

    public static CommandExecutor createCommandExecutorWithRunningSession(final URL remoteServerUrl,
            final String sessionId, final String webSocketUrl) throws MalformedURLException {
        return new HttpCommandExecutor(remoteServerUrl) {
            @Override
            public Response execute(Command command) throws IOException {
                Response response = null;

                // attach the running session instead of open brand-new one to avoid exception
                if (StringUtils.equals(DriverCommand.NEW_SESSION, command.getName())) {
                    response = new Response();
                    response.setSessionId(sessionId);
                    response.setState("success");

                    if (StringUtils.isBlank(webSocketUrl)) {
                        response.setValue(Collections.<String, String> emptyMap());
                    } else {
                        response.setValue(Collections.singletonMap(SeleniumW3CCapabilityConstant.WEB_SOCKET_URL_CAP,
                                webSocketUrl));
                    }

                    try {
                        // reflect the commandCodec
                        Field commandCodec = null;
                        commandCodec = this.getClass().getSuperclass().getDeclaredField("commandCodec");
                        commandCodec.setAccessible(true);
                        commandCodec.set(this, new W3CHttpCommandCodec());

                        // reflect the responseCodec
                        Field responseCodec = null;
                        responseCodec = this.getClass().getSuperclass().getDeclaredField("responseCodec");
                        responseCodec.setAccessible(true);
                        responseCodec.set(this, new W3CHttpResponseCodec());
                    } catch (Exception e) {
                        LogUtil.logError(e, "Exception while custom execute command");
                    }

                } else {
                    response = super.execute(command);
                }
                return response;
            }
        };
    }

    public static RemoteWebDriver createDriverFromRunningSession(final String remoteServerUrl, final String sessionId,
            final String webSocketUrl) throws MalformedURLException {
        CommandExecutor executor = createCommandExecutorWithRunningSession(remoteServerUrl, sessionId, webSocketUrl);

        if (StringUtils.isBlank(webSocketUrl)) {
            return new RemoteWebDriver(executor, new MutableCapabilities());
        } else {
            return new CRemoteWebDriver(executor, new MutableCapabilities(), 0);
        }
    }

    public static HttpCommandExecutor getHttpCommandExecutorForRemoteDriver(URL remoteWebServerUrl,
            MutableCapabilities capabilities) throws URISyntaxException, IOException {
        ProxyInformation proxyInfo = RunConfiguration.getProxyInformation();

        URL cleanRemoteWebServerUrl = AppiumDriverUtil.removeUserInfoPart(remoteWebServerUrl);

        Factory clientFactory = ConfiguredHttpClientFactory.of(proxyInfo, cleanRemoteWebServerUrl, capabilities);

        ClientConfig clientConfig = ClientConfig.defaultConfig()
                .readTimeout(Duration.ofSeconds(SeleniumConstants.DEFAULT_READ_TIMEOUT_SECONDS))
                .baseUrl(requireNonNull(cleanRemoteWebServerUrl))
                .withFilter(new SeleniumProxyBasicAuthorizationFilter(proxyInfo.getUsername(), proxyInfo.getPassword())
                        .andThen(new SeleniumAcceptEncodingFilter())
                        .andThen(new SeleniumResponseContentEncodingDecompressionFilter())
                        .andThen(new SeleniumBasicAuthorizationFilter(remoteWebServerUrl))
                        .andThen(new SeleniumCreateSessionPayloadFilter(capabilities))
                        .andThen(new SeleniumCreateSessionResponseFilter()));

        return new HttpCommandExecutor(new HashMap<String, CommandInfo>(), clientConfig, clientFactory);
    }

    /**
     * Get browserName of the current WebDriver
     * <p>
     * Possible values: "Chrome", "Firefox", etc
     * 
     * @return the browser name of the WebDriver, or null if it is not
     */
    public static String getBrowserName(WebDriver driver) {
        if (driver instanceof RemoteWebDriver remoteDriver) {
            return remoteDriver.getCapabilities().getBrowserName();
        }

        return null;
    }
}
