package com.kms.katalon.core.logging.model;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FilenameUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.logging.model.TestStatus.TestStatusValue;
import com.microsoft.sqlserver.jdbc.StringUtils;

public class TestSuiteLogRecord extends AbstractLogRecord {
 
    private static final String TEST_CASE_REMAINING_RETRY_COUNT = "remainingRetryCount";

    private String devicePlatform;

    private String logFolder;

    private Map<String, String> runData;
    
    private String testSuiteCollectionId;

    private Map<String, String> buildData;
    
    private boolean isCompleted;

    public TestSuiteLogRecord(String name, String logFolder) {
        super(name);
        this.logFolder = logFolder;
        runData = new HashMap<String, String>();
        buildData = new HashMap<String, String>();
        setType(ILogRecord.LOG_TYPE_TEST_SUITE);
    }
    
    public void setStatus(boolean isCompleted) {
        this.isCompleted = isCompleted;
    }
    
    public boolean isCompleted() {
        return isCompleted;
    }

    public String getBrowser() {
        if (getRunData().containsKey("browser")) {
            if (getRunData().containsKey("remoteOS")) {
                return getRunData().get("remoteOS") + " - " + getRunData().get("browser");
            }
            return getRunData().get("browser");
        }
        return org.apache.commons.lang.StringUtils.EMPTY;
    }
    
    public String getTestCloudPlatform() {
        Map<String, Object> inputDesiredCaps = RunConfiguration
                .getDriverPreferencesProperties(RunConfiguration.REMOTE_DRIVER_PROPERTY);

        String space = " ";
        String hyphen = " - ";
        String platformName = (String) inputDesiredCaps.get("platformName");
        String platformVersion = (String) inputDesiredCaps.get("platformVersion");
        String deviceName = (String) inputDesiredCaps.get("deviceName");

        if (getRunData().containsKey("browser")) {
            String browser = getRunData().get("browser").split(" ")[0];
            return platformName + space + platformVersion + hyphen + deviceName + hyphen + browser;
        }
        return org.apache.commons.lang.StringUtils.EMPTY;
    }

    public String getLogFolder() {
        return logFolder;
    }

    public int getTotalTestCases() {
        return getTotalTestCasesWithTestStatusValue(null);
    }

    public int getTotalPassedTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.PASSED);
    }

    public int getTotalFailedTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.FAILED);
    }

    public int getTotalErrorTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.ERROR);
    }

    public int getTotalIncompleteTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.INCOMPLETE);
    }

    public int getTotalSkippedTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.SKIPPED);
    }

    public TestStatusValue getSummaryStatus() {
        if (getTotalIncompleteTestCases() > 0) {
            return TestStatusValue.INCOMPLETE;
        }
        
        if (getTotalErrorTestCases() > 0) {
            return TestStatusValue.ERROR;
        }
        
        if (getTotalFailedTestCases() > 0) {
            return TestStatusValue.FAILED;
        }
        
        if (getTotalSkippedTestCases() == getTotalTestCases()) {
            return TestStatusValue.SKIPPED;
        }
        
        return TestStatusValue.PASSED;
    }

    public ILogRecord[] filterFinalTestCasesResult() {
        ILogRecord[] childLogRecords = getChildRecords();
        List<ILogRecord> filterChildLogRecords = new ArrayList<>();
        for (int i = 0; i < childLogRecords.length; i++) {
            ILogRecord childLogRecord = childLogRecords[i];
            if (isFailedOrErrorTestCase(childLogRecord)) {
                TestCaseLogRecord testCaseLog = (TestCaseLogRecord) childLogRecord;
                Map<String, String> testCaseProperties = testCaseLog.getProperties();
                if (testCaseProperties != null && testCaseProperties.containsKey(TEST_CASE_REMAINING_RETRY_COUNT)) {
                    int remainingRetryCount = Integer.parseInt(testCaseProperties.get(TEST_CASE_REMAINING_RETRY_COUNT));
                    if (remainingRetryCount > 0) {
                        if (i == childLogRecords.length - 1) {
                            filterChildLogRecords.add(childLogRecord);
                        }
                        continue;
                    }
                }
            }
            filterChildLogRecords.add(childLogRecord);
        }
        return filterChildLogRecords.toArray(new ILogRecord[filterChildLogRecords.size()]);
    }

    private boolean isFailedOrErrorTestCase(ILogRecord logRecord) {
        if (!(logRecord instanceof TestCaseLogRecord)) {
            return false;
        }
        switch (logRecord.getStatus().getStatusValue()) {
            case FAILED:
            case ERROR:
                return true;
            default:
                return false;
        }
    }

    private int getTotalTestCasesWithTestStatusValue(TestStatusValue testStatusValue) {
        ILogRecord[] childLogRecords = filterFinalTestCasesResult();
        int total = 0;
        for (ILogRecord childLogRecord : childLogRecords) {
            if (childLogRecord instanceof TestCaseLogRecord) {
                TestCaseLogRecord testCaseLog = (TestCaseLogRecord) childLogRecord;
                if (testStatusValue == null || testCaseLog.getStatus().statusValue == testStatusValue) {
                    total++;
                }
            }
        }
        return total;
    }

    public String getDeviceName() {
        return (getRunData().containsKey(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY))
                ? getRunData().get(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY)
                : getValueFromDesiredCaps("deviceName");
    }

    public String getDeviceId() {
        return (getRunData().containsKey(StringConstants.XML_LOG_DEVICE_ID_PROPERTY))
                ? getRunData().get(StringConstants.XML_LOG_DEVICE_ID_PROPERTY) : "";
    }

    public String getDevicePlatform() {
		if (org.apache.commons.lang3.StringUtils.isBlank(devicePlatform)) {
			String platform = getValueFromDesiredCaps("platform");
			return org.apache.commons.lang3.StringUtils.defaultIfEmpty(platform, devicePlatform);
		}
        return devicePlatform;
    }

    @SuppressWarnings("unchecked")
    private String getValueFromDesiredCaps(String key) {
        if (getRunData().containsKey("desiredCapabilities")) {
            String json = getRunData().get("desiredCapabilities");
            try {
                Map<String, String> desiredCaps = new ObjectMapper().readValue(json, HashMap.class);
                return desiredCaps.get(key);
            } catch (JsonProcessingException e) {
                return org.apache.commons.lang3.StringUtils.EMPTY;
            }
        }
        return org.apache.commons.lang3.StringUtils.EMPTY;
    }

    public void setDevicePlatform(String devicePlatform) {
        this.devicePlatform = devicePlatform;
    }

    public String getOs() {
        return (getRunData().containsKey(RunConfiguration.HOST_OS)) ? getRunData().get(RunConfiguration.HOST_OS) : "";
    }

    public String getHostName() {
        return (getRunData().containsKey(RunConfiguration.HOST_NAME)) ? getRunData().get(RunConfiguration.HOST_NAME)
                : "";
    }

    public String getAppVersion() {
        return (getRunData().containsKey(RunConfiguration.APP_VERSION)) ? getRunData().get(RunConfiguration.APP_VERSION)
                : "";
    }

    public String getQtestBuildNumber() {
        return getBuildData().containsKey("qTestBuildNumber") ? getBuildData().get("qTestBuildNumber") : "";
    }

    public String getQtestBuildURL() {
        return getBuildData().containsKey("qTestBuildURL") ? getBuildData().get("qTestBuildURL") : "";
    }

    public String getAzurebBuildDefinitionId() {
        if (getBuildData().containsKey("adoDefinitionId")) {
            return getBuildData().get("adoDefinitionId");
        }
        if (getBuildData().containsKey("adoDefinitionID")) {
            return getBuildData().get("adoDefinitionID");
        }
        if (getBuildData().containsKey("adoBuildDefId")) {
            return getBuildData().get("adoBuildDefId");
        }
        if (getBuildData().containsKey("adoBuildDefID")) {
            return getBuildData().get("adoBuildDefID");
        }
        return StringUtils.EMPTY;
    }

    public String getAzurebReleaseDefinitionId() {
        if (getBuildData().containsKey("adoReleaseDefId")) {
            return getBuildData().get("adoReleaseDefId");
        }
        if (getBuildData().containsKey("adoReleaseDefID")) {
            return getBuildData().get("adoReleaseDefID");
        }
        return StringUtils.EMPTY;
    }

    public Map<String, String> getRunData() {
        return runData;
    }

    public void addRunData(Map<String, String> runData) {
        this.runData.putAll(runData);
    }

    public Map<String, String> getBuildData() {
        return buildData;
    }

    public void addBuildData(Map<String, String> buildData) {
        this.buildData.putAll(buildData);
    }

    public <T extends ILogRecord> int getChildIndex(T child) {
        return Arrays.asList(getChildRecords()).indexOf(child);
    }

    public List<String> getLogFiles() {
        List<String> logFiles = new ArrayList<String>();
        for (String childFile : new File(getLogFolder()).list()) {
            if (!FilenameUtils.getExtension(childFile).equals("log")) {
                continue;
            }
            logFiles.add(childFile);
        }
        return logFiles;
    }

    @Override
    public String getSystemOutMsg() {
        return getJUnitMessage();
    }

    @Override
    public String getSystemErrorMsg() {
        TestStatus status = getStatus();
        String stackTrace = status.getStackTrace();
        if (status.getStatusValue().isError()) {
            return getJUnitMessage() + stackTrace;
        }
        return stackTrace;
    }

    public String getTestSuiteCollectionId() {
        return testSuiteCollectionId;
    }

    public void setTestSuiteCollectionId(String testSuiteCollectionId) {
        this.testSuiteCollectionId = testSuiteCollectionId;
    }
    
    public TestCaseLogRecord getLastTestCaseLogRecord() {
        return getChildRecords().length > 0 
                ? (TestCaseLogRecord) getChildRecords()[getChildRecords().length - 1]
                : null;
    }
}
