package com.kms.katalon.core.util.download;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.URISyntaxException;
import java.net.URLConnection;
import java.security.GeneralSecurityException;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.eclipse.core.runtime.IProgressMonitor;

import com.kms.katalon.core.network.ProxyOption;
import com.kms.katalon.core.preferences.ProxyPreferences;
import com.kms.katalon.core.util.NetworkUtil;
import com.kms.katalon.core.util.internal.ProxyUtil;

public class FileDownloader {

	public static final long UNKNOWN_SIZE = -1L;

	public static final long NOTIFICATION_DELAY = 500L;

	public static final ProxyOption DEFAULT_PROXY_OPTION = ProxyOption.NO_PROXY;

	private final long fileSizeInBytes;

	private List<DownloadProgressListener> listeners = new LinkedList<>();

	private Boolean notificationSending = false;

	public FileDownloader(long fileSizeInBytes) {
		this.fileSizeInBytes = fileSizeInBytes;
	}

	public void addListener(DownloadProgressListener l) {
		listeners.add(l);
	}

	private void notifyProgressChanges(long progress, long total, long speedInKBPs) {
		synchronized (notificationSending) {
			if (notificationSending) {
				return;
			}
		}
		new Thread(() -> {
			notificationSending = true;
			long startTime = System.currentTimeMillis();
			while (System.currentTimeMillis() - startTime <= NOTIFICATION_DELAY) {
				try {
					Thread.sleep(50L);
				} catch (InterruptedException ignored) {
				}
			}

			listeners.stream().forEach(l -> l.onProgressUpdate(progress, total, speedInKBPs));
			notificationSending = false;
		}).start();
	}

	public void download(String url, OutputStream os, IProgressMonitor monitor)
			throws FileDownloaderException, InterruptedException {
		InputStream inputStream = null;
		URLConnection connection = null;
		try {
			connection = createConnection(url);
			inputStream = connection.getInputStream();

			byte data[] = new byte[1024 * 4];

			int count;
			long progress = 0L;
			long startDownloadTime = System.currentTimeMillis();

			while ((count = inputStream.read(data)) != -1) {
				if (monitor != null && monitor.isCanceled()) {
					throw new InterruptedException();
				}
				progress += count;

				os.write(data, 0, count);

				long speedInKBPs = progress * 1000L / Math.max(System.currentTimeMillis() - startDownloadTime, 1L);
				notifyProgressChanges(progress, fileSizeInBytes, speedInKBPs);
			}
		} catch (IOException | GeneralSecurityException | URISyntaxException e) {
			throw new FileDownloaderException(e);
		} finally {
			try {
				if (inputStream != null) {
					inputStream.close();
				}
			} catch (IOException ignored) {
			}

			IOUtils.close(connection);
		}
	}

	private HttpURLConnection createConnection(String url)
			throws IOException, GeneralSecurityException, URISyntaxException {
		try {
			return NetworkUtil.createURLConnection(url, ProxyUtil.getProxy(ProxyPreferences.getAuthProxyInformation()));
		} catch (ConnectException e) {
			return NetworkUtil.createURLConnection(url, ProxyUtil.getSystemProxy());
		}
	}
}
