package com.kms.katalon.core.util.internal;

import java.io.Reader;
import java.lang.reflect.Type;
import java.text.MessageFormat;
import java.util.List;
import java.util.Map;

import org.assertj.core.util.Arrays;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import com.kms.katalon.core.constants.StringConstants;

public class JsonUtil {

    /**
     * Convert an object to JSON string. Pretty print by default.
     * 
     * @param src source object
     * @return JSON string
     */
    public static String toJson(Object src) {
        return toJson(src, true);
    }

    /**
     * Convert an object to JSON string.
     * 
     * @param src source object
     * @param prettyPrint pretty print
     * @return JSON string
     */
    public static String toJson(Object src, boolean prettyPrint) {
        return toJson(src, prettyPrint, false);
    }

    /**
     * Convert an object to JSON string.
     * 
     * @param src source object
     * @param prettyPrint pretty print
     * @param excludeFieldsWithoutExposeAnnotation exclude the fields without expose annotation
     * @return JSON string
     */
    public static String toJson(Object src, boolean prettyPrint, boolean excludeFieldsWithoutExposeAnnotation) {
        GsonBuilder gsonBuilder = new GsonBuilder();
        Gson gson;
        if (prettyPrint || excludeFieldsWithoutExposeAnnotation) {
            if (prettyPrint) {
                gsonBuilder = gsonBuilder.setPrettyPrinting();
            }
            if (excludeFieldsWithoutExposeAnnotation) {
                gsonBuilder = gsonBuilder.excludeFieldsWithoutExposeAnnotation();
            }
            gson = gsonBuilder.create();
        } else {
            gson = new Gson();
        }
        return gson.toJson(src);
    }

    /**
     * Convert an object to JSON string. Pretty print by default.
     * 
     * @param src source object
     * @param typeOfSrc type of source
     * @return JSON string
     */
    public static String toJson(Object src, Type typeOfSrc) {
        return toJson(src, typeOfSrc, true);
    }

    /**
     * Convert an object to JSON string.
     * 
     * @param src source object
     * @param typeOfSrc type of source
     * @param prettyPrint pretty print
     * @return JSON string
     */
    public static String toJson(Object src, Type typeOfSrc, boolean prettyPrint) {
        Gson gson = prettyPrint ? new GsonBuilder().setPrettyPrinting().create() : new Gson();
        return gson.toJson(src, typeOfSrc);
    }

    @SuppressWarnings("unchecked")
    public static <T> List<T> fromJsonArray(String json, Class<T[]> arrayType) {
        Object[] items = (Object[]) fromJson(json, arrayType);
        return (List<T>) Arrays.asList(items);
    }

    /**
     * Convert JSON string to Object
     * 
     * @param json JSON string
     * @param typeOfT type of target object
     * @return target object
     * @throws IllegalArgumentException if the input is invalid JSON syntax
     */
    public static <T> T fromJson(String json, Type typeOfT) throws IllegalArgumentException {
        try {
            return new Gson().fromJson(json, typeOfT);
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    public static <T> T fromJson(Reader reader, Type typeOfT) throws IllegalArgumentException {
        try {
            return new Gson().fromJson(reader, typeOfT);
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    public static <T> T fromJson(Reader reader, Class<T> classOfT) throws IllegalArgumentException {
        try {
            return new Gson().fromJson(reader, classOfT);
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    /**
     * Convert JSON string to Object
     * 
     * @param json JSON string
     * @param classOfT class of target object
     * @return target object
     * @throws IllegalArgumentException if the input is invalid JSON syntax
     */
    public static <T> T fromJson(String json, Class<T> classOfT) throws IllegalArgumentException {
        try {
            return (T) new Gson().fromJson(json, classOfT);
        } catch (JsonSyntaxException e) {
            throw new IllegalArgumentException(
                    MessageFormat.format(StringConstants.EXC_MSG_INVALID_JSON_SYNTAX, e.getMessage()));
        }
    }

    public static JsonObject toJsonObject(Object src) {
        Gson gson = new Gson();
        return gson.toJsonTree(src).getAsJsonObject();
    }

    public static void mergeJsonObject(JsonObject src, JsonObject dst) {
        for (Map.Entry<String, JsonElement> entry : src.entrySet()) {
            dst.add(entry.getKey(), entry.getValue());
        }
    }

    public static boolean isValidJsonObject(String json) {
        try {
            Object parsedObject = new Gson().fromJson(json, Object.class);
            if (parsedObject == null) {
                return false;
            }
            Class<?> type = parsedObject.getClass();
            return Map.class.isAssignableFrom(type);
        } catch (JsonSyntaxException e) {
            return false;
        }
    }
}
