package com.kms.katalon.core.mobile.helper;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.text.MessageFormat;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.WebElement;

import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.mobile.common.MobileXPathBuilder;
import com.kms.katalon.core.mobile.constants.StringConstants;
import com.kms.katalon.core.mobile.driver.AppiumDriverSession;
import com.kms.katalon.core.mobile.driver.AppiumSessionCollector;
import com.kms.katalon.core.mobile.keyword.internal.AndroidProperties;
import com.kms.katalon.core.mobile.keyword.internal.GUIObject;
import com.kms.katalon.core.mobile.keyword.internal.MobileDriverFactory;
import com.kms.katalon.core.testobject.TestObject;
import com.kms.katalon.core.util.ConsoleCommandBuilder;
import com.kms.katalon.core.util.internal.ProcessUtil;

import io.appium.java_client.AppiumDriver;
import io.appium.java_client.FindsByIosClassChain;
import io.appium.java_client.TouchAction;
import io.appium.java_client.android.AndroidDriver;
import io.appium.java_client.ios.IOSDriver;
import io.appium.java_client.touch.WaitOptions;
import io.appium.java_client.touch.offset.PointOption;

public class MobileCommonHelper {

    private static final KeywordLogger logger = KeywordLogger.getInstance(MobileCommonHelper.class);

    private static final String ATTRIBUTE_NAME_FOR_ANDROID_RESOURCE_ID = "resourceId";

    private static final String ATTRIBUTE_NAME_FOR_ANDROID_CONTENT_DESC = "name";

    public static final String PROPERTY_NAME_DEVICE_PIXEL_RATIO = "devicePixelRatio";

    public static final String PROPERTY_NAME_OS_STATUS_BAR_HEIGHT = "osStatusBarHeight";

    public static final String PROPERTY_NAME_STATUS_BAR_HEIGHT = "statusBarHeight";

    public static final String PROPERTY_NAME_SCALE_FACTOR = "scaleFactor";

    public static final String PROPERTY_NAME_IOS_BUNDLEID = "iosBundleId";

    @SuppressWarnings("rawtypes")
    public static void swipe(AppiumDriver driver, int startX, int startY, int endX, int endY) {
        TouchAction swipe = new TouchAction(driver).press(PointOption.point(startX, startY))
                .waitAction(WaitOptions.waitOptions(Duration.ofMillis(500L)))
                .moveTo(PointOption.point(endX, endY))
                .waitAction(WaitOptions.waitOptions(Duration.ofMillis(500L)))
                .release();
        swipe.perform();
    }

    public static Map<String, String> deviceModels = new HashMap<String, String>();
    static {
        deviceModels.put("iPhone3,1", "iPhone 4");
        deviceModels.put("iPhone3,3", "iPhone 4");
        deviceModels.put("iPhone4,1", "iPhone 4S");

        deviceModels.put("iPhone5,1", "iPhone 5");
        deviceModels.put("iPhone5,2", "iPhone 5");
        deviceModels.put("iPhone5,3", "iPhone 5c");
        deviceModels.put("iPhone5,4", "iPhone 5c");
        deviceModels.put("iPhone6,1", "iPhone 5s");
        deviceModels.put("iPhone6,2", "iPhone 5s");
        deviceModels.put("iPhone7,1", "iPhone 6 Plus");
        deviceModels.put("iPhone7,2", "iPhone 6");
        deviceModels.put("iPad1,1", "iPad");
        deviceModels.put("iPad2,1", "iPad 2");
        deviceModels.put("iPad2,2", "iPad 2");
        deviceModels.put("iPad2,3", "iPad 2");
        deviceModels.put("iPad2,4", "iPad 2");
        deviceModels.put("iPad2,5", "iPad mini");
        deviceModels.put("iPad2,6", "iPad mini");
        deviceModels.put("iPad2,7", "iPad mini");

        deviceModels.put("iPad3,1", "iPad 3");
        deviceModels.put("iPad3,2", "iPad 3");
        deviceModels.put("iPad3,3", "iPad 3");
        deviceModels.put("iPad3,4", "iPad 4");
        deviceModels.put("iPad3,5", "iPad 4");
        deviceModels.put("iPad3,6", "iPad 4");
        deviceModels.put("iPad4,1", "iPad Air");
        deviceModels.put("iPad4,2", "iPad Air");
        deviceModels.put("iPad4,3", "iPad Air");
        deviceModels.put("iPad4,4", "iPad mini 2");
        deviceModels.put("iPad4,5", "iPad mini 2");
        deviceModels.put("iPad4,6", "iPad mini 2");
        deviceModels.put("iPad4,7", "iPad mini 3");
        deviceModels.put("iPad4,8", "iPad mini 3");
        deviceModels.put("iPad4,9", "iPad mini 3");
        deviceModels.put("iPad5,3", "iPad Air 2");
        deviceModels.put("iPad5,4", "iPad Air 2");

    }

    public static Map<String, String> airPlaneButtonCoords = new HashMap<String, String>();
    static {
        airPlaneButtonCoords.put("iPhone 5s", "40;195");
        airPlaneButtonCoords.put("iPhone 5", "40;195");

        airPlaneButtonCoords.put("iPad 2", "260;905");
        airPlaneButtonCoords.put("iPad 3", "260;905");
        airPlaneButtonCoords.put("iPad 4", "260;905");

        airPlaneButtonCoords.put("iPad Air", "260;905");
        airPlaneButtonCoords.put("iPad Air 2", "260;905");

        airPlaneButtonCoords.put("iPhone 6", "50;290");
        airPlaneButtonCoords.put("iPhone 6 Plus", "59;359");

        airPlaneButtonCoords.put("iPad mini", "265;905");
        airPlaneButtonCoords.put("iPad mini 2", "265;905");
        airPlaneButtonCoords.put("iPad mini 3", "265;905");
    }

    public static String getAttributeLocatorValue(TestObject testObject) {
        if (testObject == null || testObject.getProperties().isEmpty()) {
            return null;
        }
        MobileXPathBuilder xpathBuilder = new MobileXPathBuilder(testObject.getActiveProperties());
        return xpathBuilder.build();
    }

    public static String getAttributeValue(WebElement element, String attributeName) {
        switch (attributeName.toString()) {
            case GUIObject.HEIGHT:
                return String.valueOf(element.getSize().height);
            case GUIObject.WIDTH:
                return String.valueOf(element.getSize().width);
            case GUIObject.X:
                return String.valueOf(element.getLocation().x);
            case GUIObject.Y:
                return String.valueOf(element.getLocation().y);
            case AndroidProperties.ANDROID_RESOURCE_ID:
                if (MobileDriverFactory.getDriver() instanceof AndroidDriver) {
                    return element.getAttribute(ATTRIBUTE_NAME_FOR_ANDROID_RESOURCE_ID);
                }
            case AndroidProperties.ANDROID_CONTENT_DESC:
                if (MobileDriverFactory.getDriver() instanceof AndroidDriver) {
                    return element.getAttribute(ATTRIBUTE_NAME_FOR_ANDROID_CONTENT_DESC);
                }
            default:
                try {
                    return element.getAttribute(attributeName);
                } catch (NoSuchElementException e) {
                    // attribute not found, return null
                    return null;
                }
        }
    }

    public static void checkXAndY(Number x, Number y) {
        logger.logDebug(StringConstants.COMM_LOG_INFO_CHECKING_X);
        if (x == null) {
            throw new StepFailedException(
                    MessageFormat.format(StringConstants.KW_MSG_FAILED_PARAM_X_CANNOT_BE_NULL, "x"));
        }
        logger.logDebug(StringConstants.COMM_LOG_INFO_CHECKING_Y);
        if (y == null) {
            throw new StepFailedException(
                    MessageFormat.format(StringConstants.KW_MSG_FAILED_PARAM_X_CANNOT_BE_NULL, "y"));
        }
    }

    public static void setCommonAppiumSessionProperties(AppiumDriver<? extends WebElement> driver) {
        AppiumDriverSession session = AppiumSessionCollector.getSession(driver);
        session.getProperties().put(PROPERTY_NAME_STATUS_BAR_HEIGHT, getStatusBarHeight(driver));
        session.getProperties().put(PROPERTY_NAME_SCALE_FACTOR, getScaleFactor(driver));
    }

    public static int getStatusBarHeight(AppiumDriver<? extends WebElement> driver) {
        int statusBar = 0;
        if (driver instanceof AndroidDriver) {
            statusBar = AndroidHelper.getStatusBarHeight(driver);
        }
        if (driver instanceof IOSDriver) {
            statusBar = IOSHelper.getStatusBarHeight(driver);
        }
        return statusBar;
    }

    public static float getScaleFactor(AppiumDriver<? extends WebElement> driver) {
        float scaleFactor = 1;
        if (driver instanceof IOSDriver) {
            scaleFactor = IOSHelper.getScaleFactor(driver);
        }
        return scaleFactor;
    }

    @SuppressWarnings("rawtypes")
    public static WebElement findElementByIosClassChain(IOSDriver iosDriver, String type, String name, String label) {
        FindsByIosClassChain driver = iosDriver;
        String locator = "**/%s[`name == '%s' OR label = '%s'`]";
        return driver.findElementByIosClassChain(String.format(locator, type, name, label));
    }

    @SuppressWarnings("rawtypes")
    public static WebElement findElementByIosClassChain(IOSDriver iosDriver, String type, String name) {
        FindsByIosClassChain driver = iosDriver;
        String locator = "**/%s[`name == '%s'`]";
        return driver.findElementByIosClassChain(String.format(locator, type, name));
    }

    public static int getMajorVersion(String version) {
        return Integer.parseInt(version.split("\\.")[0]);
    }

    public static boolean isIPhoneXOrLater(String deviceModel) {
        if (!deviceModel.contains("iPhone")) {
            return false;
        }
        String[] versionNumbers = deviceModel.replace("iPhone", "").split(",");
        int majorVersion = Integer.parseInt(versionNumbers[0]);
        int minorVersion = Integer.parseInt(versionNumbers[1]);

        // https://www.theiphonewiki.com/wiki/Models
        return (majorVersion >= 11 || (majorVersion == 10 && (minorVersion == 3 || minorVersion == 6)));
    }

    public static boolean isIPad(String deviceModel) {
        return deviceModel.contains("iPad");
    }

    public static long getAndroidPackageSize(String deviceId, String appId, String adbFolder) {
        try {
            String command = MessageFormat.format(
                    "adb -s {0} shell \"stat -c %s $(pm path {1} | grep base.apk | cut -d : -f 2)\"", deviceId, appId);
            List<String> packageSize = ConsoleCommandBuilder.create(command).path(adbFolder).execSync();
            return Long.valueOf(packageSize.get(0));
        } catch (Exception error) {
            return -1;
        }
    }

    public static String getBundleId(String packagePath, String aaptFolder) {
        return grepBundleInfo(packagePath, aaptFolder, "package: name='(.*?)'");
    }

    public static String getBundleVersion(String packagePath, String aaptFolder) {
        return grepBundleInfo(packagePath, aaptFolder, "versionName='(.*?)'");
    }

    public static String grepBundleInfo(String packagePath, String aaptFolder, String infoPattern) {
        try {
            String command = MessageFormat.format("aapt dump badging \"{0}\"", packagePath);
            List<String> infoLines = ConsoleCommandBuilder.create(command).path(aaptFolder).execSync();
            Pattern idPattern = Pattern.compile(infoPattern);
            for (String lineI : infoLines) {
                Matcher idMatcher = idPattern.matcher(lineI);
                if (idMatcher.find()) {
                    return idMatcher.group(1);
                }
            }
            return null;
        } catch (IOException | InterruptedException error) {
            return null;
        }
    }

    public static String getInstalledAppVersion(String deviceId, String appId, String adbFolder) {
        return grepInstalledAppInfo(deviceId, appId, adbFolder, "versionName=(.*?)(\\s|$)");
    }

    public static String grepInstalledAppInfo(String deviceId, String appId, String adbFolder, String infoPattern) {
        try {
            String command = MessageFormat.format("adb -s {0} shell dumpsys package \"{1}\"", deviceId, appId);
            List<String> infoLines = ConsoleCommandBuilder.create(command).path(adbFolder).execSync();
            Pattern idPattern = Pattern.compile(infoPattern);
            for (String lineI : infoLines) {
                Matcher idMatcher = idPattern.matcher(lineI);
                if (idMatcher.find()) {
                    return idMatcher.group(1);
                }
            }
            return null;
        } catch (IOException | InterruptedException error) {
            return null;
        }
    }

    public static boolean isSameSize(String packagePath, String deviceId, String appId, String adbFolder) {
        long installedPackageSize = getAndroidPackageSize(deviceId, appId, adbFolder);
        try {
            return Files.size(new File(packagePath).toPath()) == installedPackageSize;
        } catch (IOException error) {
            return false;
        }
    }

    public static boolean isSameVersion(String packagePath, String deviceId, String appId, String adbFolder,
            String aaptFolder) {
        String bundleVersion = getBundleVersion(packagePath, aaptFolder);
        String installedAppVersion = getInstalledAppVersion(deviceId, appId, adbFolder);
        return StringUtils.equals(bundleVersion, installedAppVersion);
    }

    public static boolean isSameApp(String packagePath, String deviceId, String appId, String adbFolder,
            String aaptFolder) {
        return isSameVersion(packagePath, deviceId, appId, adbFolder, aaptFolder)
                && isSameSize(packagePath, deviceId, appId, adbFolder);
    }

    public static boolean inAndroidAppInstalled(String deviceId, String appId, String adbFolder)
            throws IOException, InterruptedException {
        List<String> result = ConsoleCommandBuilder
                .create(MessageFormat.format("adb -s \"{0}\" shell pm path \"{1}\"", deviceId, appId))
                .path(adbFolder)
                .execSync();
        return ProcessUtil.includes(result, appId);
    }

    public static void uninstallAndroidApp(String deviceId, String appId, String adbFolder)
            throws IOException, InterruptedException {
        ConsoleCommandBuilder.create(MessageFormat.format("adb -s \"{0}\" uninstall \"{1}\"", deviceId, appId))
                .path(adbFolder)
                .execSync();
    }
}
