package com.kms.katalon.core.util;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;

import com.kms.katalon.core.configuration.RunConfiguration;

public class FileUtil {
    private static final String ANY_EXTENSION = "*";

    public static File[] getFiles(File folder, String suffix, Date origin) throws IOException {
        final List<File> filePaths = new ArrayList<>();
        enumFiles(folder, suffix, origin, new FileProcessor() {

            @Override
            public void process(File f) {
                filePaths.add(f);
            }
        });
        return filePaths.toArray(new File[] {});
    }

    public static void enumFiles(File folder, String suffix, Date origin, FileProcessor fileProcessor) throws IOException {
        if (folder.isDirectory() == false) {
            return;
        }
        List<File> files = new ArrayList<>();
        files.addAll(Arrays.asList(folder.listFiles()));
        for (int i = 0; i < files.size(); ++i) {
            File file = files.get(i);
            if (file.isDirectory()) {
                files.addAll(Arrays.asList(file.listFiles()));
                continue;
            }
            if (isFileSatisfy(file, suffix, origin)) {
                fileProcessor.process(file);
            }
        }
    }

    private static boolean isFileSatisfy(File file, String suffix, Date origin) throws IOException {
        String fileName = file.getName();
        boolean satisfyName = suffix.equals(ANY_EXTENSION) ? true : fileName.endsWith(suffix);
        if (!satisfyName) {
            return false;
        }

        return isFileCreateAfter(file, origin);
    }
    
    public static boolean isFileCreateAfter(File file, Date time) throws IOException {
        if (time == null) {
            return true;
        } else {
            return time.getTime() < Files.readAttributes(file.toPath(), BasicFileAttributes.class)
                    .creationTime()
                    .toMillis();
        }
    }

    public static int countAllFiles(File folder, String suffix, Date origin) throws IOException {
        final int[] count = { 0 };
        enumFiles(folder, suffix, origin, new FileProcessor() {

            @Override
            public void process(File f) {
                count[0]++;
            }
        });

        return count[0];
    }

    public static boolean isValidPath(String path) {
        return path != null && new File(path).exists();
    }

    public static boolean fileExists(String root, String pattern) {
        return findFileByName(root, pattern) != null;
    }

    public static File findFileByName(String root, String pattern) {
        return findFile(root, file -> {
            return file.getName().matches(pattern);
        });
    }

    public static File findFile(String root, FileFilter filter) {
        return find(root, filter, FileType.FILE);
    }

    public static List<File> findAllFiles(String root, FileFilter filter) {
        return findAll(root, filter, FileType.FILE);
    }

    public static File findFolderByName(String root, String pattern) {
        return findFolder(root, file -> {
            return file.getName().matches(pattern);
        });
    }

    public static File findFolder(String root, FileFilter filter) {
        return find(root, filter, FileType.FOLDER);
    }

    public static List<File> findAllFolders(String root, FileFilter filter) {
        return findAll(root, filter, FileType.FOLDER);
    }

    public static File find(String root, FileFilter filter) {
        return find(root, filter, FileType.ANY);
    }

    public static File find(String root, FileFilter filter, FileType type) {
        List<File> matches = findAll(root, filter, type);
        return matches.isEmpty() ? null : matches.get(0);
    }

    public static List<File> findAll(String root, FileFilter filter) {
        return findAll(root, filter, FileType.ANY);
    }

    public static List<File> findAll(String root, FileFilter filter, FileType type) {
        if (StringUtils.isBlank(root) || !new File(root).exists()) {
            return Collections.emptyList();
        }
        File rootFile = new File(root);
        String[] directories = rootFile.list(new FilenameFilter() {
            @Override
            public boolean accept(File parentFolder, String name) {
                File file = new File(parentFolder, name);
                boolean typeCheck = type == FileType.ANY || (type == FileType.FILE && !file.isDirectory())
                        || (type == FileType.FOLDER && file.isDirectory());
                return filter.test(file) && typeCheck;
            }
        });
        return Arrays.asList(directories).stream().map(subDirectory -> new File(root, subDirectory)).collect(Collectors.toList());
    }

    public static interface FileProcessor {
        public void process(File f);
    }

    public static interface FileFilter {
        public boolean test(File file);
    }

    public enum FileType {
        FILE, FOLDER, ANY
    }
    
    public static File getFile(String path) {
        File file = new File(path);
        if (file.isAbsolute()) {
            return file;
        }
        if (StringUtils.isNotBlank(RunConfiguration.getProjectDir())) {
            return new File(RunConfiguration.getProjectDir(), path);
        }
        return null;
    }
}
