package com.kms.katalon.core.util;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;

import io.netty.util.internal.StringUtil;

public class ObjectUtil {
    public static boolean equals(Object a, Object b) {
        if (a == null && b == null) {
            return true;
        }
        return a != null ? a.equals(b) : b.equals(a);
    }

    public static <T> T invokeGet(Object object, String getterMethodName) {
        return invokeGet(object, getterMethodName, null);
    }

    @SuppressWarnings("unchecked")
    public static <T> T invokeGet(Object object, String getterMethodName, T defaultValue) {
        Method getter = findMethod(object, getterMethodName);
        if (getter != null) {
            return valueOrDefault((T) safeInvoke(object, getter), defaultValue);
        }
        return defaultValue;
    }

    @SuppressWarnings("unchecked")
    public static <T> void invokeSet(Object object, String method, T data) {
        if (data != null) {
            Method setter1 = findMethod(object, method, data.getClass());
            if (setter1 != null) {
                safeInvoke(object, setter1, data);
                return;
            }

            Method setter2;
            try {
                setter2 = findMethod(object, method, (Class<T>) data.getClass().getField("TYPE").get(null));
                if (setter2 != null) {
                    safeInvoke(object, setter2, data);
                    return;
                }
            } catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException | SecurityException e) {
                // Just skip
            }
        }
        try {
            Method setter3 = findMethod(object, method,
                    (Class<T>) ((ParameterizedType) data.getClass().getGenericSuperclass()).getActualTypeArguments()[0]);
            if (setter3 != null) {
                safeInvoke(object, setter3, data);
                return;
            }
        } catch (Exception error) {
            // Just skip
        }
    }

    public static <T> void invokeSet(Object object, String method, T data, Class<T> type) {
        Method setter = findMethod(object, method, type);
        if (setter != null) {
            safeInvoke(object, setter, data);
        }
    }

    @SafeVarargs
    public static <T> boolean hasMethod(Object object, String method, Class<T>... args) {
        return findMethod(object, method, args) != null;
    }

    @SafeVarargs
    public static <T> Method findMethod(Object object, String methodName, Class<T>... args) {
        if (object == null) {
            return null;
        }
        return findMethod(object, object.getClass(), methodName, args);
    }

    @SafeVarargs
    public static <T> Method findMethod(Object object, Class<?> clazz, String methodName, Class<T>... args) {
        if (object == null || clazz == null) {
            return null;
        }
        try {
            Method foundPublicMethod = args.length == 1 && args[0] == void.class ? clazz.getMethod(methodName)
                    : clazz.getMethod(methodName, args);
            if (foundPublicMethod != null) {
                return foundPublicMethod;
            }
        } catch (NoSuchMethodException | SecurityException | IllegalArgumentException error) {
            try {
                Method foundPrivateMethod = clazz.getDeclaredMethod(methodName, args);
                if (foundPrivateMethod != null) {
                    return foundPrivateMethod;
                }
            } catch (NoSuchMethodException | SecurityException error1) {
                // Just skip
            }
        }
        return findMethod(object, clazz.getSuperclass(), methodName, args);
    }

    public static Method findMethod(Object object, String methodName) {
        return findMethod(object, object.getClass(), methodName);
    }

    public static Method findMethod(Object object, Class<?> clazz, String methodName) {
        if (object == null || clazz == null) {
            return null;
        }
        try {
            Method foundPublicMethod = clazz.getMethod(methodName);
            if (foundPublicMethod != null) {
                return foundPublicMethod;
            }
        } catch (NoSuchMethodException | SecurityException | IllegalArgumentException error) {
            try {
                Method foundPrivateMethod = clazz.getDeclaredMethod(methodName);
                if (foundPrivateMethod != null) {
                    return foundPrivateMethod;
                }
            } catch (NoSuchMethodException | SecurityException error1) {
                // Just skip
            }
        }
        return findMethod(object, clazz.getSuperclass(), methodName);
    }

    public static <T> Object safeInvoke(Object object, String methodName) {
        return safeInvoke(object, methodName, new Object[] {});
    }

    public static <T> Object safeInvoke(Object object, String methodName, Object... args) {
        Method method = findMethod(object, methodName);
        return safeInvoke(object, method, args);
    }

    public static <T> Object safeInvoke(Object object, Method method) {
        return safeInvoke(object, method, new Object[] {});
    }

    public static <T> Object safeInvoke(Object object, Method method, Object... args) {
        if (object == null || method == null) {
            return null;
        }
        try {
            if (!method.isAccessible()) {
                method.setAccessible(true);
            }
            return method.invoke(object, args);
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException error) {
            return null;
        }
    }

    public static String getStringField(Object object, String fieldName) {
        Object fieldValue = getField(object, fieldName, null);
        return fieldValue instanceof String ? (String) fieldValue : StringUtil.EMPTY_STRING;
    }

    public static <T> T getField(Object object, String fieldName) {
        return getField(object, fieldName, null);
    }

    @SuppressWarnings("unchecked")
    public static <T> T getField(Object object, String fieldName, T defaultValue) {
        if (object == null) {
            return defaultValue;
        }
        try {
            Field field = findField(object, fieldName);
            if (field == null) {
                return defaultValue;
            }
            if (!field.isAccessible()) {
                field.setAccessible(true);
            }
            return valueOrDefault((T) field.get(object), defaultValue);
        } catch (SecurityException | IllegalAccessException | IllegalArgumentException exception) {
            // Just skip
        }
        return defaultValue;
    }

    public static <T> void setField(Object object, String fieldName, T value) {
        if (object == null) {
            return;
        }
        try {
            Field field = findField(object, fieldName);
            if (field == null) {
                return;
            }
            if (!field.isAccessible()) {
                field.setAccessible(true);
            }
            field.set(object, value);
        } catch (SecurityException | IllegalAccessException | IllegalArgumentException exception) {
            // Just skip
        }
    }

    public static <T> boolean hasField(Object object, String fieldName) {
        if (object == null) {
            return false;
        }
        try {
            Field field = findField(object, fieldName);
            return field != null;
        } catch (SecurityException | IllegalArgumentException exception) {
            return false;
        }
    }

    public static Field findField(Object object, String fieldName) {
        if (object == null) {
            return null;
        }
        return findField(object.getClass(), fieldName);
    }

    public static Field findField(Class<?> clazz, String fieldName) {
        if (clazz == null) {
            return null;
        }
        try {
            Field field = clazz.getDeclaredField(fieldName);
            return field != null ? field : findField(clazz.getSuperclass(), fieldName);
        } catch (SecurityException | IllegalArgumentException | NoSuchFieldException exception) {
            return findField(clazz.getSuperclass(), fieldName);
        }
    }

    @SuppressWarnings("unchecked")
    public static <T> T clone(T source) {
        try {
            return (T) clone(source, source.getClass().newInstance());
        } catch (InstantiationException | IllegalAccessException error) {
            error.printStackTrace();
        }
        return null;
    }

    public static <T1, T2> T2 clone(T1 source, T2 destination) {
        return clone(source, source.getClass(), destination);
    }

    public static <T1, T2> T2 clone(T1 source, Class<?> sourceClass, T2 destination) {
        if (sourceClass.getSuperclass() != null) {
            clone(source, sourceClass.getSuperclass(), destination);
        }
        Field[] fields = sourceClass.getDeclaredFields();
        String fieldName;
        for (Field field : fields) {
            fieldName = field.getName();
            if (hasField(destination, fieldName)) {
                setField(destination, fieldName, getField(source, fieldName));
            }
        }
        return destination;
    }

    @SuppressWarnings("unchecked")
    public static <T> T valueOrDefault(Object value, T defaultValue) {
        return value != null ? (T) value : defaultValue;
    }

    public static void parseAndSet(Object object, String fieldName, String rawValue) {
        Field field = ObjectUtil.findField(object, fieldName);
        if (field == null) {
            return;
        }
        Class<?> type = field.getType();
        if (type.isArray()) {
            return;
        }
        if (type == String.class) {
            ObjectUtil.setField(object, fieldName, rawValue);
        } else if (type == Boolean.class || type == boolean.class) {
            ObjectUtil.setField(object, fieldName, Boolean.valueOf(rawValue));
        } else if (type == Integer.class || type == int.class) {
            ObjectUtil.setField(object, fieldName, Integer.valueOf(rawValue, 10));
        } else if (type == Float.class || type == float.class) {
            ObjectUtil.setField(object, fieldName, Float.valueOf(rawValue));
        } else if (type == Double.class || type == double.class) {
            ObjectUtil.setField(object, fieldName, Double.valueOf(rawValue));
        } else if (type.isEnum()) {
            Method valueOf = ObjectUtil.findMethod(type, type, "valueOf", String.class);
            if (valueOf != null) {
                Object enumValue = ObjectUtil.safeInvoke(type, valueOf, rawValue);
                ObjectUtil.setField(object, fieldName, enumValue);
            }
        }
    }
}
